/*
 * QIC_02.C - Routines to communicate with a Wangtek PC36 or compatible
 * QIC-02 tape controller
 *
 * $Author:   Phlash  $
 * $Date:   31 Oct 1994 22:13:10  $
 * $Revision:   1.4  $
 */

#ifdef DEVICE_MODE
#define FP_SEG(fp)  (*((int *)&(fp)+1))
#define FP_OFF(fp)  (*((int *)&(fp)))
#define GETINTR(x)      getVector(x)
#define SETINTR(x, y)   setVector(x, y)
extern void _disable(void);
extern void _enable(void);

#else
#include <dos.h>
#define GETINTR(x)      _dos_getvect(x)
#define SETINTR(x, y)   _dos_setvect(x, y)

#endif
#include "qic_02.h"

/* Use in-line code for these operations */
#pragma intrinsic (outp, inp)

/* Debugging */
#define DS(str) if(qicDebug) _qicDebug(str)
#define DC(ch)  if(qicDebug) { \
                        screen[screenPtr+1] = 7;  \
                        screen[screenPtr] = (ch); \
                        screenPtr += 2;           \
                        if(screenPtr >= 3840)     \
                           screenPtr = 160;       \
                        screen[screenPtr+1] = 12; \
                        }

/* Delay for controller reset (in 18.2Hz ticks) */
#define RESET_DELAY     54

/* Interrupt controller constants */
#define INTR_OFFSET     8
#define INTR_EOI        0x20

/* Interrupt controller registers */
#define INTR_CONTROL    0x20
#define INTR_MASK       0x21

/* DMA controller IO port addresses */
#define DMA_ADDR_OFF    0
#define DMA_CNTR_OFF    1
#define DMA_STAT_PORT   0x08
#define DMA_MASK_PORT   0x0A
#define DMA_MODE_PORT   0x0B
#define DMA_LOAD_PORT   0x0C

/* DMA page register addresses */
#define DMA_PAGE_0      0x87
#define DMA_PAGE_1      0x83
#define DMA_PAGE_2      0x81
#define DMA_PAGE_3      0x82

/* DMA Mask port values (OR in channel) */
#define DMA_OFF         0x04
#define DMA_ON          0

/* DMA mode register values (OR in channel) - deep magik ! */
#define DMA_MEMREAD     0x48
#define DMA_MEMWRITE    0x44

/* DMA transfer block sizes */
#define DMA_BLOCKSIZE   512

/* Driver state */
#define DRV_IDLE        0

/* Other (!) */
#define CTRL_BREAK_VECTOR 0x1B
#define TICK_TIMER_VECTOR 0x1C

/*--------------------------------------------------------------------------
   Library Globals
--------------------------------------------------------------------------*/
// Debugging and version string (public access allowed)
int qicDebug = 0;
char qicVersion[] = "$Revision:   1.4  $";

// CGA screen address and current location pointers
static char far *screen = (char far *)0xB8000000L;
static int screenPtr = 160;

// Initialisation data
static int init=0;           // Driver initialised
static int port=0;           // PC-36 I/O port base
static int intr=0;           // PC-36 Interrupt
static int dma=0;            // PC-36 DMA Channel

// Old interrupt vectors
static void (interrupt far *oldCtrlBreak)();
static void (interrupt far *oldTapeIntr)();
static void (interrupt far *oldTickTimer)();

// Interrupt counters
static int keyBreak=0;
static int tapeIntr=0;
static int tickCount=0;

// State machine variables
static int drvState=0;       // Device driver state (QIC-02 command outstanding)
static int lastError=0;      // Last device error
static int portBits=0;       // PC-36 Control port value
static int tickPoll=0;       // Poll tape interrupt in tick timer (yuk!)
static int dmaGoing=0;       // Are we actually doing DMA?
static int noBlock=0;        // Do we block on DMA transfers?
static int eofFlag=0;        // Tape at EOM/EOF

// DMA Page register lookup table
static int dmaPages[4] = { DMA_PAGE_0, DMA_PAGE_1, DMA_PAGE_2, DMA_PAGE_3 };

// DMA transfer data
static DWORD dmaBuf=0;       // Current DMA transfer address (physical)
static WORD dmaLen=0;        // Current DMA transfer length (blocks)
static WORD dmaCnt=0;        // Current DMA transfer count (blocks)

// QIC-02 Drive status
static status_t status;

/*--------------------------------------------------------------------------
   Library Internal function prototypes
--------------------------------------------------------------------------*/
static int _qicCommand(BYTE command);
static int _qicGetStatus(void);
static int _qicReset(void);
static int _qicPollStatus(BYTE mask, BYTE bits);
static void _qicWait(void);
static DWORD _qicLogToPhys(void far *ptr);
static int _qicStartDMA(void);
static int _qicStopDMA(void);
static int _qicError(void);
static int ready(void);
static int exception(void);
static int direction(void);
static int dmadone(void);
static WORD dmacntr(void);
static int keybreak(void);
static void _qicDebug(char far *str);

#ifdef DEVICE_MODE
#define CTRL_TRAP _qicCtrlTrap
#define TICK_TRAP _qicTickTrap
#define TAPE_TRAP _qicTapeTrap

extern void _qicCtrlTrap(void);
extern void _qicTickTrap(void);
extern void _qicTapeTrap(void);

void _qicCtrlBreak(void);
void _qicTickTimer(void);
void _qicTapeIntr(void);
static void (interrupt far *getVector(int intr))();
static void setVector(int intr, void far *vect);
#else
#define CTRL_TRAP _qicCtrlBreak
#define TICK_TRAP _qicTickTimer
#define TAPE_TRAP _qicTapeIntr

static void interrupt far _qicCtrlBreak(void);
static void interrupt far _qicTapeIntr(void);
static void interrupt far _qicTickTimer(void);
#endif

static void wtoh(char far *output, WORD w);

/*--------------------------------------------------------------------------
   Library External function definitions
--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------
   qicInit - attempts to reset the controller at the specified IO port address
   Returns:  0 = OK
             1 = No tape found
--------------------------------------------------------------------------*/
int qicInit(int newPort, int newInt, int newDma, int blocking)
{
int maskBits;

// Are we initialised already?
   DS("Init:{");
   if(init)
   {
      DS("}:0");
      return 0;
   }

// Trap tick timer interrupt
   oldTickTimer = GETINTR(TICK_TIMER_VECTOR);
   SETINTR(TICK_TIMER_VECTOR, TICK_TRAP);

// attempt to reset the controller
   port = newPort;                            // Let's assume it's there..
   intr = newInt;                             // and set globals up.
   dma = newDma;
   
   if(_qicReset())                            // attempt a reset on device
   {
   // Reset the global data, and return ERROR
      SETINTR(TICK_TIMER_VECTOR, oldTickTimer);
      tickCount = 0;
      port = 0;
      intr = 0;
      dma  = 0;
      DS("}:1");
      return 1;
   }

// All OK, so trap tape interrupt and keyboard break
   DS("traps");
   oldTapeIntr = GETINTR(intr + INTR_OFFSET);
   SETINTR(intr + INTR_OFFSET, TAPE_TRAP);
   maskBits = 1 << intr;                      // Enable interrupt on 8259
   outp(INTR_MASK, inp(INTR_MASK) & ~maskBits);
   oldCtrlBreak = GETINTR(CTRL_BREAK_VECTOR);
   SETINTR(CTRL_BREAK_VECTOR, CTRL_TRAP);
   
// set blocking mode
   if(!blocking)
      noBlock = 1;

// set init flag & return OK
   init = TRUE;
   DS("}:0");
   return 0;
}

/*--------------------------------------------------------------------------
   qicSelect - selects tape density
   Returns:  0 = OK
             !0 = ERROR
--------------------------------------------------------------------------*/
int qicSelect(int density)
{
// Send the command
   DS("Select:{");
   if(_qicCommand(density))
   {
      _qicGetStatus();
      DS("}:-1");
      return _qicError();
   }

// Wait for completion
   _qicWait();

// Check for OK or error
   if(lastError)
   {
      DS("}:-2");
      return _qicError();
   }

   DS("}:0");
   return 0;
}

/*--------------------------------------------------------------------------
   qicEraseTape - erases tape
   Returns:  0 = OK
             !0 = ERROR
--------------------------------------------------------------------------*/
int qicEraseTape(void)
{
// Send the command
   DS("Erase:{");
   if(_qicCommand(ERASE))
   {
      _qicGetStatus();
      DS("}:-1");
      return _qicError();
   }

// Wait for completion
   _qicWait();

// Check for OK or error (ignore BOM)
   if(lastError && !status.BOM)
   {
      DS("}:-2");
      return _qicError();
   }
   DS("}:0");
   return 0;
}

/*--------------------------------------------------------------------------
   qicRewind - attempts to rewind the current tape
   Returns:  0 = OK
             !0 = ERROR
--------------------------------------------------------------------------*/
int qicRewind(void)
{
// If the tape is On-line..
   DS("Rewind:{");
   if(portBits & CONTROL_ONL)
   {
   // wait for RDY or EXP
      while(!exception() && !ready());

   // Check for error
      if(exception())
      {
         _qicGetStatus();
         DS("}:-1");
         return _qicError();
      }

   // take it off-line causing a rewind to occur..
      DS("Offline");
      drvState = REWIND;
      lastError = 0;
      eofFlag = 0;
      portBits = portBits & ~CONTROL_ONL | CONTROL_DMA;
      outp(port, portBits);
   }
   else
   {
   // otherwise, send the rewind command.
      if(_qicCommand(REWIND))
      {
         _qicGetStatus();
         DS("}:-2");
         return _qicError();
      }
   }

// Wait for completion
   _qicWait();

// Check for OK or error (ignore BOM)
   if(lastError && !status.BOM)
   {
      DS("}:-3");
      return _qicError();
   }

   DS("}:0");
   return 0;
}

/*--------------------------------------------------------------------------
   qicSkipFile - attempts to skip to the end of the current file
   Returns:  0 = OK
             !0 = ERROR
--------------------------------------------------------------------------*/
int qicSkipFile(void)
{
// Send the command
   DS("Skip:{");
   if(_qicCommand(READ_MARK))
   {
      _qicGetStatus();
      DS("}:-1");
      return _qicError();
   }

// Wait for completion
   _qicWait();

// Check FIL or EOM status
   if(!lastError || !(status.FIL | status.EOM))
   {
      DS("}:-2");
      return _qicError();
   }

// OK
   DS("}:0");
   return 0;
}

/*--------------------------------------------------------------------------
   qicReadBlocks - reads data blocks from tape (if any), into supplied buffer
   Returns:  >0 = number of blocks read
             0  = End of file (file mark encountered) OR non-blocking read
             <0 = ERROR
--------------------------------------------------------------------------*/
int qicReadBlocks(char far *buffer, int nBlocks)
{
char str[5];

// debugging
   DS("Read:{");
   wtoh(str, FP_SEG(buffer));
   DS(str);
   DS(":");
   wtoh(str, FP_OFF(buffer));
   DS(str);
   DS(":");
   wtoh(str, nBlocks);
   DS(str);

// EOF check
   if(eofFlag)
   {
      DS("}:EOF");
      return 0;
   }

// Sanity check
   if(nBlocks < 1)
   {
      DS("}:-1");
      return -1;
   }

// See if we are currently doing a read
   if(READ_DATA == drvState)
   {
   // Wait for current transfer to complete
      _qicWait();

   // Reset driver state
      drvState = READ_DATA;
   }
   else
   {
   // Send command to tape drive
      if(_qicCommand(READ_DATA))
      {
         _qicGetStatus();
         DS("}:-2");
         return _qicError();
      }
   }

// wait for drive ready or exception
   while(!exception() && !ready());
   if(exception())
   {
      _qicGetStatus();
      DS("}:-3");
      return _qicError();
   }

// Set DMA transfer data
   dmaBuf = _qicLogToPhys(buffer);
   dmaLen = nBlocks;
   dmaCnt = 0;

// start DMA transfer
   _qicStartDMA();

// if non-blocking return here
   if(noBlock)
   {
      DS("}:NB");
      return 0;
   }

// wait for DMA transfer to complete
   _qicWait();

// Check for errors
   if(lastError)
   {
      if(status.FIL || status.EOM)
         eofFlag = 1;
      else
      {
         DS("}:-4");
         return _qicError();
      }
   }

// All read OK
   DS("}:OK");
   return dmaCnt;
}

/*--------------------------------------------------------------------------
   qicWriteFileMark - writes a file mark on the tape
   Returns:  0 = OK
             !0 = ERROR
--------------------------------------------------------------------------*/
int qicWriteFileMark(void)
{
// Send the command
   DS("FileMark:{");
   if(_qicCommand(WRITE_MARK))
   {
      _qicGetStatus();
      DS("}:-1");
      return _qicError();
   }

// Wait for completion
   _qicWait();

// Check for exception
   if(lastError)
   {
      DS("}:-1");
      return _qicError();
   }

// OK
   DS("}:0");
   return 0;
}

/*--------------------------------------------------------------------------
   qicWriteBlocks - writes data blocks from from supplied buffer to tape
   Returns:  >0 = Number of blocks written
             0  = Non-blocking write
             <0 = ERROR
--------------------------------------------------------------------------*/
int qicWriteBlocks(char far *buffer, int nBlocks)
{
char str[5];

// debugging
   DS("Write:{");
   wtoh(str, FP_SEG(buffer));
   DS(str);
   DS(":");
   wtoh(str, FP_OFF(buffer));
   DS(str);
   DS(":");
   wtoh(str, nBlocks);
   DS(str);


// Sanity check
   if(nBlocks < 1)
   {
      DS("}:-1");
      return -1;
   }

// Check if we are currently doing a write
   if(WRITE_DATA == drvState)
   {
   // Wait for current transfer to complete
      _qicWait();

   // Reset driver state
      drvState = WRITE_DATA;
   }
   else
   {
   // Send command to tape drive
      if(_qicCommand(WRITE_DATA))
      {
         _qicGetStatus();
         DS("}:-2");
         return _qicError();
      }
   }

// wait for drive ready or exception
   while(!exception() && !ready());
   if(exception())
   {
      _qicGetStatus();
      DS("}:-3");
      return _qicError();
   }

// Set DMA data
   dmaBuf = _qicLogToPhys(buffer);
   dmaLen = nBlocks;
   dmaCnt = 0;

// start DMA transfer
   _qicStartDMA();

// if non-blocking return here
   if(noBlock)
   {
      DS("}:0");
      return 0;
   }

// wait for DMA to complete
   _qicWait();

// If exception then bug out
   if(lastError)
   {
      DS("}:-4");
      return _qicError();
   }

// All written OK
   DS("}:OK");
   return dmaCnt;
}

/*--------------------------------------------------------------------------
   qicComplete - waits for current DMA transfer to complete, return status
   Returns:  0 = EOF on read
             >0 = Number of blocks transferred
             <0 = ERROR
--------------------------------------------------------------------------*/
int qicComplete(void)
{
// Check EOF flag
   DS("Complete:{");
   if(eofFlag)
   {
      DS("}:EOF");
      return 0;
   }

// Wait for transfer to complete
   _qicWait();

// Check for errors
   if(lastError)
   {
      if(status.FIL || status.EOM)
         eofFlag = 1;
      else
      {
         DS("}:-1");
         return _qicError();
      }
   }
   DS("}:OK");
   return dmaCnt;
}


/*--------------------------------------------------------------------------
   Library Internal function definitions
--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------
   _qicReset - attempts to reset the controller
   Returns:  0 = OK
             1 = ERROR
--------------------------------------------------------------------------*/
int _qicReset(void)   
{
int start;

// Start timing here
   DS("Reset:{");
   start = tickCount;
   while(start == tickCount);

// Set control port RESET & ONLINE bits
   outp(port, portBits = CONTROL_RES | CONTROL_ONL);

// Wait 1 tick
   start = tickCount;
   while(start == tickCount);

// Clear RESET bit
   outp(port, portBits &= ~CONTROL_RES);

// Timed wait for EXC
   while(!exception() && (tickCount - start) < RESET_DELAY);

// If nothing happened, die
   if( ((tickCount - start) >= RESET_DELAY) || !exception() )
   {
      DS("}:1");
      return 1;
   }

// Now try to read the EXCEPTION status
   if(_qicGetStatus())
   {
      DS("}:1");
      return 1;
   }

// Finally check for POR bit in status
   if(!status.POR)
      return 1;
   DS("}:0");
   return 0;
}

/*--------------------------------------------------------------------------
   _qicWait - waits for driver to become idle (all operations completed)
--------------------------------------------------------------------------*/
void _qicWait(void)
{
int i, oldKeyBreak = keyBreak;
char *idles = "-\|/";

// Enable tick timer polling
//   tickPoll = 1;

// Do something useless while waiting for DRV_IDLE or Keybreak
   DS("Wait:{");
   while(drvState != DRV_IDLE && oldKeyBreak == keyBreak)
      if(qicDebug) screen[screenPtr] = idles[i = (i+1)%4];

// Disable tick timer polling
//   tickPoll = 0;

// Did somebody call?
   if(oldKeyBreak != keyBreak)
      lastError = 1;
   DS("}");
}

/*--------------------------------------------------------------------------
   _qicCommand - sends the specified command to the controller. The calling
                 function must call _qicWait() if it needs to wait for the
                 completion of the command.
   Returns:  0 = OK
             1 = ERROR
--------------------------------------------------------------------------*/
int _qicCommand(BYTE command)
{
// Wait for controller to be ready
   DS("Command:{");
   _qicWait();
   if(!exception() && !ready())
   {
      DS("}:1");
      return 1;
   }
   if(exception())
   {
      DS("}:2");
      return 2;
   }

// Send the command
   outp(port+1, command);
   outp(port, portBits |= CONTROL_REQ);

// Wait for ready (command accepted)
   while(!ready());

// set driver state for interrupt handler
   drvState = command;

// clear last error & EOF flag
   lastError = 0;
   eofFlag = 0;

// Clear REQUEST bit, enable interrupts & DMA transfers
   portBits = portBits & ~CONTROL_REQ | CONTROL_DMA;
   outp(port, portBits);

// wait for !RDY
   while(ready());

// Done
   DS("}:0");
   return 0;
}

/*--------------------------------------------------------------------------
   _qicGetStatus - reads controller status
   Returns:  0 = OK
             1 = ERROR
--------------------------------------------------------------------------*/
int _qicGetStatus(void)
{
int i;
BYTE byte, far *statBytes;

// Check the controller is ready or an exception is pending
   DS("GetStatus:{");
   if(!ready() && !exception())
   {
      DS("}:1");
      return 1;
   }

// Send the command
   outp(port+1, READ_STATUS);

// Assert REQ and ONL, clear interrupt enable
   outp(port, portBits = CONTROL_REQ | CONTROL_ONL);

// Wait for ready (command accepted)
   while(!ready());

// Clear REQUEST bit
   outp(port, portBits &= ~CONTROL_REQ);

// Wait for DIRC
   while(!direction());

// Now handshake using ready/request lines to get six bytes
   DS("xfer");
   statBytes = (BYTE far *)&status;
   for(i=0; i<6; i++)
   {
   // Wait for RDY | EXC
      while(!exception() && !ready());
      
   // Check for EXC
      if(exception())
      {
         DS("}:2");
         return 2;
      }

   // Read byte
      byte = inp(port+1);

   // Assert REQ
      outp(port, portBits |= CONTROL_REQ);

   // Wait !ready
      while(ready());

   // Deassert REQ
      outp(port, portBits &= ~CONTROL_REQ);

   // Process byte
      switch(i)
      {
      case 0:                      // Assign status bit fields
      case 1:
         statBytes[i] = byte;
         break;
      case 2:                      // Byte swap WORD values for Intel CPU
         statBytes[3] = byte;
         break;
      case 3:
         statBytes[2] = byte;
         break;
      case 4:
         statBytes[5] = byte;
         break;
      case 5:
         statBytes[4] = byte;
         break;
      }
   }
   DS("}:0");
   return 0;
}

/*--------------------------------------------------------------------------
   _qicError - returns error code based on current status value
--------------------------------------------------------------------------*/
int _qicError(void)
{
WORD *wp = (WORD *)&status;
char str[5];

// Print error bits
   wtoh(str, *wp);
   DS("Error:{");
   DS(str);
   DS("}");
   return -1;
}

/*--------------------------------------------------------------------------
   _qicLogToPhys - converts a far pointer to a physical address
--------------------------------------------------------------------------*/
DWORD _qicLogToPhys(void far *ptr)
{
WORD seg, off;

// Calculate physical address from SEG:OFF address
   seg = FP_SEG(ptr);
   off = FP_OFF(ptr);
   return ((DWORD)seg << 4) + (DWORD)off;
}

/*--------------------------------------------------------------------------
   _qicStartDMA - initiates a DMA transfer
   Returns: 0 = OK
            !0 = ERROR
--------------------------------------------------------------------------*/
int _qicStartDMA(void)
{
WORD byteCnt;

// Set DMA controller mode (on selected channel)
   DC('[');
   switch(drvState)
   {
   case READ_DATA:
      outp(DMA_MODE_PORT, DMA_MEMWRITE | dma);
      break;
   case WRITE_DATA:
      outp(DMA_MODE_PORT, DMA_MEMREAD | dma);
      break;
   default:
      DS("1]");
      return 1;
   }

// Clear address load flip-flop
   outp(DMA_LOAD_PORT, 0);

// Set starting address (lower 16-bits) in DMA controller
   outp((2 * dma) + DMA_ADDR_OFF, dmaBuf & 0xFF);
   outp((2 * dma) + DMA_ADDR_OFF, (dmaBuf >> 8) & 0xFF);

// Set starting address (top 8-bits) in page register
   outp(dmaPages[dma], (dmaBuf >> 16) & 0xFF);

// Set block size to transfer (minus 1 for some reason?)
   byteCnt = DMA_BLOCKSIZE - 1;
   outp((2 * dma) + DMA_CNTR_OFF, byteCnt & 0xFF);
   outp((2 * dma) + DMA_CNTR_OFF, (byteCnt >> 8) & 0xFF);

// Enable DMA on selected channel of controller
   dmaGoing = 1;
   outp(DMA_MASK_PORT, DMA_ON | dma);

// Done!
   DS("0]");
   return 0;
}

/*--------------------------------------------------------------------------
   _qicStopDMA - terminates DMA transfer (if any)
   Returns:  0 - OK
             !0 - ERROR
--------------------------------------------------------------------------*/
int _qicStopDMA(void)
{
int rv = !dmadone();

   DC('(');
   DC( (rv) ? '1' : '0' );
   outp(DMA_MASK_PORT, DMA_OFF | dma);
   dmaGoing = 0;
   DC(')');
   return rv;
}

/*--------------------------------------------------------------------------
   ready - returns current state of READY bit in status register
   Returns:  0 = Not ready
             !0 = Ready
--------------------------------------------------------------------------*/
int ready(void)
{
// return ready status of controller
   return !(inp(port) & STATUS_RDY);
}

/*--------------------------------------------------------------------------
   exception - returns current state of EXCEPTION bit in status register
   Returns:  0 = No exception
             !0 = Exception
--------------------------------------------------------------------------*/
int exception(void)
{
// return exception status of controller
   return !(inp(port) & STATUS_EXC);
}

/*--------------------------------------------------------------------------
   direction - returns current state of DIRECTION bit in status register
   Returns:  0 = Host->Controller
             !0 = Controller->Host
--------------------------------------------------------------------------*/
int direction(void)
{
// return direction status of controller
   return !(inp(port) & STATUS_DIR);
}

/*--------------------------------------------------------------------------
   dmadone - returns terminal count flag of DMA channel (not done / done)
   Returns:  0 =  Terminal Count not reached
             !0 = Terminal Count reached
--------------------------------------------------------------------------*/
int dmadone(void)
{
// return TC bit for selected DMA channel
   return (inp(DMA_STAT_PORT) & (1 << dma));
}

/*--------------------------------------------------------------------------
   dmacntr - returns current count value for DMA transfer
   Returns:  current counter value from DMA controller
--------------------------------------------------------------------------*/
WORD dmacntr(void)
{
WORD count;

// Read counter value from DMA controller in two bytes (LSB first)
   outp(DMA_LOAD_PORT, 0);
   count = inp((2 * dma) + DMA_CNTR_OFF);
   count += (inp((2 * dma) + DMA_CNTR_OFF) << 8);
   return count;
}

/*--------------------------------------------------------------------------
   keybreak - returns status of keyboard break flag
   Returns:  0 = No keyboard break
             >0 = number of keyboard breaks detected
--------------------------------------------------------------------------*/
int keybreak(void)
{
int oldBreak = keyBreak;

// Reset break count
   keyBreak = 0;

// return count of number of breaks
   return oldBreak;
}

/*--------------------------------------------------------------------------
   _qicDebug - writes string to debug screen
   Returns:  none.
--------------------------------------------------------------------------*/
void _qicDebug(char far *str)
{
int i;

   screen[screenPtr+1] = 7;
   for(i=0; str[i]; i++)
   {
      screen[screenPtr] = str[i];
      screenPtr += 2;
      if(screenPtr >= 3840)
         screenPtr = 160;
   }
   screen[screenPtr] = '*';
   screen[screenPtr+1] = 12;
}

/*--------------------------------------------------------------------------
   _qicCtrlBreak - increments flag to indicate CTRL-Break has been pressed
   Returns:  none.
--------------------------------------------------------------------------*/
#ifdef DEVICE_MODE
void _qicCtrlBreak(void)
#else
void interrupt far _qicCtrlBreak(void)
#endif
{
// Increment break counter
   DC('k');
   keyBreak++;
}

/*--------------------------------------------------------------------------
   _qicTapeIntr - handles interrupts from the tape drive (RDY | EXC)
   Returns:  none.
--------------------------------------------------------------------------*/
#ifdef DEVICE_MODE
void _qicTapeIntr(void)
#else
void interrupt far _qicTapeIntr(void)
#endif
{
int i, oldDrvState;
BYTE stat;
char hex[5];

#ifdef DEVICE_MODE
   _enable();
#endif

// Send End Of Interrupt (EOI) to interrupt controller
   DC('<');
   outp(INTR_CONTROL, INTR_EOI);

// Read device status port
   stat = inp(port);

// See if this is for me
   if( (stat & (STATUS_RDY | STATUS_EXC)) == (STATUS_RDY | STATUS_EXC) )
   {
      DC('*');
      DC('>');
      return;
   }

// Increment interrupt counter
   tapeIntr++;

// Dump debugging data
   if(qicDebug)
   {
      wtoh(hex, tapeIntr);
      for(i=0; i<4; i++)
         screen[40+i*2] = hex[i];

      DC('(');
      if(!(stat & STATUS_RDY))
      {
         DC('R');
      }
      else
      {
         DC(' ');
      }
      if(!(stat & STATUS_EXC))
      {
         DC('E');
      }
      else
      {
         DC(' ');
      }
      if(!(stat & STATUS_DIR))
      {
         DC('D');
      }
      else
      {
         DC(' ');
      }
      DC(')');
   }

// What were we doing then?
   switch(drvState)
   {
   case 0:
      DC('I');
      break;

   case QIC11_FORMAT:
   case QIC24_FORMAT:
   case QIC120_FORMAT:
   case QIC150_FORMAT:
      DC('D');
      drvState = DRV_IDLE;
      if(!(stat & STATUS_EXC))
      {
         _qicGetStatus();
         lastError = 1;
      }
      break;

   case REWIND:
      DC('R');
      drvState = DRV_IDLE;
      _qicGetStatus();
      if(!status.BOM)
         lastError = 1;
      break;

   case READ_MARK:
      DC('S');
      drvState = DRV_IDLE;
      if(!(stat & STATUS_EXC))
      {
         _qicGetStatus();
         lastError = 1;
      }
      break;

   case ERASE:
      DC('E');
      drvState = DRV_IDLE;
      _qicGetStatus();
      if(!status.BOM)
         lastError = 1;
      break;

   case WRITE_MARK:
      DC('M');
      drvState = DRV_IDLE;
      if(!(stat & STATUS_EXC))
      {
         _qicGetStatus();
         lastError = 1;
      }
      break;

   case READ_DATA:
      DC('R');
      break;

   case WRITE_DATA:
      DC('W');
      break;

   default:
      DC('!');
      break;
   }

// Handle DMA stuff
   if((drvState == READ_DATA || drvState == WRITE_DATA) && dmaGoing)
   {
      oldDrvState = drvState;
      drvState = DRV_IDLE;
      _qicStopDMA();
      dmaCnt++;

      if(!(stat & STATUS_EXC))
      {
         _qicGetStatus();
         lastError = 1;
      }
      else
      {
         if(dmaCnt < dmaLen)
         {
            dmaBuf += DMA_BLOCKSIZE;
            drvState = oldDrvState;
            _qicStartDMA();
         }
      }
   }
   DC('>');
}

/*--------------------------------------------------------------------------
   _qicTickTimer - increments flag to indicate PC timer interrupt has occured
   Returns:  none.
--------------------------------------------------------------------------*/
#ifdef DEVICE_MODE
void _qicTickTimer(void)
#else
void interrupt far _qicTickTimer(void)
#endif
{
int i;
char hex[5];

// Increment tick counter
   tickCount++;

// Display on screen
   if(qicDebug)
   {
      wtoh(hex, tickCount);
      for(i=0; i<4; i++)
         screen[i*2] = hex[i];
   }

// See if we need to poll the tape interrupt
   if(tickPoll)
      _qicTapeIntr();

// Call previous interrupt owner
   (*oldTickTimer)();
}

/*--------------------------------------------------------------------------
   qicDeInit - resets the execution environment on program exit
   Returns:  none.
--------------------------------------------------------------------------*/
void qicDeInit(void)
{
int maskBits, i;

// Stop all DMA activity
   _qicStopDMA();

// Disable DMA requests from tape controller
   portBits = CONTROL_CLR;
   outp(port, portBits);

// Reset keyboard break handler
   SETINTR(CTRL_BREAK_VECTOR, oldCtrlBreak);

// Reset tape interrupt
   maskBits = 1 << intr;
   outp(INTR_MASK, inp(INTR_MASK) | maskBits);
   SETINTR(intr + INTR_OFFSET, oldTapeIntr);

// Reset tick timer interrupt
   SETINTR(TICK_TIMER_VECTOR, oldTickTimer);
}

/*--------------------------------------------------------------------------
   wtoh - convert a WORD to a hex string
   Returns:  none.
--------------------------------------------------------------------------*/
void wtoh(char far *output, unsigned short w)
{
unsigned short nibble;

/* convert WORD to Hexadecimal format string */
   nibble = (w & 0xF000) >> 12;
   if(nibble > 9)
      output[0] = (char)(nibble - 10 + 'A');
   else
      output[0] = (char)(nibble + '0');

   nibble = (w & 0x0F00) >> 8;
   if(nibble > 9)
      output[1] = (char)(nibble - 10 + 'A');
   else
      output[1] = (char)(nibble + '0');

   nibble = (w & 0x00F0) >> 4;
   if(nibble > 9)
      output[2] = (char)(nibble - 10 + 'A');
   else
      output[2] = (char)(nibble + '0');

   nibble = (w & 0x000F);
   if(nibble > 9)
      output[3] = (char)(nibble - 10 + 'A');
   else
      output[3] = (char)(nibble + '0');

   output[4] = 0;
}

#ifdef DEVICE_MODE
/*--------------------------------------------------------------------------
   getVector - return the current value of an interrupt vector
   Returns:  void (interupt far *)
--------------------------------------------------------------------------*/
void (interrupt far *getVector(int intr))()
{
void (interrupt far * far *vp)();

   FP_SEG(vp) = 0;
   FP_OFF(vp) = intr * 4;
   return *vp;
}

/*--------------------------------------------------------------------------
   setVector - set the value of an interrupt vector
   Returns:  void
--------------------------------------------------------------------------*/
void setVector(int intr, void far *vect)
{
void far * far *vp;

   FP_SEG(vp) = 0;
   FP_OFF(vp) = intr * 4;
   _disable();
   *vp = vect;
   _enable();
}
#endif

/* End of QIC_02.C */
