/**<feal.c>**************************************************
*                       V O U C H                           *
*   VOUCH 1.1  Copyright (c) 1993, 1994  Awais M Hussain    *
*                   All rights reserved                     *
*************************************************************/
/* Consult <feal.man> for documentation. */

/* uses FEAL-N/X (Fast Encipherment Algorithm)             */
/* for little endian machines;      */
/* see f_K(), f_ow(), keySchedule(), feal_e() */
/* unsigned is 16 bits long */

#include <stdio.h>
#include <string.h>
#include <mem.h>
#include <stdlib.h>
#include <fcntl.h>
#include <io.h>
#include "vch.h"


#define ROL2(x) ( ((x) << 2) | ((x) >> 6) )
#define NN 32 					/* FEAL-32/X */
#define NBUFF 4096

long unsigned  key[4] = {0,0,0,0};
static unsigned  Kw[NN + 7 + 1];
static long unsigned K1, K2, K3, K4 ;

/********************             ********************/

/*1
encrypt/decrypt a LongNumber (Lvv, vv) using 8-32-char password */
/* (enc = true) => encrypt */
/* encryted vector length is incremented by 1, if the input-length is odd */

void  cryptLN( int  enc, char  *pswd, lenLN  *Lvv, ptrLN  vv ) ;

/* 2
encrypt <infile> */
/* caller must fill key[] with the hex password  */

int  fecrypt( char *infile, char *outfile ) ;


/* 3
decrypt <infile> */
/* caller must fill key[] with the hex password  */

int  fdcrypt( char *infile, char *outfile ) ;


/*4
fill hex key[] from pswd string */

void  str2key( char *pswd ) ;

/*******************                  ******************/
static long unsigned  f_K( long unsigned awp, long unsigned bwp) ;
long unsigned reverse( long unsigned xx )  ;
static void keySchedule( void ) ;
void feal_e( long unsigned *Lf, long unsigned *Rf )  ;
/********************************************************/

/* function called by keySchedule() */
/************************************/
static long unsigned  f_K( long unsigned awp, long unsigned bwp)
{
	static long unsigned aw, bw;
	static long unsigned fw;
	static unsigned char *pa3 = (unsigned char*) &aw,
		   	     *pa2 = (unsigned char*) &aw + 1,
			     *pa1 = (unsigned char*) &aw + 2,
			     *pa0 = (unsigned char*) &aw + 3;
	static unsigned char *pb3 = (unsigned char*) &bw,
			     *pb2 = (unsigned char*) &bw + 1,
			     *pb1 = (unsigned char*) &bw + 2,
			     *pb0 = (unsigned char*) &bw + 3;
	static unsigned char *pf3 = (unsigned char*) &fw,
 			     *pf2 = (unsigned char*) &fw + 1,
			     *pf1 = (unsigned char*) &fw + 2,
			     *pf0 = (unsigned char*) &fw + 3;

	aw = awp;
	bw = bwp;
	*pf1 = *pa1 ^ *pa0;
	*pf2 = *pa2 ^ *pa3;
 	*pf1 = ROL2( (unsigned char) (*pf1 + (*pf2 ^ *pb0) + 1) );
	*pf2 = ROL2( (unsigned char) (*pf2 + (*pf1 ^ *pb1)) );
	*pf0 = ROL2( (unsigned char) (*pa0 + (*pf1 ^ *pb2)) );
	*pf3 = ROL2( (unsigned char) (*pa3 + (*pf2 ^ *pb3) + 1) );
	return fw ;
}/* f_K */

/* reverse the order of bytes */
long unsigned reverse( long unsigned xx ) 
{
  static long unsigned yy ;
  static char *y1 = (char*) &yy,
              *y2 = (char*) &yy+1,
              *y3 = (char*) &yy+2,	 
              *y4 = (char*) &yy+3 ;
  char tt ;

         yy = xx ;
	 tt = *y4 ; *y4 = *y1 ; *y1 = tt ;
	 tt = *y3 ; *y3 = *y2 ; *y2 = tt ;
         return(yy) ;
}
/*reverse*/


/* expands key[] into Kw[] */
/*******************************/
static void keySchedule( void )
{
	long unsigned        B1, B0, D1, D0=0, A1, A0;
	unsigned  *pB10, *pB11;
	long unsigned        QQ[3];
	int             r, rm;

	QQ[0] = key[2] ^ key[3];
	QQ[1] = key[2];
	QQ[2] = key[3];
	pB11 = (unsigned*) &B1;
	pB10 = pB11 + 1;
	A0 = key[0];
	B0 = key[1];
	rm = NN / 2 + 4;
	for (r = 0; r < rm; ++r) {
		D1 = A0;
		A1 = B0;
		B1 = f_K(A0, B0 ^ D0 ^ QQ[r % 3]);
		Kw[2 * r] = *pB10;
		Kw[2 * r + 1] = *pB11;
		D0 = D1;
		B0 = B1;
		A0 = A1;
		switch (2*r) {
			case NN: K1 = reverse(B1) ; break ;
			case NN+2: K2 = reverse(B1) ; break ;
			case NN+4: K3 = reverse(B1) ; break ;
			case NN+6: K4 = reverse(B1) ; break ;
		}
	}
}/* keySchedule */


/* one-way function called by feal_e() */
/***************************************/
static long unsigned f_ow( long unsigned awp, unsigned bxp)
{
	static long unsigned  aw;
	static unsigned  bx;
	static long unsigned fw;
	static unsigned char *pa0 = (unsigned char*) &aw,
			     *pa1 = (unsigned char*) &aw + 1,
			     *pa2 = (unsigned char*) &aw + 2,
	                     *pa3 = (unsigned char*) &aw + 3;
	static unsigned char *pb1 = (unsigned char*) &bx,
			     *pb0 = (unsigned char*) &bx + 1;
	static unsigned char *pf0 = (unsigned char*) &fw,
			     *pf1 = (unsigned char*) &fw + 1,
			     *pf2 = (unsigned char*) &fw + 2,
			     *pf3 = (unsigned char*) &fw + 3;

	aw = awp;
	bx = bxp;
 	*pf1 = *pa1 ^ *pb0;
	*pf2 = *pa2 ^ *pb1;
	*pf1 ^= *pa0;
	*pf2 ^= *pa3;
	*pf1 = ROL2( (unsigned char) ( *pf1 + *pf2 + 1) ) ;
	*pf2 = ROL2( (unsigned char) (*pf2 + *pf1) ) ;
	*pf0 = ROL2( (unsigned char) (*pa0 + *pf1) );
	*pf3 = ROL2( (unsigned char) (*pa3 + *pf2 + 1) );
	return fw ;
}/* f_ow */


/* basic FEAL-N/X alogorithm for encrypting  (Lw,Rw)  */
/******************************************************/
static void  feal_e( long unsigned *Lw, long unsigned *Rw )
{
	long unsigned Tw;
	int             r;

	*Lw ^= K1 ;
	*Rw ^= K2 ;
	*Rw ^= *Lw;
	for (r = 1; r <= NN; r++) {
		Tw = *Rw;
		*Rw = *Lw ^ f_ow(*Rw, Kw[r - 1]);
		*Lw = Tw;
	}
	*Lw ^= *Rw;
	*Rw ^= K3;
	*Lw ^= K4 ;
	Tw = *Lw;
	*Lw = *Rw;
	*Rw = Tw;
}
/* feal_e */


/* fill hex-key[] from pswd string */
/***********************************/
void 	str2key( char *pswd )
{
	int             r, k, len, d8, m8;
	unsigned char   ips;

	setmem( key, 16, 0 ) ;
	len = strlen(pswd) - 1;
	if ( len>31) len = 31 ;
	d8 = len / 8;
	m8 = len % 8;
	for (r = 0; r < d8; r++) {
		for (k = 0; k < 8; k++) {
			ips = pswd[r * 8 + k];
			key[r] |= (ips & 0x0FUL) << 4 * k ;
		}
	}
	for (k = 0; k < m8; k++) {
		ips = pswd[8 * d8 + k];
		key[d8] |= (ips & 0x0FUL) << 4 * k ;
	}
}/* str2key */


/* encrypt/decrypt a LongNumber (Lvv, vv) using 8-32-char password */
/* (enc = true) => encrypt */
/* encryted vector length is incremented by 1, if the input length is odd */
/**************************************************************************/
void 	cryptLN( int  enc, char  *pswd, lenLN  *Lvv, ptrLN  vv )
{
	int i ;
	long unsigned L1, L0, R1, R0 ;

	if (strlen(pswd)==0) return ;
	str2key( pswd ) ;
	keySchedule() ;
	if (*Lvv & 01) {                   /* Lvv odd */
	      *(vv+*Lvv) = 0 ;  (*Lvv)++ ;
	}
	L1 = key[0] ; R1 = key[1] ;
	feal_e( &L1, &R1 ) ;
	for (i=0; i< *Lvv/2; ++i) {
		      L0 = *(vv+2*i) ; R0 = *(vv+2*i+1) ;
		      L1 ^= L0 ; R1 ^= R0 ;
		      *(vv+2*i) = L1 ; *(vv+2*i+1) = R1 ;
		      if (!enc)  {
			      L1 = L0 ; R1 = R0 ;
		      }
		      feal_e( &L1, &R1 ) ;
	}
	if (!enc)
	      while ( (*(vv+*Lvv-1)==0) && (*Lvv>1) )  (*Lvv)-- ;
	setmem( key, 16, 0 ) ;
}/* cryptLN */


/* encrypt <infile> */
/* caller must fill key[] with the hex password  */
/*************************************************/
int 	fecrypt( char *infile, char *outfile )
{
	FILE           *fin, *fout;
	unsigned char  *buff;
	int             r, nr, onr, nw, napp, nbp;
	long unsigned   Lw1, Rw1, Lw2, Rw2 ;
	long unsigned   *pLw, *pRw;

	if ( (fin = fopen(infile, "rb")) == (FILE*) NULL) return(errno) ;
	if ( (fout = fopen(outfile, "ab")) == (FILE*) NULL) return(errno) ;
	buff = (unsigned char*) malloc(NBUFF) ;
	keySchedule();
	Lw1 = key[0] ; Rw1 = key[1] ;
	feal_e( &Lw1, &Rw1 ) ;
	Lw2 = Rw1 ; Rw2 = Lw1 ;
	onr = nr = NBUFF;
	while (onr == NBUFF ) {
		onr = nr = fread((unsigned char*)buff, 1, NBUFF, fin) ;
		if (nr<NBUFF) {
			napp = 8 - (nr % 8) ;
			nbp = nr + napp ;
			for (r = nr; r < nbp; r++) {
				buff[r] = random(0x100);
				nr++;
			}
			buff[nr-1] = napp;
		}
		nbp = nr / 8;
		pLw = (long unsigned *) &buff[0];
		pRw = (long unsigned *) &buff[4];
		for (r = 0; r < nbp; ++r) {
			*pLw ^= Lw1 ; *pRw ^= Rw1 ;
			Lw1 = *pLw ; Rw1 = *pRw ;
			feal_e(&Lw1, &Rw1);
			pLw += 2;
			pRw += 2;
		}
		pLw = (long unsigned *) &buff[nr-8];
		pRw = (long unsigned *) &buff[nr-4];
		for (r = 0; r < nbp; ++r) {
			*pLw ^= Lw2 ; *pRw ^= Rw2 ;
			Lw2 = *pLw ; Rw2 = *pRw ;
			feal_e(&Lw2, &Rw2);
			pLw -= 2;
			pRw -= 2;
		}
		nw = fwrite((unsigned char*)buff, 1, nr, fout);
		if (nw != nr) {
                        fprintf(stderr, "fcrypt: error writing.\n") ;
			free(buff) ; fclose(fin) ; fclose(fout) ;
			return(31) ;
		}	
	}
	free(buff) ;
	fclose(fin);
	fclose(fout);
	return(0) ;
}/* fecrypt */


/* decrypt <infile> */
/* caller must fill key[] with the hex password  */
/*************************************************/
int 	fdcrypt( char *infile, char *outfile )
{
	FILE           *fin, *fout;
	unsigned char  *buff;
	int             r, nr, onr, nw, napp, nbp, fo;
	long unsigned   Lw1, Rw1, Lw2, Rw2, Lw0, Rw0, fsize ;
	long unsigned   *pLw, *pRw;


	if ( (fin = fopen(infile, "rb")) == (FILE*) NULL) return(errno) ;
	if ( (fout = fopen(outfile, "wb")) == (FILE*) NULL) return(errno) ;
	buff = (unsigned char*) malloc(NBUFF) ;
	fgets( buff, 2048, fin ) ;
	fgets( buff, 2048, fin ) ;
	keySchedule();
	Lw1 = key[0] ; Rw1 = key[1] ;
	feal_e( &Lw1, &Rw1 ) ;
	Lw2 = Rw1 ; Rw2 = Lw1 ;
	onr = nr = NBUFF;
	while (onr == NBUFF ) {
		onr = nr = fread((unsigned char*)buff, 1, NBUFF, fin);
		if (nr>0) {
		  nbp = nr / 8;
		  pLw = (long unsigned *) &buff[nr-8];
		  pRw = (long unsigned *) &buff[nr-4];
		  for (r = 0; r < nbp; ++r) {
			  Lw0 = *pLw ; Rw0 = *pRw ;
			  *pLw ^= Lw2 ; *pRw ^= Rw2 ;
			  Lw2 = Lw0 ; Rw2 = Rw0 ;
			  feal_e(&Lw2, &Rw2);
			  pLw -= 2;
			  pRw -= 2;
		  }
		  pLw = (long unsigned *) &buff[0];
		  pRw = (long unsigned *) &buff[4];
		  for (r = 0; r < nbp; ++r) {
			  Lw0 = *pLw ; Rw0 = *pRw ;
			  *pLw ^= Lw1 ; *pRw ^= Rw1 ;
			  Lw1 = Lw0 ; Rw1 = Rw0 ;
			  feal_e(&Lw1, &Rw1);
			  pLw += 2;
			  pRw += 2;
		  }
		  napp = buff[nr-1];
		  nw = fwrite((unsigned char*)buff, 1, nr, fout);
		  if (nw != nr) {
                          fprintf(stderr, "fcrypt: error writing.\n") ;
 			  free(buff) ; fclose(fin) ; fclose(fout) ;
                          return(31) ;
		  }
	        }
	}
	free(buff) ;
	fclose(fin);
	fclose(fout);
	if (napp>0) {
		fo = open( outfile, O_RDWR) ;
		fsize = filelength(fo) - napp ;
		chsize( fo, fsize ) ;
		close(fo) ;
	}
        return(0) ;
}/* fdcrypt */

