/*
   wren.c - Wildcard REName.

   Jason Hood, 8 November, 2002.

   Use a simple pattern substitution mechanism to rename files.

   v1.01 - 6 April, 2003:
     Win32 port;
     added "=" as substitution for a group of literals;
     fixed bugs using "$" literal substitution.

   v1.02 - 6 September & 9 October, 2003
     -F option to force rename (delete existing files);
     made option letters case sensitive.
*/

#define PVERS "1.02"
#define PDATE "9 October, 2003"


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include "lfn.h"
#include "pattern.h"
#include "letters.h"

#ifdef _WIN32
# include <direct.h>
# define getdisk _getdrive
#else // DOS
# define _cputs   cputs
# define _cprintf cprintf
#endif


enum { E_OK,		// 0 - no problems
       E_OPT,		// 1 - invalid option, option without pattern
       E_PAT,		// 2 - unable to compile pattern
       E_NOFILES,	// 3 - no files matched the pattern
       E_NOMEM, 	// 4 - not enough memory to store the filenames
       E_EXISTS,	// 5 - a renamed file already exists
       E_CONFLICT,	// 6 - a renamed file occurs twice
       E_ERROR		// 7 - unable to rename, no undo file
     };


typedef struct
{
  int all;		// -a include hidden and system files
//int match;		// -c match case (uses caseless defined in pattern.c)
  int dir;		// -d match directories
  int force;		// -F force rename of existing files
  int test;		// -t test (display without rename)
//int undo;		// -u undo (processed directly in parse_opt())
  int list;		//    no substitution, list matching files only
} s_option;

typedef struct
{
  char first;		// The first subpattern number
  char last;		// The last subpattern number
  char flag;		// Case conversion flag
} s_sub;

typedef struct
{
  char* old;
  char* new;
} s_name;


s_option option;

s_sub sub[MAX_SUBPATS];
int   subs;
int   smetas[5];

s_name* name;
int	names;

char old[MAX_LEN];
char new[MAX_LEN];

FILE* undo_fp;
char  undo_name[MAX_LEN];

char* locate_pattern( const char*, int* );
void  compile_subst( const char*, char* );
int   subnum( const char** );
void  open_undo_file( const char* );
void  undo_rename( void );
int   parse_opt( int, char** );
int   cmp_name( const void*, const void* );


#ifdef _WIN32
/*
   Copy a string, returning a pointer to the destination NUL.
*/
char* stpcpy( char* dest, const char* src )
{
  while ((*dest = *src++))
    ++dest;

  return dest;
}
#endif


void help( int brief )
{
  puts( "\
WRen by Jason Hood <jadoxa@yahoo.com.au>.\n\
Version "PVERS" ("PDATE"). Freeware.\n\
http://wren.adoxa.cjb.net/\n\
\n\
WREN [-acdFt] [old_path]old_pattern [[new_path]new_pattern]\n\
\n\
	-a	all files (include hidden and system files)\n\
	-c	match case\n\
	-d	include directories\n\
	-F	force rename (delete existing files)\n\
	-t	test (display instead of rename)\n\
	-u	undo the previous rename\
" );
  if (brief)
    puts( "\
	-h	help" );

  puts( "\n\
Notes:	old_path without new_path will rename within old_path;\n\
	old_path with new_path will rename and move to the new directory;\n\
	-t without new_pattern will display matching files\n\
	   with a colon separating each subpattern;\n\
	-tF will display all problem names (instead of the old & new names);\n\
	-u should only be used immediately after a rename.\
" );

  if (!brief)
    puts( "\
\n\
old_pattern can include:\n\
\n\
	*	match zero or more characters (maximise)\n\
	**	match zero or more characters (minimise)\n\
	?	match exactly one character\n\
	#	match a digit (0-9)\n\
	[set]	match one character in the set of characters\n\
	[!set]	match one character NOT in the set (can also use \"^\")\n\
	:	force the start of a new subpattern (within literals only)\n\
	$#	a literal \"#\"\n\
	$[	a literal \"[\"\n\
	$$	a literal \"$\"\n\
\n\
Notes:	\"*\"	will match all files\n\
	\"*.\"	will match files without an extension\n\
	\"*.*\"	will match files with an extension (NOT all files)\n\
\n\
	set	is a group (\"aeiou\") or range (\"a-z\") of characters\n\
		to include \"]\" place it first\n\
		to include \"-\" place it first or last\n\
		the above characters can also be escaped by \"$\"\n\
		if set contains \"?\" the match is optional\n\
\n\
new_pattern can include:\n\
\n\
	*	match equivalent \"*\" or \"**\" in old_pattern\n\
	?	match equivalent \"?\" in old_pattern\n\
	#	match equivalent \"#\" in old_pattern\n\
	[	match equivalent \"[set]\" or \"[!set]\" in old_pattern\n\
	=	match equivalent literal group in old_pattern\n\
	$0	the original filename\n\
	$n	the nth subpattern of old_pattern\n\
	$n+	all subpatterns from the nth\n\
	$n-m	subpatterns n to m\n\
	$-n	lowercase subpattern n\n\
	$+n	uppercase subpattern n\n\
	$^n	capitalise subpattern n (upper first, lower rest)\n\
	:	ignored (useful to distinguish \"$1\" and \"0\" from \"$10\")\n\
	$#	a literal \"#\"\n\
	$[	a literal \"[\"\n\
	$=	a literal \"=\"\n\
	$$	a literal \"$\"\n\
\n\
Subpatterns are formed from each wildcard and group of literals.\n\
Example: \"[aeiou]*.z?#\" contains five subpatterns:\n\
\n\
	$1	the character matching \"[aeiou]\"\n\
	$2	the characters matching \"*\" (possibly none)\n\
	$3	the \".z\"\n\
	$4	the character matching \"?\"\n\
	$5	the character matching \"#\"\n\
\n\
The original filename could thus be recreated by:\n\
\n\
	[*.z?#\n\
	[*=?#\n\
	$0\n\
	$1+\n\
	$1-5\n\
	$1$2$3$4$5\n\
\n\
Note that the first method will always use a lowercase \"z\",\n\
whilst the second will preserve its case.\
" );

  exit( E_OK );
}


int main( int argc, char* argv[] )
{
  static char mask[MAX_LEN];
  static char fn[MAX_LEN];
  s_name path;
  int	old_drv, new_drv;
  int	new_path = 0;
  int	total, pc = 0;
  char* fname;
  char* n;
  char* begin;
  char* p;
  struct my_ffblk ff;
  int	attrib;
  int	done;
  int	len, max;
  int	s, j;
  int	rc;

  init_lfn();

  s = parse_opt( argc, argv );
  begin = locate_pattern( argv[s], &old_drv );
  subpats = compile_pattern( begin, old );
  if (subpats == 0)
  {
    puts( "Unable to compile the pattern." );
    return E_PAT;
  }
  if (begin == argv[s])
  {
    path.old = "";
  }
  else
  {
    path.old = argv[s];
    *begin = '\0';
  }

  if (!option.list)
  {
    begin = locate_pattern( argv[s+1], &new_drv );
    compile_subst( begin, new );
    if (begin == argv[s+1])
    {
      path.new = path.old;
    }
    else if (old_drv != new_drv)
    {
      puts( "Cannot rename across drives." );
      return E_ERROR;
    }
    else
    {
      path.new = argv[s+1];
      *begin = '\0';
      // Assume if new_path is given, it will not obviously be the same as
      // old_path (ie: "wren FILE ./file" will fail, since it thinks it's
      // moving to a new path).
      if (strcomp( path.old, path.new ) != 0)
	new_path = 1;
    }
  }

  make_appr_mask( old, stpcpy( mask, path.old ) );
  if (caseless)
    strupper( old );

  s = 0;
  attrib = (option.all) ? (FA_SYSTEM | FA_HIDDEN) : 0;
  if (option.dir)
    attrib |= FA_DIREC;
  done = efindfirst( old, mask, &ff, attrib );
  while (!done)
  {
    if (option.list)
    {
      if (option.test)
      {
	for (j = 1; j < subpats; ++j)
	{
	  if (j != 1)
	    putchar( ':' );
	  printf( "%.*s", subpat[j].len, subpat[j].str );
	}
	putchar( '\n' );
      }
      else
      {
	fputs( path.old, stdout );
	puts( ff.name );
      }
    }
    else
    {
      ++names;
      s += strlen( ff.name ) + 1;
    }
    done = efindnext( old, &ff );
  }
  if (option.list)
    return E_OK;

  if (!names)
  {
    puts( "No matching files." );
    return E_NOFILES;
  }
  name = malloc( names * sizeof(s_name) + s );
  if (!name)
  {
    puts( "Not enough memory." );
    return E_NOMEM;
  }

  rc = E_OK;
  p = (char*)(name + names);
  total = names;
  if (total > 100)
  {
    pc = 1;
    _cputs( "Processing:   0%\b" );
  }
  names = max = 0;
  fname = stpcpy( fn, path.new );
  efindfirst( old, mask, &ff, attrib );
  do
  {
    name[names].old = p;
    p = stpcpy( p, ff.name ) + 1;
    n = fname;
    for (s = 0; s < subs; ++s)
    {
      begin = n;
      for (j = sub[s].first; j <= sub[s].last; ++j)
      {
	memcpy( n, subpat[j].str, subpat[j].len );
	n += subpat[j].len;
      }
      *n = '\0';
      if (sub[s].flag && *begin)
      {
	if (sub[s].flag == '-')
	{
	  strlower( begin );
	}
	else if (sub[s].flag == '+')
	{
	  strupper( begin );
	}
	else // (sub[s].flag == '^')
	{
	  *begin = upper( *begin );
	  strlower( begin + 1 );
	}
      }
    }
    // Ignore identical names.
    if (!new_path && !strcmp( ff.name, fname ))
      continue;

    // Check if the new name already exists (and is different to the original).
    if (file_exists( fn ) && (new_path || strcomp( ff.name, fname )))
    {
      int exists;
      if (option.force)
      {
	if (option.test)
	  exists = 1;
	else
	{
	  my_chmod( fn, 0 );
	  exists = my_unlink( fn );
	}
      }
      else
	exists = 1;
      if (exists)
      {
	rc = E_EXISTS;
	if (pc) putch( '\r' );
	printf( "File exists: %s%s ==> %s\n", path.old, ff.name, fn );
	if (!(option.test && option.force))
	  return rc;
      }
    }

    name[names].new = strdup( fname );
    if (name[names].new == NULL)
    {
      if (pc) putch( '\r' );
      puts( "Not enough memory." );
      return E_NOMEM;
    }

    if (option.test && !option.force)
    {
      len = strlen( ff.name );		// Determine the longest name
      if (len > max) max = len; 	//  to nicely align the display
    }

    ++names;
    if (pc)
      _cprintf( "\b\b\b%3d", (int)(100L * names / total) );
  }
  while (!efindnext( old, &ff ));

  if (pc) putch( '\r' );

  // Sort the names to easily determine conflicts.
  qsort( name, names, sizeof(s_name), cmp_name );
  for (j = 1; j < names; ++j)
  {
    if (strcomp( name[j-1].new, name[j].new ) == 0)
    {
      rc = E_CONFLICT;
      printf( "\"%s\" and \"%s\" conflict ==> %s\n",
	      name[j-1].old, name[j].old, name[j].new );
      if (!(option.test && option.force))
	return rc;
    }
  }

  if (option.test && option.force)
    return rc;

  if (!option.test)
  {
    fname = stpcpy( old, path.old );
    n	  = stpcpy( new, path.new );
    open_undo_file( "w" );
    if (pc)
      _cputs( "Renaming:   0%  \b\b\b" );
  }
  for (j = 0; j < names; ++j)
  {
    if (option.test)
    {
      printf( "%s%-*s ==> %s%s\n", path.old, max, name[j].old,
				   path.new, name[j].new );
    }
    else
    {
      strcpy( fname, name[j].old );
      strcpy( n,     name[j].new );
      if (my_rename( old, new ) != 0)
      {
	if (pc) putch( '\r' );
	printf( "Error: %s ==> %s\n", old, new );
	rc = E_ERROR;
	if (pc)
	  _cputs( "Renaming:    %\b" );
      }
      else
      {
	fprintf( undo_fp, "%s\n%s\n", new, old );
      }
      if (pc)
	_cprintf( "\b\b\b%3d", (int)(100L * (j+1) / total) );
    }
  }
  if (!option.test)
  {
    if (pc)
      _cputs( "\r              \r" );
    fclose( undo_fp );
  }

  return rc;
}


/*
   Separate the path from the pattern.

   Returns a pointer to the pattern and sets drive appropriately.
*/
char* locate_pattern( const char* path, int* drive )
{
  const char* pat;
  const char* p;

  pat = p = path;
  if (p[1] == ':')
  {
    *drive = upper( *p ) - 'A';
    // If it's not a legitimate drive letter, assume a
    // subpattern separator and no path.
    if (*drive < 0 || *drive > 25)
      return (char*)path;
    p += 2;
    pat = p;
  }
  else
  {
    *drive = getdisk();
  }

  for (; *p; ++p)
    if (*p == '/' || *p == '\\')
      pat = p + 1;

  return (char*)pat;
}


/*
   Compile the substitute pattern. Translates each subpattern and wildcard
   into a subpattern number (or a range of numbers) and a case conversion
   flag. Literals in the substitution are turned into subpatterns.

   Assumes the arrays will not overflow.
*/
void compile_subst( const char* pat, char* buf )
{
  int  s = subpats;
  char c;
  int  text = 1;

  subs = 0;
  while (*pat)
  {
    c = *buf++ = *pat++;
    switch (c)
    {
      case '*': c = 0; break;
      case '?': c = 1; break;
      case '#': c = 2; break;
      case '[': c = 3; break;
      case '=': c = 4; break;
    }
    if (c < 5)
    {
      text = 1;
      if (smetas[c] < metas[c])
      {
	sub[subs].first = sub[subs].last = meta[c][smetas[c]++];
	sub[subs].flag = 0;
	++subs;
      }
    }
    else if (c != ':')
    {
      if (c == '$')
      {
	c = *pat;
	if (c == '-' || c == '+' || c == '^')
	{
	  sub[subs].flag = c;
	  c = *++pat;
	}
	else
	{
	  sub[subs].flag = 0;
	}
	c = subnum( &pat );
	if (c != -1)
	{
	  sub[subs].first = sub[subs].last = c;
	  if (*pat == '+')
	  {
	    ++pat;
	    sub[subs].last = subpats - 1;
	  }
	  else if (*pat == '-')
	  {
	    ++pat;
	    c = subnum( &pat );
	    if (c > sub[subs].first)
	      sub[subs].last = c;
	  }
	  ++subs;
	  text = 1;
	  continue;
	}
	buf[-1] = *pat++;	// Not a legitimate subpattern, process as text
      }
      if (text)
      {
	text = 0;
	sub[subs].first = sub[subs].last = s;
	sub[subs].flag = 0;
	++subs;
	subpat[s].str = buf - 1;
	subpat[s++].len = 0;
      }
      ++subpat[s-1].len;
    }
  }
}


/*
   Translate one or two digit characters, or a metacharacter, to a number.

   s points to the string, which is updated to the end of the number.
   Returns the number or -1 if none.
*/
int subnum( const char** s )
{
  const char* n = *s;
  char c = *n;

  if (c >= '0' && c <= '9')
  {
    c -= '0';
    if (c >= subpats)
      c = -1;
    else if (*++n >= '0' && *n <= '9' && c * 10 + *n - '0' < subpats)
      c = c * 10 + *n++ - '0';
    *s = n;
  }
  else
  {
    switch (c)
    {
      case '*': c = 0;  break;
      case '?': c = 1;  break;
      case '#': c = 2;  break;
      case '[': c = 3;  break;
      case '=': c = 4;  break;
      default:	c = -1; break;
    }
    if (c != -1)
    {
      if (c >= 2 && (*s)[-1] == '$')
	c = -1;
      else
      {
	c = (smetas[c] < metas[c]) ? meta[c][smetas[c]++] : -1;
	++*s;
      }
    }
  }

  return c;
}


/*
   Determine the name of the undo file and open it.
*/
void open_undo_file( const char* mode )
{
  char* tmp = getenv( "TEMP" );
  if (tmp == NULL)
  {
    tmp = getenv( "TMP" );
    if (tmp == NULL)
      tmp = "C:\\";
  }
  tmp = stpcpy( undo_name, tmp );
  if (tmp[-1] != '\\' && tmp[-1] != '/')
    *tmp++ = '\\';
  strcpy( tmp, "wrenundo.lst" );

  undo_fp = fopen( undo_name, mode );
  if (undo_fp == NULL)
  {
    if (*mode == 'r')
    {
      puts( "Unable to open the undo file!" );
      exit( E_ERROR );
    }
    puts( "Warning: unable to create the undo file." );
    undo_fp = fopen( "NUL", "w" );
  }
  else
  {
    if (*mode == 'r')
      fscanf( undo_fp, "%d\n", &names );
    else
      fprintf( undo_fp, "%d\n", names );
  }
}


/*
   Undo the previous rename. Assumes the current directory is the same as the
   initial rename. Will not redo.
*/
void undo_rename( void )
{
  int rc = E_OK;
  int pc;

  open_undo_file( "r" );
  pc = (names > 100);
  if (pc)
    _cputs( "Restoring:   0%\b" );

  while (fgets( new, sizeof(new), undo_fp ) != NULL)
  {
    fgets( old, sizeof(old), undo_fp );
    new[strlen( new ) - 1] = '\0';
    old[strlen( old ) - 1] = '\0';
    if (my_rename( new, old ) != 0)
    {
      if (pc) putch( '\r' );
      printf( "Error: %s ==> %s\n", new, old );
      rc = E_ERROR;
      if (pc)
	_cputs( "Restoring:    %\b" );
    }
    if (pc)
      _cprintf( "\b\b\b%3d", (int)(100L * pc++ / names) );
  }
  unlink( undo_name );

  if (pc)
    _cprintf( "\r               \r" );

  exit( rc );
}


/*
   Process the command line options.

   Returns the number of the pattern.
*/
int parse_opt( int argc, char** argv )
{
  int o;
  int c;

  if (argc == 1 || !strcmp( argv[1], "/?" ) ||
      (*argv[1] == '-' && lower( argv[1][1] ) == 'h' ||
       !strcomp( argv[1] + 1, "help" )))
    help( argc == 1 );	// does not return

  for (o = 1; o < argc; ++o)
  {
    if (*argv[o] != '-')
      break;
    for (c = 1; argv[o][c]; ++c)
    {
      switch (argv[o][c])
      {
	case 'a': option.all   = 1; break;
	case 'c': caseless     = 0; break;
	case 'd': option.dir   = 1; break;
	case 'F': option.force = 1; break;
	case 't': option.test  = 1; break;
	case 'u': undo_rename();    break;      // does not return
	default:
	  printf( "No such option: %c\n", argv[o][c] );
	  exit( E_OPT );
      }
    }
  }
  if (o == argc)
  {
    puts( "It would help to specify a pattern.\n" );
    exit( E_OPT );
  }

  if (o + 1 == argc)
    option.list = 1;

  return o;
}


/*
   Compare two new names for the sort.
*/
int cmp_name( const void* n1, const void* n2 )
{
  return strcomp( ((const s_name*)n1)->new, ((const s_name*)n2)->new );
}
