/*
   lfn.c - Long filename functions for WRen.

   Jason Hood, 14 June, 2000 & 25 November, 2002.
*/


#include "lfn.h"
#include <dos.h>
#include <string.h>
#include <stdio.h>
#include <io.h>

#define CF 1                 // Carry flag


int use_lfn;		     // Non-zero if LFN is available


/*
   Determine if LFN is available on the current drive. If so, assume it is
   (not) available on all drives.
*/
int init_lfn( void )
{
  struct REGPACK regs;
  static char* buf = "?:\\";

  *buf = getdisk() + 'A';
  regs.r_ax = 0x71a0;		// Get Volume Information
  regs.r_ds = FP_SEG( buf );
  regs.r_dx = FP_OFF( buf );
  regs.r_cx = 0;		// Ignore the volume name (zero-size buffer)
  regs.r_bx = 0;		// Initialise, for older DOS compatibility
  intr( 0x21, &regs );
  use_lfn = (regs.r_bx & 0x4000);

  return use_lfn;
}


int lfn_findfirst( const char* spec, struct lfn_ffblk* ffblk, int attrib )
{
  struct REGPACK regs;

  regs.r_ax = 0x714e;		// LFN Find First Matching File
  regs.r_cx = attrib;
  regs.r_si = 1;                // DOS time format
  regs.r_ds = FP_SEG( spec );
  regs.r_dx = FP_OFF( spec );
  regs.r_es = FP_SEG( ffblk );
  regs.r_di = FP_OFF( ffblk );
  intr( 0x21, &regs );
  if (regs.r_flags & CF)
    return -1;
  ffblk->handle = regs.r_ax;
  return 0;
}


int lfn_findnext( struct lfn_ffblk* ffblk )
{
  struct REGPACK regs;

  regs.r_ax = 0x714f;		// LFN Find Next Matching File
  regs.r_bx = ffblk->handle;
  regs.r_si = 1;
  regs.r_es = FP_SEG( ffblk );
  regs.r_di = FP_OFF( ffblk );
  intr( 0x21, &regs );
  if (regs.r_flags & CF)
  {
    regs.r_ax = 0x71a1; 	// LFN FindClose
    regs.r_bx = ffblk->handle;
    intr( 0x21, &regs );
    return -1;
  }
  return 0;
}


int my_findfirst( const char* pathname, struct my_ffblk* ffblk, int attrib )
{
  int rc;

  if (use_lfn)
  {
    ffblk->name = ffblk->ff.lf.fname;
    rc = lfn_findfirst( pathname, &ffblk->ff.lf, attrib );
  }
  else
  {
    ffblk->name = ffblk->ff.df.ff_name;
    rc = findfirst( pathname, &ffblk->ff.df, attrib );
  }

  return rc;
}


int my_findnext( struct my_ffblk* ffblk )
{
  return (use_lfn) ? lfn_findnext( &ffblk->ff.lf )
		   :	 findnext( &ffblk->ff.df );
}


int my_rename( const char* old, const char* new )
{
  int rc;
  struct REGPACK regs;

  if (use_lfn)
  {
    regs.r_ax = 0x7156; 	// LFN Rename File
    regs.r_ds = FP_SEG( old );
    regs.r_dx = FP_OFF( old );
    regs.r_es = FP_SEG( new );
    regs.r_di = FP_OFF( new );
    intr( 0x21, &regs );
    rc = (regs.r_flags & CF) ? -1 : 0;
  }
  else
  {
    rc = rename( old, new );
  }
  return rc;
}


int file_exists( const char* file )
{
  int rc;
  struct REGPACK regs;

  if (use_lfn)
  {
    regs.r_ax = 0x7143; 	// Extended Get/Set File Attributes
    regs.r_bx = 0;		// Get attributes
    regs.r_ds = FP_SEG( file );
    regs.r_dx = FP_OFF( file );
    intr( 0x21, &regs );
    rc = (regs.r_flags & CF) ? 0 : 1;
  }
  else
  {
    rc = (access( file, 0 ) == 0);
  }

  return rc;
}
