/*****************************************************************************
* Program:  Trie --- Create a trie in a 'C' compiler readable format.
*
* Author :  John W. M. Stevens
*****************************************************************************/

#include    <stdio.h>
#include    <stdlib.h>
#include    <fcntl.h>
#include    <string.h>
#include    <strings.h>

/* Type definition. */
typedef struct  {
    char    c;
    char    *val;
    struct  trie    *branch;
} KEY;

typedef struct  trie    {
    int     NoEls;
    int     Size;
    char    Lbl[10];
    KEY     Keys[1];
} TRIE;

static  TRIE    *Head = NULL;

/*----------------------------------------------------------------------------
| Routine:  TrieIns() --- Insert a string into the trie.
|
| Inputs :  Trie    - The trie to insert into.
|           str - The string to insert.
|           val - The value string for this key string.
| Outputs:  Returns the pointer to the trie level.
----------------------------------------------------------------------------*/

static  long    LblNo = 0L;

TRIE    *TrieIns(TRIE   *Trie,
                 char   *str,
                 char   *val)
{
    register    int     i;
    register    int     j;

    auto        int     hi;
    auto        int     lo;
    auto        int     ret;

    /* Create a level if neccesary. */
    if (Trie == NULL)
    {
        /* Calculate the size of the level in bytes.    */
        i = sizeof( TRIE ) + 4 * sizeof( KEY );

        /* Allocate a level.    */
        if ((Trie = malloc( i )) == NULL)
        {
            perror( "TrieIns (malloc) " );
            return( NULL );
        }

        /* Clear the memory to all zeros.   */
        memset((char *) Trie, '\0', i);

         /* Initialize the new trie level.   */
        Trie->NoEls = 1;
        Trie->Size  = 5;
        Trie->Keys[0].c = *str++;
        sprintf(Trie->Lbl, "%x", LblNo++);

        /* End of string?   */
        if ( *str )
        {
            Trie->Keys[0].val = "0";
            Trie->Keys[0].branch = TrieIns(NULL, str, val);
        }
        else
            Trie->Keys[0].val = val;
        return( Trie );
    }

    /* Search for the current character of the string to insert.    */
    hi = Trie->NoEls - 1;
    lo = 0;
    do
    {
        /* Find mid point of current array piece.   */
        i = (hi + lo) >> 1;

        /* Do character comparison. */
        ret = *str - Trie->Keys[i].c;

        /* Shrink the search limits.    */
        if (ret <= 0)
            hi = i - 1;
        if (ret >= 0)
            lo = i + 1;

    }  while (hi >= lo);

    /* Did we find the character?   */
    if (ret == 0)
    {
        /* Yes, continue the search.    */
        if ( *++str )
            Trie->Keys[i].branch = TrieIns(Trie->Keys[i].branch, str, val);
        return( Trie );
    }

    /* If the trie level is not big enough, reallocate it.  */
    if (Trie->Size == Trie->NoEls)
    {
        /* Calculate the size of the new level. */
        i = sizeof( TRIE ) + (5 + Trie->Size) * sizeof( KEY );

        /* Attempt to reallocate the trie level.    */
        if ((Trie = realloc(Trie, i)) == NULL)
        {
            perror( "TrieIns (realloc) " );
            return( NULL );
        }

        /* Up the size of the trie level.   */
        Trie->Size += 5;
    }

    /* Make room for new character. */
    if (hi < Trie->NoEls - 1)
    {
        j = sizeof( KEY ) * (Trie->NoEls - lo);
         memmove(Trie->Keys + lo + 1,
                Trie->Keys + lo,
                j);
    }

    /* Insert new object.   */
    i = lo;
    Trie->Keys[i].c      = *str++;
    Trie->Keys[i].branch = NULL;
    Trie->NoEls++;

    /* Continue insert if not at end of string. */
    if ( *str )
    {
        Trie->Keys[i].val = "0";
        Trie->Keys[i].branch = TrieIns(NULL, str, val);
    }
    else
        Trie->Keys[i].val = val;
    return( Trie );

}

/*----------------------------------------------------------------------------
| Routine:  TrieDump() --- Dump the trie in a form usable by the 'C'
|           compiler.
|
| Inputs :  Trie    - The trie to dump.
|           Lable   - Trie level label.
| Outputs:  Returns the pointer to the trie level.
----------------------------------------------------------------------------*/

void    TrieDump(TRIE   *Trie)
{
    register    int     i;
    register    int     j;
    auto        char    NxtLbl[12];

    /* Do a post order traversal of tree.   */
    for (i = j = 0; i < Trie->NoEls; i++)
    {
        /* Check for children.  */
        if (Trie->Keys[i].branch == NULL)
            continue;

        /* Do current branch.   */
        TrieDump( Trie->Keys[i].branch );
    }

    /* Print label for this level.  */
    printf("static\nKEY\tT%s[%d] = {\n", Trie->Lbl, Trie->NoEls + 1);
    printf("\t{\t' ',\t%15d,\t%-20s\t}", Trie->NoEls + 1, "NULL");

    /* Dump current level.  */
    for (i = j = 0; i < Trie->NoEls; i++)
    {
        /* Check for children.  */
        printf(",\n\t");
        if (Trie->Keys[i].branch == NULL)
        {
            printf("{\t'%c',\t%15s,\t%-20s\t}", Trie->Keys[i].c,
            Trie->Keys[i].val, "NULL");
            continue;
        }

        /* Print the key.   */
         printf("{\t'%c',\t%15s,\tT%-19s\t}", Trie->Keys[i].c,
            Trie->Keys[i].val, Trie->Keys[i].branch->Lbl);
    }
    printf("\n};\n\n");

}

void    main(int    argc,
         char   *argv[])
{
    auto    FILE    *fp;
    auto    char    bf[256];
    auto    char    *tp;
    auto    char    *token;
    auto    char    *val;

    /* Check the number of command line parameters. */
    if (argc != 2)
    {
        fprintf(stderr, "Syntax error: trie <token file>\n");
        exit( -1 );
    }

    /* Open the token file for reading. */
    if ((fp = fopen(argv[1], "rt")) == NULL)
    {
        perror( "trie (fopen) " );
        exit( -1 );
    }

    /* Get and insert all strings into the trie.    */
    while ( fgets(bf, 256, fp) )
    {
        /* Strip white space.   */
        tp = bf;
        while (*tp == ' ' || *tp == '\t')
            tp++;

        /* Find end of token.   */
        token = tp;
        while (*tp != ' ' && *tp != '\t' && *tp != '\n')
            tp++;
        *tp++ = '\0';

        /* Strip white space.   */
        while (*tp == ' ' || *tp == '\t')
            tp++;

        /* Find end of the value string.    */
        val = tp;
        while (*tp != ' ' && *tp != '\t' && *tp != '\n')
            tp++;
        *tp = '\0';

        /* Duplicate the value string.  */
        if ((val = strdup( val )) == NULL)
        {
            perror( "trie (strdup) " );
            exit( -1 );
        }

        /* Do string insert into trie.  */
        Head = TrieIns(Head, token, val);
    }

    TrieDump( Head );
}
