#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "cmdline.h"
#include "umem.h"
#include "module.h"
#include "data.h"
#include "section.h"
#include "public.h"
#include "extern.h"
#include "maps.h"
#include "linker.h"
#include "lexpr.h"
#include "partit.h"
#include "errors.h"
#include "input.h"
#include "libs.h"
#include "lines.h"

#define VERSION 145

extern int serrcount;
extern int errors;

#ifdef ALTERNATE
extern char formext[];
#endif

/* Command line parameters */
BOOL prm_autodepends = FALSE;
uint prm_mapfile = MAP_NORMAL;		/* Map file type */
BOOL prm_case_sensitive = TRUE;		/* True if case sensitive */
uint prm_errcount = 25;			/* Max errors to collect */
char *prm_searchpath = 0;		/* Search path for libraries */
BOOL prm_nodefaultlibs = FALSE;		/* TRUE if to ignore default lib records */
BOOL prm_response = FALSE;		/* TRUE if response file is loaded */
BOOL prm_overlapnames = FALSE;		/* TRUE if overlays need seperate name pools */
BOOL prm_relocatable = FALSE;		/* TRUE if to output a relocatable file */
BOOL prm_debug = FALSE;			/* currently unsupported */
char *prm_specify = 0;			/* specification file */

/* Global varieables */
uint modnumber=0;		    	/* module number, increments sequentially */ 
LIST *liblist = 0;			/* List of libraries */
LIST *objlist = 0;			/* List of object files */
char *exefile = 0;			/* EXE file name */
char *mapfile = 0;			/* MAP file name */

char *usage_text = "[/a/c/f/n/o/p/r/s/v/x/E/L] file list [@filename]";

void BoolSetup(char select, char *string);
void ErrorSetup(char select, char *string);
void SearchSetup(char select, char *string);
void SpecifierSetup(char select, char *string);
void ObjectSetup(char select, char *string);

ARGLIST ArgList[] = {
  { 'a', ARG_BOOL, BoolSetup },
  { 'c', ARG_BOOL, BoolSetup },
  { 'f', ARG_CONCATSTRING, SpecifierSetup },
  { 'n', ARG_BOOL, BoolSetup },
  { 'o', ARG_CONCATSTRING, ObjectSetup },
  { 'p', ARG_BOOL, BoolSetup },
  { 'r', ARG_BOOL, BoolSetup },
  { 's', ARG_BOOL, BoolSetup },
  { 'u', ARG_BOOL, BoolSetup },
  { 'v', ARG_BOOL, BoolSetup },
  { 'x', ARG_BOOL, BoolSetup },
  { 'E', ARG_CONCATSTRING, ErrorSetup },
  { 'L', ARG_CONCATSTRING, SearchSetup },
  { 0, 0, 0 }
} ;
/*
 * Setup for boolean command line args
 */
static void BoolSetup(char select, char *string)
{
  switch(select) {
    case 'u':
			prm_case_sensitive = FALSE;
			break;
    case 'a':
			prm_autodepends = TRUE;
			break;
		case 'n':
			prm_nodefaultlibs = TRUE;
			break;
		case 'p':
			prm_mapfile = MAP_PUBLICS;
			break;
		case 'r':
			prm_relocatable = TRUE;
			break;
		case 's':
			prm_mapfile = MAP_EXTENDED;
			break;
		case 'x':
			prm_mapfile = MAP_NONE;
			break;
		case 'v':
			prm_debug = TRUE;
			break;
  }
}
/*
 * Setup for the /E switch
 */
static void ErrorSetup(char select, char *string)
{
  prm_errcount = atoi(string);
}
/*
 * Setup for library search paths /L
 */
static void SearchSetup(char select, char *string)
{
  uint len = strlen(string)+1;
  prm_searchpath = (char *)AllocateMemory(len);
  strcpy(prm_searchpath, string);
}
/*
 * Setup for specifier file name
 */
static void SpecifierSetup(char select, char *string)
{
  uint len = strlen(string)+1;
	if (prm_specify)
		DeallocateMemory(prm_specify);
  prm_specify = (char *)AllocateMemory(len);
  strcpy(prm_specify, string);
}
/*
 * Set up for object file name
 */
static void ObjectSetup(char select, char *string)
{
	if (exefile)
		DeallocateMemory(exefile);
	exefile = AllocateMemory(strlen(string)+1);
	strcpy(exefile, string);
}
/*
 * Insert a file onto one of the lists.  .LIB files go on library list,
 *   anything else is assumed an .o file regardless of extension
 */
static char *InsertAnyFile(char *filename)
{
  char *newbuffer, buffer[100],*bp=buffer;

  /* Allocate buffer and make .o if no extension */
	while (*filename > 32)
		*bp++ = *filename++;
	*bp++ = 0;
	
  AddExt(buffer,".o");
  newbuffer = (char *) AllocateMemory(strlen(buffer)+1);
  strcpy(newbuffer,buffer);

  /* Insert file */
  if (strstr(newbuffer,".lib"))
    AppendToList(&liblist,newbuffer);
  else
	  AppendToList(&objlist,newbuffer);
	return(filename);
}
/*
 * Read a line of ascii text from a file
 *   Get rid of \n
 */
static void ReadLine(char *buffer, int count, FILE *file, char *name)
{
  char *pos;
  *buffer = 0;
  fgets(buffer,count,file);
  pos = buffer + strlen(buffer) -1;
  /* The test is needed because my editor doesn't put CR/LF at the end of file */
	if (*pos <32)
    *pos = 0;
}
/*
 * Check if trying to create an EXE or MAP file with the wrong extension
 */
static BOOL CheckInvalidExtension(char *buffer)
{
  if (strstr(buffer,".o"))
    return(TRUE);
  if (strstr(buffer,".lib"))
    return(TRUE);
  return(FALSE);
}
/*
 * Read the response file
 *   FIRST LINE: out FILE
 *   SECOND LINE: EMPTY OR MAP FILE
 *   OTHER LINES: OBJ AND LIB files in any order
 */
static void ReadResponse(char *filename)
{
  FILE *in;
  char buf[1024],*buffer;

  /* Open file */
  if ((in = fopen(filename,"r")) ==0)
    fatal("Missing or invalid response file %s", filename);

  /* Read EXE file name */
  ReadLine(buf,1024,in,filename);
	buffer = buf;
	while (*buffer == 32)
		buffer++;

  if (buffer[0] < 32)
    fatal("No output file specified");
	if (prm_relocatable)
		AddExt(buffer,".out");
	else
		AddExt(buffer,".abs");
	if (exefile)
		DeallocateMemory(exefile);
  exefile = (char *) AllocateMemory(strlen(buffer+1));
	strcpy(exefile, buffer);

  /* Read and verify map filename */
  ReadLine(buf,1024,in,filename);
	buffer = buf;
	while (*buffer == 32)
		buffer++;
  if (buffer[0] > 32) {
    if (CheckInvalidExtension(buffer) || strstr(buffer,".EXE"))
      fatal("Invalid MAP filename %s\n", buffer);
		AddExt(buffer,".MAP");
    mapfile = (char *)AllocateMemory(strlen(buffer+1));
	  strcpy(mapfile, buffer);
  }
  else
    prm_mapfile = MAP_NONE;

  /* Read and queue object and library file names */
  while (!feof(in)) {
    char *p = buffer;
    ReadLine(buffer,1024,in,filename);
    if (buffer[0] < 32)
      continue;
    while (*p)
      *p++ = (char)tolower(*p);
		p = buffer;
    while (TRUE) {
			while (*p == 32)
				p++;
			if (*p < 32)
				break;
			p = InsertAnyFile(p);
		}
  }
  fclose(in);
}
/* 
 * Pass 1: Read the object files
 *   Also does auto-library determination
 */  
static void Pass1Objects(void)
{
  modnumber = 0;
  if (objlist) {
    LIST *p = objlist;
    while (p) {
      FILE *in;
      if ((in = fopen((char *)p->data,"rb")) == 0)
				fatal("Missing input module %s",(char *) p->data);
#ifdef DEBUG
			printf("Pass 1:Processing module %x:%s\n", modnumber,(char *)p->data);
#endif /* DEBUG    */
			ReadModule(in,(char *)p->data,SCAN,TRUE);
			p=p->link;
		  fclose(in);
			modnumber++;
		}
  }
}
/*
 * Pass2 , read object files
 *   Libraries have already been found and moved at this point
 */
static void Pass2Objects(void)
{
  modnumber = 0;
  if (objlist) {
    LIST *p = objlist;
    while (p) {
      FILE *in;
      if ((in = fopen((char *)p->data,"rb")) == 0)
				fatal("Missing input module %s",(char *) p->data);
#ifdef DEBUG
			printf("Pass 2: Processing module %x:%s\n", modnumber,(char *)p->data);
#endif /* DEBUG     */
			ReadModule(in,(char *)p->data,RESOLVE,TRUE);
			p = p->link;
		  fclose(in);
			modnumber++;
		}
  }
}
/*
 * Main routine
 *   Read command line
 *   Make EXE and MAP filenames if not already extant
 *   Pass 1 init
 *   Pass 1
 *   Pass 1 rundown
 *   pass 2 init
 *   Pass 2
 *   Pass 2 rundown
 */
int main(int argc, char *argv[])
{
  int i;
	FILE *file;
  banner(VMSG("LINK "));

	MemoryInit(0xffff);
  /* Scan command line for switches */
  if (!parse_args(&argc,argv,TRUE) || (argc == 1))
    usage(argv[0]);

#ifdef ALTERNATE
	if (!prm_relocatable)
		formatinit(argv[0]);
#endif
  /* Scan the command line for file names or response files */
  for (i=1; i < argc; i++) {
    char *p = argv[i];
    while (*p)
      *p++ = (char)tolower(*p);
    if (argv[i][0] == '@') {
      if (prm_response)
        fatal("Too many response files");
      prm_response = TRUE;
      ReadResponse(&argv[i][1]);
    }
    else
		  InsertAnyFile(argv[i]);
  }
	/* Read the spec file */
	if (prm_specify) {
		openFile(prm_specify,"r");
		yyparse();
		if (serrcount)
			exit(1);
		if (prm_relocatable)
			Error("Ignoring '/r' switch");
		prm_relocatable = FALSE;
	}

  /* If no response file, make up EXE and MAP file names from first .o file */
  if (!prm_response) {
    LIST *p = objlist;
	  char buffer[100];

    /* If no obj file go with the first lib file.  Well, we're guaranteed */
    /* to have something if we got past the usage! */
    if (!objlist)
      p = liblist;


    /* EXE file name */
		if (!exefile) {
    	strcpy(buffer, (char *)p->data);
			StripExt(buffer);
			if (prm_relocatable)
				AddExt(buffer,".out");
			else
#ifdef ALTERNATE
				AddExt(buffer,formext);
#else
				AddExt(buffer,".abs");
#endif
	  	exefile= AllocateMemory(strlen(buffer) +1);
    	strcpy(exefile,buffer);
		}
		else
			strcpy(buffer,exefile);
    /* MAP file name */
		StripExt(buffer);
    AddExt(buffer,".MAP");
	  mapfile= AllocateMemory(strlen(buffer) +1);
    strcpy(mapfile,buffer);
  }


  /* Pass 1 |*/
	LocateLibraries(prm_searchpath, &liblist);
	LineInit();
  PublicTableInit();
  ExternTableInit();
	SectionTableInit();
	LocalTableInit();
	TypeTableInit();
  Pass1Objects();
	SearchLibraries(liblist);
	ExternTableRundown();
	ExternTableInit();

	/* Pass 2 */
	SetPartitions(RESOLVE);
	Pass2Objects();
	LoadLibraries();

	if (!(file = fopen(exefile,"wb")))
		fatal("Can't open output file");
	if (prm_relocatable)
		linkerout(file, exefile);
	else
#ifdef ALTERNATE
		format_out(file,exefile);
#else
		linkerout(file, exefile);
#endif
	fclose(file);
  WriteMapFile(mapfile,prm_mapfile);
	DeletePartitions();
	SectionRundown();
	SectionTableRundown();
  PublicTableRundown();
  ExternTableRundown();
	LocalTableRundown();
	TypeTableRundown();
	LineRundown();
#ifdef DEBUG
	printf("EXE: %s\n",exefile);
	printf("MAP: %s\n",mapfile);
	printf("SEARCH: %s\n",prm_searchpath);
#endif 
  return(errors);
}