//                            dynprec.cc
//					-- by William T. Wyatt
//
// This program demonstrates the use of the Super Precision Library to
// carry out super_precision class object computations in several different
// precisions, "dynamically".

//              Copyright (c) 1997 William T. Wyatt, Jr.

#include "super.cc"

int main ( )
{
// The term "working precision" is meant to identify the current or active
// "default precision" and "default radix" used for super_precision class
// object allocation and computation.  These values are kept in class static
// variables and are defined by super_precision::setup().  Only one "working
// precision" can be used in computations by the Super Precision Library AT
// A GIVEN TIME, except in a few protected functions.  This version of the
// Library does not automaticaly change working precision to conform to
// super_precision operands that are passed to the library.  This change
// in working precision must be initiated by the user's program.  Library
// functions do, however, test operands for conformity to this rule.
//
// Yet, several working precisions can be used successfully in a program.
// Several strategies are possible.  In the following, deallocation is
// typically done when a "return" is executed in the function that allocated
// the super_precision class objects.  Alternatively, objects that are
// allocated by a "new" operator can be deallocated by a "delete" operator
// anywhere in the program.  Of course, all objects are deallocated at the
// end of a program.
// In a single program, operations can be organized in the following ways:
// SEGMENTED FLOW:
// (1) initialize default precision (invoke super_precision::setup(...),
//     allocate and use super_precision class objects at the default
//     precision (including input and/or output), deallocate super_precision
//     class objects;
// (2) repeat (1) as many times as needed, choosing the default precision
//     each time to be whatever is desired.  Objects may be converted to
//     ASCII strings and these strings may be placed in an output stream
//     at the end of the program, for example, to allow "side-by-side"
//     comparison of results at the program's conclusion.
// In this segmented flow example, super_precision class object values can
// be passed from one segment to the next by conversion to ASCII strings
// maintained in memory or by writing and reading the strings to/from disk.
// The library provides functions to perform conversion to and from ASCII
// strings.  Super_precision objects with different precision do not coexist
// (i.e., simultaneously in memory) in segmented-flow programs.
//
// NESTED FLOW (2-level example):
// (1) initialize working precision, allocate and use super_precision class
//     objects at this working precision, but do NOT deallocate super_precision
//     class objects (yet);
// (2) initialize a second working precision, allocate and use these objects
//     at this new working precision, and deallocate the objects allocated in
//     THIS step;
// (3) reinitialize the first working precision (for step 1), use objects
//     allocated in step (1), and finally deallocate the objects allocated
//     in step (1).
// Deeper nests may be used.  Super_precision objects with different precision
// coexist in memory, but are not used with each other in Super Precision
// Library functions -- except for a few functions that convert class objects
// from one precision (or radix) to another.  Thus class object values can be
// transferred from one object to another possessing a different precision by
// using ASCII strings as intermediaries, or by using conversion functions.
//
// INTERWOVEN FLOW:
// (1) initialize a working precision and allocate some objects, initialize
//     another working precision and allocate more objects, etc.
// (2) reinitialize a working precision and compute with objects possessing
//     that working precision, repeat for another working precision, etc.
// (3) deallocate each object.
// This is much trickier because the user program must set up the correct
// working precision with super_precision::setup() BEFORE using the
// corresponding objects in a library function call.  There is some overhead
// associated with many repeated calls to super_precision::setup(), as well.
//
// SHREDDED FLOW:
// This is similar to interwoven flow, except that allocation and deallocation
// may be scattered throughout the program flow.  Computation is interspersed
// throughout, also.  There is overhead associated with repeated calls to
// super_precision::setup(), but also intensive use of runtime memory
// management routines.  Because memory blocks released and requested may not
// "fit" well, there may be progressive fragmentation of the Free Store.
//
// ALL OF THE ABOVE:
// Combinations of the above can also be used.  This sample program is a
// combination of several of the mentioned strategies.


// Declare a function prototype:
super_precision hyper_sqrt ( super_precision a);


//************************* SEGMENTED FLOW ****************************
//*************************    EXAMPLE     ****************************

	cout << "------------ SEGMENTED FLOW EXAMPLE ------------" << endl;
	char *out1;
	char *out2;
	out1 = new char [ 60];		// Get two strings for output.
	out2 = new char [ 70];		// Make sure they are long enough!!

//---------------------------------------------------------------------
// Start "segment 1":
	super_precision::setup ( 40);	// Request 40 decimal digits.
	super_precision *a_ptr = new super_precision;	// "new" usage
	
	*a_ptr = hyper_sqrt ( (super_precision) 1001);
	cout << "hyper_sqrt of 1001 to" << endl;
	cout << "  40 digits ----> " << setprecision (40) << *a_ptr << endl;

	// Convert to string that we save.  Allow extra space for blanks,
	// exponent, and decimal point!  Output digits in blocks of 10 for
        // readability.  Leave in scientific notation (use exponent).
	strcpy ( out1, super_output ( *a_ptr));
	// Note that the pointer returned by super_output points to a
	// string INTERNAL to the library.  This string will be altered the
	// next time super_output is invoked (including instances of "<<").
	// So we copy the ENTIRE STRING to *out1 immediately to save it.

	delete a_ptr;			// Deallocate "new" objects.
// End of "segment 1".
//---------------------------------------------------------------------

//---------------------------------------------------------------------
// Start "segment 2":
	super_precision::setup ( 50);	// Request 50 decimal digits.
	a_ptr = new super_precision;	// "new" usage, using old pointer
	
	*a_ptr = hyper_sqrt ( (super_precision) 1001);
	cout << "  50 digits ----> " << setprecision (50) << *a_ptr << endl;

	// Convert to string that we save.  Allow extra space for blanks,
	// exponent, and decimal point!  Output digits in blocks of 10 for
	// readability.  Leave in scientific notation (use exponent).
	strcpy ( out2, super_output ( *a_ptr));

	delete a_ptr;			// Deallocate "new" objects.
// End of "segment 2".
//---------------------------------------------------------------------

//---------------------------------------------------------------------
// Demonstrate final output of saved strings containing converted
// super_precision objects.
	cout << endl << "Final output (notice the different format!):" << endl;
	cout << "  40 digits ----> " << out1 << endl;
	cout << "  50 digits ----> " << out2 << endl;
// Demonstrate mixing of different precisions, by converting strings to
// super_precision objects possessing same precision.
	int string_base;	// Use string_base to receive the input radix.
	// The setup to 50 decimal digits is still in effect from "segment 2"!
	super_precision s = super_convert_string ( out1, string_base);
	if ( string_base != 10)
	{		// If the input base is not the default radix (10),
			// we could convert its base to it.  Like so:
		s = base_conv ( s, string_base, FALSE, FALSE);
	}
	super_precision t = super_convert_string ( out2, string_base);
	if ( string_base != 10)
	{
		t = base_conv ( t, string_base, FALSE, FALSE);
	}
	// Now output the difference of s and t.  setprecision is still 50.
	cout << endl << "Difference is " << t - s << endl;
//---------------------------------------------------------------------


//************************** NESTED FLOW ******************************
//**************************   EXAMPLE   ******************************

	cout << endl << "------------ NESTED FLOW EXAMPLE ------------" << endl;
// Reuse character pointers out1 and out2 and the strings they point to.
// Reuse super_precision pointer a_ptr.  Get another one called b_ptr.

//---------------------------------------------------------------------
// Start "nest 1":
	super_precision::setup ( 40);	// Request 40 decimal digits.
        a_ptr = new super_precision;			// "new" usage
	
	*a_ptr = hyper_sqrt ( (super_precision) 1001);
	cout << "hyper_sqrt of 1001 to" << endl;
	cout << "  40 digits ----> " << setprecision (40) << *a_ptr << endl;

//---------------------------------------------------------------------
// Start "nest 2":
	super_precision::setup ( 50);	// Request 50 decimal digits.
	super_precision *b_ptr = new super_precision;	// "new" usage
	
	*b_ptr = hyper_sqrt ( (super_precision) 1001);
	cout << "hyper_sqrt of 1001 to" << endl;
	cout << "  50 digits ----> " << setprecision (50) << *b_ptr << endl;

// Do comparison of *a_ptr and *b_ptr at 50-digit precision.
	super_precision *d_ptr = new super_precision;
			// convert *a_ptr to 50 digits
	*d_ptr = *b_ptr - precision_conv ( *a_ptr);
	cout << "Difference is " << *d_ptr << endl;

	delete b_ptr, d_ptr;
// End of "nest 2".
//---------------------------------------------------------------------

	delete a_ptr;	//"delete" doesn't care how many digits a_ptr points to!
	super_precision::setup ( 40);	// Restore 40 decimal digits.
	// Could do some more 40-digit stuff here.  Like the following.
	cout << "sqrt (2) = " << sqrt ( super_precision (2)) << endl;

// End of "nest 1".
//---------------------------------------------------------------------


//************************ INTERWOVEN FLOW ****************************
//************************     EXAMPLE     ****************************

	cout << endl;
	cout << "------------ INTERWOVEN FLOW EXAMPLE ------------" << endl;

//---------------------------------------------------------------------
// Allocate super_precision objects at several precisions.
	super_precision :: setup ( 40);		// Set up 40 digits.
	super_precision ax, *bx_ptr;		// Allocation for 40 digits.
	bx_ptr = new super_precision;		// Allocation for 40 digits.

	super_precision :: setup ( 50);		// Set up 50 digits.
	super_precision cy, dy, *ey_ptr;	// Allocation for 50 digits.
	ey_ptr = new super_precision;		// Allocation for 50 digits.

//---------------------------------------------------------------------
// Start calculations at one precision, say, 40 digits.
	super_precision :: setup ( 40);
	ax = 1001;
	*bx_ptr = hyper_sqrt ( ax);

// Start calculations at other precision, 50 digits.
	super_precision :: setup ( 50);
	dy = 1001;
	*ey_ptr = hyper_sqrt ( dy);

// Do comparison of *bx_ptr and *ey_ptr at 50-digit precision.
			// convert *bx_ptr to 50 digits
	cy = *ey_ptr - precision_conv ( *bx_ptr);
	cout << "Difference is " << cy << endl;

// Test difference at 40-digit precision.
	super_precision :: setup ( 40);
			// convert *ey_ptr to 40 digits
	cy = precision_conv ( *ey_ptr) - *bx_ptr;
			// We "know" (risky) that cy is "long enough" to hold
			// 40 digits, else we could declare a 40-digit variable
			// to receive the difference.
	cout << "Difference when operands are rounded to 40 digits is just ";
	cout << cy << endl;
	
//---------------------------------------------------------------------
// Deallocate as desired.
	delete bx_ptr, ey_ptr;
	

//************************ END OF EXAMPLES ****************************
//NOTE: This main uses a lot of "new" and "delete" operators to demonstrate
// the different program flow characteristics.  If the segments/nests were
// contained in called functions, it would be simpler to use objects instead
// of pointers to objects, and allow the objects to go out of scope when a
// return is done, instead of "new"ing and "delete"ing everything.
}

super_precision hyper_sqrt ( super_precision a)
{	// This function takes the square root of (a-1)/(a+1) at the current
	// working precision, and returns the answer.

	super_precision b, c;	// couple of scratch variables

	b = a + 1;
	c = ( a - 1) / b;

	return sqrt ( c);	// b and c are automatically deallocated
}
