DECLARE SUB Initialize_mnUnits (BYVAL nvGroup%)
' Updated:  October, 1996 by Christopher King
'
' ------------------------------------------------------------------------
'               Copyright (C) 1996 by Christopher King
'
' You have a royalty-free right to use, modify, reproduce and distribute
' the OneOhOne Files (and/or any modified version) in any way you find
' useful, provided that you agree that Christopher King has no warranty,
' obligations or liability for any OneOhOne Files.
' ------------------------------------------------------------------------
DECLARE FUNCTION nOptionButton (Index%, BYVAL nvGroup%, Src AS FORM) AS INTEGER
DECLARE SUB GasOption (BYVAL Index%, Src AS FORM)
DECLARE SUB SwitchBottom (BYVAL nvFarRightIndex%, BYVAL nvGroup%, Src AS FORM)
DECLARE SUB SwitchTop (BYVAL nvFarRightIndex%, BYVAL nvGroup%, Src AS FORM)
DECLARE SUB SwitchGeneral (BYVAL nvToFill%, BYVAL nvToEmpty%, BYVAL nvDistance%, BYVAL nvGroup%, Src AS FORM)
' Save as MolrMss2.bas when compile.
'$FORM frmChoose
'$FORM frmReaction
'$FORM frmIdealGas
DECLARE FUNCTION nGroupOfIndex (BYVAL Index AS INTEGER) AS INTEGER
'$FORM frmInfo
DECLARE FUNCTION bUnitConversion (BYVAL dvInNum AS DOUBLE, BYVAL dvInUnc AS DOUBLE, BYVAL nvUnitsIn%, BYVAL nvUnitsOut%, dOutNum AS DOUBLE, dOutUnc AS DOUBLE, nConvert%) AS INTEGER
'$FORM frmGas
DECLARE SUB CalculateFormula (BYVAL Index%)
DECLARE SUB Disconnect (BYVAL nvIndex%)
DECLARE SUB NumberFormat (BYVAL dNumber AS DOUBLE, BYVAL dStdDev AS DOUBLE, BYVAL TabStop%, srMsg AS STRING, rPrintX AS INTEGER)
'$FORM frmTitration
DECLARE SUB NextFocus (BYVAL nvIndex%, nrNextIndex%)
DECLARE SUB AddThousandsSeparators (srNeedCommas AS STRING, nrDecimalPosition%)
DECLARE FUNCTION nGetConnectColor () AS INTEGER
'$FORM frmGramMole
'formGone'$FORM frmDensity
'formgone'$FORM frmMolarity
'$FORM frmPercent
'formGone'$FORM frmDilution
' This module was created so that MOLARMSS.BAS will be small enough to
' compile without having to change memory settings.
' The routines moved here don't use the module level variables in MOLARMSS.BAS.
'
DECLARE FUNCTION bAnalyzeInput (BYVAL sText AS STRING, dNumber AS DOUBLE, dError AS DOUBLE) AS INTEGER
DECLARE FUNCTION nSetNextFocus (BYVAL nvIndex%) AS INTEGER
OPTION EXPLICIT
DEFINT A-Z
'$INCLUDE: 'common.bi'

CONST KEY_RETURN = 13           ' Enter key
CONST KEY_ESCAPE = 27
CONST KEY_UP = 38
CONST KEY_DOWN = 40
CONST KEY_BACK = 8
CONST KEY_DELETE = 127          ' Delete key returns 46 in Visual Basic for Windows.

' BackColor, ForeColor (form, controls)
'CONST BLACK = 0
CONST CYAN = 3
CONST WHITE = 7
'CONST YELLOW = 14

CONST BLUE = 1
CONST GREEN = 2
CONST RED = 4
CONST MAGENTA = 5
CONST BROWN = 6
CONST GRAY = 8
CONST BRIGHT_BLUE = 9
CONST BRIGHT_GREEN = 10
CONST BRIGHT_CYAN = 11
CONST BRIGHT_RED = 12
CONST PINK = 13
CONST BRIGHT_WHITE = 15

CONST ML = 5
CONST LITER = 7
CONST KELVIN = 9
CONST CELSIUS = 10
CONST FAHRENHEIT = 11
CONST PA = 12
CONST KPA = 13
CONST BAR = 14
CONST ATM = 15
CONST MMHG = 16
CONST TORR = 17
CONST PSI = 18

' Add thousands separators, as in 1,430,345.
' Called by NumberFormat.
'     Arguments:  srNeedCommas         String to which commas are added.
'                 nrDecimalPosition    Position of decimal in srNeedCommas
'                                      both before and after adding commas.
SUB AddThousandsSeparators (srNeedCommas AS STRING, nrDecimalPosition)
DIM N
   N = 5
   DO WHILE N < nrDecimalPosition + 1
      srNeedCommas = LEFT$(srNeedCommas, nrDecimalPosition - N + 1) + gsIntComma + MID$(srNeedCommas, nrDecimalPosition - N + 2)

      ' Move print position over one space so decimals stay aligned.
      nrDecimalPosition = nrDecimalPosition + 1

      N = N + 4
   LOOP
END SUB

' Convert user input into a number and uncertainty.
'  Input:
'     sText    a string to be converted to a number and uncertainty
'  Output:
'     dNumber  number component of sText
'     dError   uncertainty of that number
'  Function:
'     True     input had error
'     False    input understood
'
'  Called by:  GeneralLostFocus
'
FUNCTION bAnalyzeInput (BYVAL sText AS STRING, dNumber AS DOUBLE, dError AS DOUBLE) AS INTEGER
' Accepted input includes
'     1.23(2)E-2 or 1.23(2)e-2         upper or lowercase e
'     1.23(2e-2 or 1.23(2E-2           closing parenthesis is optional
'     1.23(2.3e-2                      uncertainty of .023 in number
'     1.23(23e-2                       uncertainty of .23 in number
'     1.23e-2                          uncertainty is zero
'     .0123(2 or .0123(2)              nonexponential notation
'     .0123                            nonexponential notation, no uncertainty
'     1.2(-1)                          neg. uncertainty converted to positive
'
' Rejected input
'     1.23e-2(2)                       uncertainty must come before exponent
'     1(1)                             uncertainty must be less than number

DIM nBraceLoc
DIM nE_Loc
DIM nNumDecimalLoc, nErrorDecimalLoc, nCommaLocation
DIM sMsg AS STRING, sTemp AS STRING, sExp AS STRING
DIM N, nExp

ON LOCAL ERROR GOTO OverflowError2

' Remove commas.
DO
   nCommaLocation = INSTR(1, sText, gsIntComma)
   IF nCommaLocation > 0 THEN
      sText = LEFT$(sText, nCommaLocation - 1) + MID$(sText, nCommaLocation + 1)
   ELSE
      EXIT DO
   END IF
LOOP

' Find location of (.
nBraceLoc = INSTR(1, sText, "(")
IF nBraceLoc = 0 THEN
   ' No close parenthesis to analyze.
   dNumber = VAL(sText) ' VAL returns a double number    ' 1e400 overflows
                                                         ' 1e-400 returns 0
   ' If sText is 2.3), 2.3 is returned.
   ' (2.3 and (2.3) both return zero.
   ' 2.3e) and 2.3e-) both return 2.3.
   'dError = 0 ' "Goes without saying"
ELSE
   ' Convert to uppercase so don't have to distinguish between e and E.
   sText = UCASE$(sText)
   ' Find location of E
   nE_Loc = INSTR(1, sText, "E")

   ' Get position of decimal in number.  (Used to evaluate uncertainty.)
   nNumDecimalLoc = INSTR(1, sText, ".")
   ' If no decimal, or decimal is beyond (, set to length of number string.
   IF nNumDecimalLoc > nBraceLoc - 1 OR nNumDecimalLoc = 0 THEN
      nNumDecimalLoc = nBraceLoc - 1
   END IF

   IF nE_Loc > 0 THEN
      ' Exponent in number.

      IF nE_Loc < nBraceLoc THEN
        ' Error message because E comes before (.
                sMsg = "The parenthesis part must" + CHR$(13)
         sMsg = sMsg + "come BEFORE the exponent." + CHR$(13) + CHR$(13)
         sMsg = sMsg + "Instead of:  1.2E-2(1)" + CHR$(13)
         sMsg = sMsg + "     enter:  1.2(1)E-2" + CHR$(13) + CHR$(13)
         sMsg = sMsg + "Tip:  You can type e or E" + CHR$(13)
         sMsg = sMsg + "for the exponent.  Also, " + CHR$(13)
         sMsg = sMsg + "the final ) is optional" + CHR$(13)
         MSGBOX sMsg, 0, "Unrecognized Input"
         bAnalyzeInput = TRUE
         EXIT FUNCTION
      END IF

      ' Get the number by removing the part from open parenthesis to E
      sExp = MID$(sText, nE_Loc) ' End of string from E on.
      dNumber = VAL(LEFT$(sText, nBraceLoc - 1) + sExp)  ' 1(2)e400 overflow
                                                         ' 1(2)e-400 returns 0
      ' Interpret the uncertainty
      ' Uncertainty exponent will be number exponent minus length of number
      ' beyond the decimal.
      sTemp = FORMAT$(VAL("1" + sExp), "0E+00")    ' Format the exponent
      IF MID$(sTemp, 2, 1) = "0" THEN
         ' A bug with format$:  9.8E-5 is rounded to 10E-5, not 1E-4.
         ' Also, 1E60 is 10.e59; 1E59 stays 1E59
         sTemp = "1E" + FORMAT$(VAL(MID$(sTemp, 4)) + 1)
      END IF
      nExp = VAL(MID$(sTemp, 3))          ' Get the value of the exponent.
      nExp = nExp - (nBraceLoc - 1 - nNumDecimalLoc) ' Decrease for uncertainty.
      ' Get the mantissa part of uncertainty if ) is present or absent.
      ' Absolute value means uncertainty always positive.
      dError = ABS(VAL(MID$(sText, nBraceLoc + 1, nE_Loc - nBraceLoc)))
      ' If error is zero, set to 1 in last digit.  For example,
      ' if 2.3(e-3 or 2.3(.e-3 is entered, error will be zero now.
      IF dError = 0 THEN dError = 1!
   ELSE
      ' No  exponent in number with uncertainty.
      ' Get the number part.
      dNumber = VAL(sText)

      ' Interpret the uncertainty
      ' Uncertainty exponent will be number exponent, 1, minus length of
      ' number beyond the decimal.
      nExp = 1 - (nBraceLoc - nNumDecimalLoc)  ' Decrease for uncertainty.
      ' Get the mantissa part of uncertainty if ) is present or absent.
      ' Replace negative uncertainty with positive uncertainty.
      ' Absolute value means uncertainty always positive.
      dError = ABS(VAL(MID$(sText, nBraceLoc + 1)))
      ' If error is zero, set to 1 in last digit.  For example,
      ' if 2.3( or 2.3(. is entered, error will be zero now.
      IF dError = 0 THEN dError = 1!

   END IF
   IF dError < .95 THEN
      N = DOEVENTS()
             sMsg = "The number's uncertainty is less than the" + CHR$(13)
      sMsg = sMsg + "precision of the number.  This level of " + CHR$(13)
      sMsg = sMsg + "uncertainty requires more significant " + CHR$(13)
      sMsg = sMsg + "figures in the number.  Extra zeroes will" + CHR$(13)
      sMsg = sMsg + "be added to the number." + CHR$(13) + CHR$(13)
      sMsg = sMsg + "Example:  Suppose .23(.2) is entered." + CHR$(13)
      sMsg = sMsg + "          Because the number has an " + CHR$(13)
      sMsg = sMsg + "          uncertainty of .002, the number " + CHR$(13)
      sMsg = sMsg + "          is treated as .230, not .23" + CHR$(13)
      MSGBOX sMsg, 0, "Low Precision in Number"
   END IF
   dError = VAL(STR$(dError) + "E" + STR$(nExp))
   IF dError > ABS(dNumber) THEN

      ' Trigger overflow error if exponent too large.
      ' Too large exponent gives dNumber=0, and dError 3E310, or something
      ' similar, which lands us here.
      ' 1E308 is safe, 9.9e308 overflows.
      IF ABS(VAL(MID$(sText, nE_Loc + 1))) > 290 THEN ERROR 32767


             sMsg = "This program does not allow the number's " + CHR$(13)
      sMsg = sMsg + "uncertainty,  " + STR$(dError) + CHR$(13)
      sMsg = sMsg + "to be larger than the magnitude" + CHR$(13)
      sMsg = sMsg + "of the number," + STR$(dNumber) + CHR$(13)
      sMsg = sMsg + "The uncertainty will therefore be" + CHR$(13)
      sMsg = sMsg + "given the same value as the number." + CHR$(13) + CHR$(13)
      sMsg = sMsg + "NOTE:  To enter a number such as 1(2)C," + CHR$(13)
      sMsg = sMsg + "       switch to Kelvin and enter 274(2)."
      MSGBOX sMsg, 0, "Large Uncertainty"
      dError = ABS(dNumber)
   END IF
END IF
EXIT FUNCTION

OverflowError2:      ' 2 Because use OverflowError in "Calculate".
   IF ERR = 6 THEN   ' overflow
      sMsg = "The number you entered is too large to" + CHR$(10) + "handle.  The exponent is too large."
   ELSEIF ERR = 32767 THEN
      ' User-defined error.
      sMsg = "The number you entered is too small to" + CHR$(10) + "handle.  The exponent is too small."
      dError = 0
   ELSE
      sMsg = "Error number" + STR$(ERR) + CHR$(10) + ERROR$ + CHR$(10)
      sMsg = sMsg + "If you can reproduce this, the author would like to know about it."
   END IF
   MSGBOX sMsg, 0, "Number Entry"
   bAnalyzeInput = TRUE
   EXIT FUNCTION

END FUNCTION

' Converts a number from one unit to another.
'
'      Input:     dvInNum     Incomming number to convert
'                 dvInUnc     Incomming uncertainty to convert
'                 nvUnitsIn   Units of incomming numbers
'                 nvUnitsOut  Units numbers are converted to.
'     Output:     dOutNum     Converted number
'                 dOutUnc     Converted uncertainty
'                 nConvert    Index of string giving the conversion formula.
'
'  Return value:  TRUE on error, FALSE otherwise.
'
'               Called by:  bMakeConnection, CalculateFormula
'                           ChangeTempUnit, ChangeVolumeUnit, ChangePressureUnit
'  Module level variables:  None
'        Global variables:  None
FUNCTION bUnitConversion (BYVAL dvInNum AS DOUBLE, BYVAL dvInUnc AS DOUBLE, BYVAL nvUnitsIn, BYVAL nvUnitsOut, dOutNum AS DOUBLE, dOutUnc AS DOUBLE, nConvert) AS INTEGER

DIM dConversionFactor AS DOUBLE
ON LOCAL ERROR GOTO ConversionError

IF nvUnitsIn = nvUnitsOut OR nvUnitsOut = 0 OR nvUnitsIn = 0 THEN
   dOutNum = dvInNum
   dOutUnc = dvInUnc
ELSE
   SELECT CASE nvUnitsIn
      CASE ML
         ' Units out are LITER
         dOutNum = dvInNum * .001#
         dOutUnc = dvInUnc * .001#
         nConvert = 1
      CASE LITER
         ' Units out are ML
         dOutNum = dvInNum * 1000#
         dOutUnc = dvInUnc * 1000#
         nConvert = 2
      CASE KELVIN
         IF nvUnitsOut = CELSIUS THEN
            ' Kelvin to Celsius
            dOutNum = dvInNum - 273.15#
            dOutUnc = dvInUnc
            nConvert = 3
         ELSE  ' Kelvin to Fahrenheit
            dOutNum = (dvInNum - 273.15#) * 1.8# + 32#
            dOutUnc = dvInUnc * 1.8#
            nConvert = 4
         END IF
      CASE CELSIUS
         IF nvUnitsOut = KELVIN THEN
            ' Celsius to Kelvin
            dOutNum = dvInNum + 273.15#
            dOutUnc = dvInUnc
            nConvert = 5
         ELSE  ' Celsius to Fahrenheit
            dOutNum = dvInNum * 1.8# + 32#
            dOutUnc = dvInUnc * 1.8#
            nConvert = 6
         END IF
      CASE FAHRENHEIT
         IF nvUnitsOut = CELSIUS THEN
            ' Fahrenheit to Celsius
            dOutNum = (dvInNum - 32#) / 1.8#
            dOutUnc = dvInUnc / 1.8#
            nConvert = 7
         ELSE  ' Fahrenheit to Kelvin
            dOutNum = (dvInNum - 32#) / 1.8# + 273.15#
            dOutUnc = dvInUnc / 1.8#
            nConvert = 8
         END IF
      CASE PA  ' pascal
         IF nvUnitsOut = KPA THEN      ' Pascal to kPa
            dConversionFactor = .001#
            nConvert = 9
         ELSEIF nvUnitsOut = BAR THEN  ' Pascal to bar
            dConversionFactor = .00001#
            nConvert = 10
         ELSEIF nvUnitsOut = ATM THEN  ' Pascal to atm
            dConversionFactor = 1 / 101325#
            nConvert = 11
         ELSEIF nvUnitsOut = MMHG OR nvUnitsOut = TORR THEN  ' Pascal to mm Hg
            dConversionFactor = 760 / 101325#
            nConvert = 12
         ELSEIF nvUnitsOut = PSI THEN  ' Pascal to psi
            dConversionFactor = 14.69594# / 101325
            nConvert = 13
         END IF
            dOutNum = dvInNum * dConversionFactor
            dOutUnc = dvInUnc * dConversionFactor
      CASE KPA  ' kPa
         IF nvUnitsOut = PA THEN      ' kPa to pascal
            dConversionFactor = 1000#
            nConvert = 14
         ELSEIF nvUnitsOut = BAR THEN  ' kPa to bar
            dConversionFactor = .01#
            nConvert = 15
         ELSEIF nvUnitsOut = ATM THEN  ' kPa to atm
            dConversionFactor = 1 / 101.325#
            nConvert = 16
         ELSEIF nvUnitsOut = MMHG OR nvUnitsOut = TORR THEN  ' kPa to mm Hg
            dConversionFactor = 760 / 101.325#
            nConvert = 17
         ELSEIF nvUnitsOut = PSI THEN  ' kPa to psi
            dConversionFactor = 14.69594# / 101.325#
            nConvert = 18
         END IF
            dOutNum = dvInNum * dConversionFactor
            dOutUnc = dvInUnc * dConversionFactor
      CASE BAR ' bar
         IF nvUnitsOut = PA THEN      ' bar to pascal
            dConversionFactor = 100000#
            nConvert = 19
         ELSEIF nvUnitsOut = KPA THEN  ' bar to kPa
            dConversionFactor = 100#
            nConvert = 20
         ELSEIF nvUnitsOut = ATM THEN  ' bar to atm
            dConversionFactor = 1 / 1.01325#
            nConvert = 21
         ELSEIF nvUnitsOut = MMHG OR nvUnitsOut = TORR THEN  ' bar to mm Hg
            dConversionFactor = 760 / 1.01325#
            nConvert = 22
         ELSEIF nvUnitsOut = PSI THEN  ' bar to psi
            dConversionFactor = 14.69594# / 1.01325#
            nConvert = 23
         END IF
            dOutNum = dvInNum * dConversionFactor
            dOutUnc = dvInUnc * dConversionFactor
      CASE ATM ' atm
         IF nvUnitsOut = PA THEN      ' atm to pascal
            dConversionFactor = 101325#
            nConvert = 24
         ELSEIF nvUnitsOut = KPA THEN  ' atm to kPa
            dConversionFactor = 101.325#
            nConvert = 25
         ELSEIF nvUnitsOut = BAR THEN  ' atm to bar
            dConversionFactor = 1.01325#
            nConvert = 26
         ELSEIF nvUnitsOut = MMHG OR nvUnitsOut = TORR THEN  ' atm to mm Hg
            dConversionFactor = 760#
            nConvert = 27
         ELSEIF nvUnitsOut = PSI THEN  ' atm to psi
            dConversionFactor = 14.69594#
            nConvert = 28
         END IF
            dOutNum = dvInNum * dConversionFactor
            dOutUnc = dvInUnc * dConversionFactor
      CASE MMHG, TORR   ' torr or mmHg
         IF nvUnitsOut = PA THEN      ' torr or mmHg to pascal
            dConversionFactor = 101325# / 760
            nConvert = 29
         ELSEIF nvUnitsOut = KPA THEN  ' torr or mmHg to kPa
            dConversionFactor = 101.325# / 760
            nConvert = 30
         ELSEIF nvUnitsOut = ATM THEN  ' torr or mmHg to atm
            dConversionFactor = 1 / 760#
            nConvert = 31
         ELSEIF nvUnitsOut = BAR THEN   ' torr or mmHg to bar
            dConversionFactor = 1.01325# / 760
            nConvert = 32
         ELSEIF nvUnitsOut = PSI THEN  ' torr or mmHg to psi
            dConversionFactor = 14.69594# / 760
            nConvert = 33
         ELSE  ' torr or mm Hg to mm Hg or torr.
            dConversionFactor = 1
            nConvert = 39
         END IF
            dOutNum = dvInNum * dConversionFactor
            dOutUnc = dvInUnc * dConversionFactor
      CASE PSI
         IF nvUnitsOut = PA THEN      ' psi to pascal
            dConversionFactor = 101325 / 14.69594#
            nConvert = 34
         ELSEIF nvUnitsOut = KPA THEN  ' psi to kPa
            dConversionFactor = 101.325# / 14.69594#
            nConvert = 35
         ELSEIF nvUnitsOut = ATM THEN  ' psi to atm
            dConversionFactor = 1 / 14.69594#
            nConvert = 36
         ELSEIF nvUnitsOut = MMHG OR nvUnitsOut = TORR THEN  ' psi to mm Hg
            dConversionFactor = 760 / 14.69594#
            nConvert = 37
         ELSEIF nvUnitsOut = BAR THEN  ' psi to bar
            dConversionFactor = 1.01325# / 14.69594#
            nConvert = 38
         END IF
            dOutNum = dvInNum * dConversionFactor
            dOutUnc = dvInUnc * dConversionFactor
   END SELECT
END IF
bUnitConversion = FALSE

EXIT FUNCTION

ConversionError:
   ' Caused by overflow.
   BEEP     ' No message box needed.  To cause this error, you have to try.
   dOutNum = 0
   dOutUnc = 0
   bUnitConversion = TRUE
   EXIT FUNCTION

END FUNCTION

' Handle calculations for Reaction form.
'   Input:  nvIndex     nvIndex that changed
'           Src         Form name (easier than typing frmReaction all the time)
'   Called by:  CalculateFormula, frmReaction
SUB CalcReaction (BYVAL nvIndex, Src AS FORM)
DIM nInputType, nReactionGroup, N1, N2, N3, N
DIM srMsg AS STRING, nrPrintPosition
DIM nUsedReactants, nTemp
DIM dLimReag AS DOUBLE
DIM dLimReagUncert AS DOUBLE
DIM nLimReagNum
DIM bShowGrams, dTemp AS DOUBLE, dTemp2 AS DOUBLE
DIM bFlag(3)

ON LOCAL ERROR GOTO ReactionError

' 39 = m.m.
' 40 = g used
' 41 = mol used
' 42 = coefficient
' 43 = theor. mol
' 44 = theor. g
' 75, 76, and 77 are for mole/coefficient.

'************************* Calculate moles used ****************************
nReactionGroup = (nvIndex - 39) \ 6
nInputType = (nvIndex - 38) MOD 6 ' 1=M.M.; 2=g; 3=mol
IF nInputType < 4 THEN
   IF Src.cmdMolUse(nReactionGroup).Tag = "" THEN
      ' Only calculate moles if molar mass or grams changed.
      IF nInputType = 1 THEN
         ' Input is molar mass; calculate moles.
         N1 = nvIndex + 2: N2 = nvIndex + 1: N3 = nvIndex
      ELSE
         ' Input is grams; calculate moles. (Input can't be moles, type = 0.)
         N1 = nvIndex + 1: N2 = nvIndex: N3 = nvIndex - 1
      END IF
      IF gdNumber(N3) > 0 THEN
         gdNumber(N1) = gdNumber(N2) / gdNumber(N3)
         gdUncertainty(N1) = SQR(gdUncertainty(N2) ^ 2 + (gdNumber(N1) * gdUncertainty(N3)) ^ 2) / gdNumber(N3)
      ELSE
         gdNumber(N1) = 0
         gdUncertainty(N1) = 0
      END IF
   ELSE
      ' Only calculate grams if molar mass, or moles changed.
      IF nInputType = 1 THEN
         ' Input is molar mass; calculate grams.
         N1 = nvIndex + 1: N2 = nvIndex + 2: N3 = nvIndex
      ELSEIF nInputType = 3 THEN
         ' Input is moles; calculate grams.
         N1 = nvIndex - 1: N2 = nvIndex: N3 = nvIndex - 2
      END IF
      gdNumber(N1) = gdNumber(N2) * gdNumber(N3)
      gdUncertainty(N1) = SQR((gdNumber(N3) * gdUncertainty(N2)) ^ 2 + (gdNumber(N2) * gdUncertainty(N3)) ^ 2)
   END IF
   NumberFormat gdNumber(N1), gdUncertainty(N1), 51, srMsg, nrPrintPosition
   IF srMsg = "0" THEN srMsg = ""
   Src.txtVariable(N1).Text = srMsg
END IF
'************************ Calculate Reactant Things ************************
' Determine how many reactants are non-zero.
FOR N = 0 TO 2
   nTemp = 41 + N * 6                                       ' moles used
   IF gdNumber(nTemp) > 0 AND gdNumber(nTemp + 1) > 0 THEN
      ' 75, 76, and 77 are for mole/coefficient.
      gdNumber(75 + N) = gdNumber(nTemp) / gdNumber(nTemp + 1)
      gdUncertainty(75 + N) = gdUncertainty(nTemp) / gdNumber(nTemp + 1)
      IF gdNumber(75 + N) > 0 THEN
         NumberFormat gdNumber(75 + N), gdUncertainty(75 + N), 51, srMsg, nrPrintPosition
         IF srMsg = "0" THEN srMsg = ""
      ELSE
         srMsg = ""
      END IF
      Src.txtLRMol(N).Text = srMsg
      IF gdNumber(75 + N) < dLimReag OR dLimReag = 0 THEN
         dLimReag = gdNumber(75 + N)
         dLimReagUncert = gdUncertainty(75 + N)
      END IF
      nUsedReactants = nUsedReactants + 1
   ELSE
      gdNumber(75 + N) = 0
      gdUncertainty(75 + N) = 0
      srMsg = ""
   END IF
   Src.txtLRMol(N).Text = srMsg
NEXT N
                              ' Limiting reagent flags
FOR N = 0 TO 2
   dTemp = gdNumber(75 + N) - gdUncertainty(75 + N) - dLimReag + dLimReagUncert
   dTemp2 = dLimReag / 100000000000000#  ' Needed because of "nature of floating point".
   IF dTemp <= dTemp2 AND gdNumber(75 + N) > 0 THEN
      bFlag(N) = TRUE
      nLimReagNum = nLimReagNum + 1
   END IF
NEXT N
IF nLimReagNum = nUsedReactants THEN
   bFlag(0) = FALSE
   bFlag(1) = FALSE
   bFlag(2) = FALSE
END IF
Src.lblLRFlag(0).Visible = bFlag(0)
Src.lblLRFlag(1).Visible = bFlag(1)
Src.lblLRFlag(2).Visible = bFlag(2)

'IF nUsedReactants > 1 THEN            ' Moles needed
   FOR N = 0 TO 2
      ' Moles needed.
      nTemp = 43 + N * 6
      IF Src.lblLRFlag(N).Visible THEN
         gdNumber(nTemp) = gdNumber(nTemp - 2)
         gdUncertainty(nTemp) = gdUncertainty(nTemp - 2)
      ELSEIF gdNumber(nTemp - 1) > 0 THEN
         gdNumber(nTemp) = dLimReag * gdNumber(nTemp - 1)
         gdUncertainty(nTemp) = dLimReagUncert * gdNumber(nTemp - 1)
      ELSE
         gdNumber(nTemp) = 0
         gdUncertainty(nTemp) = 0
      END IF
      NumberFormat gdNumber(nTemp), gdUncertainty(nTemp), 51, srMsg, nrPrintPosition
      IF srMsg = "0" THEN srMsg = ""
      Src.txtVariable(nTemp).Text = srMsg
      IF gnConnected(nTemp) THEN
         gdNumber(gnConnected(nTemp)) = gdNumber(nTemp)
         gdUncertainty(gnConnected(nTemp)) = gdUncertainty(nTemp)
         CalculateFormula gnConnected(nTemp)
      END IF
   NEXT N
'ELSE
'   FOR N = 0 TO 2
'      nTemp = 43 + N * 6
'      gdNumber(nTemp) = 0
'      gdUncertainty(nTemp) = 0
'      Src.txtVariable(nTemp).Text = ""
'      IF gnConnected(nTemp) THEN
'         gdNumber(gnConnected(nTemp)) = gdNumber(nTemp)
'         gdUncertainty(gnConnected(nTemp)) = gdUncertainty(nTemp)
'         CalculateFormula gnConnected(nTemp)
'      END IF
'   NEXT N
'END IF
                                   ' Grams needed
FOR N = 0 TO 2
   nTemp = 44 + N * 6
   IF gdNumber(nTemp - 1) > 0 AND gdNumber(nTemp - 5) > 0 THEN
      gdNumber(nTemp) = gdNumber(nTemp - 1) * gdNumber(nTemp - 5)
      gdUncertainty(nTemp) = SQR((gdNumber(nTemp - 1) * gdUncertainty(nTemp - 5)) ^ 2 + (gdNumber(nTemp - 5) * gdUncertainty(nTemp - 1)) ^ 2)
      NumberFormat gdNumber(nTemp), gdUncertainty(nTemp), 51, srMsg, nrPrintPosition
      IF srMsg = "0" THEN srMsg = ""
   ELSE
      gdNumber(nTemp) = 0
      gdUncertainty(nTemp) = 0
      srMsg = ""
   END IF
   IF gnConnected(nTemp) THEN
      gdNumber(gnConnected(nTemp)) = gdNumber(nTemp)
      gdUncertainty(gnConnected(nTemp)) = gdUncertainty(nTemp)
      CalculateFormula gnConnected(nTemp)
   END IF
   Src.txtVariable(nTemp).Text = srMsg
NEXT N

FOR N = 3 TO 5
   nTemp = 43 + N * 6
   IF gdNumber(nTemp - 1) > 0 THEN        ' Theoretical yield, moles
      gdNumber(nTemp) = dLimReag * gdNumber(nTemp - 1)
      gdUncertainty(nTemp) = dLimReagUncert * gdNumber(nTemp - 1)
      NumberFormat gdNumber(nTemp), gdUncertainty(nTemp), 51, srMsg, nrPrintPosition
      IF srMsg = "0" THEN srMsg = ""
      Src.txtVariable(nTemp).Text = srMsg
                                          ' Theoretical yield, g
      IF gdNumber(nTemp - 4) > 0 THEN
         gdNumber(nTemp + 1) = gdNumber(nTemp) * gdNumber(nTemp - 4)
         gdUncertainty(nTemp + 1) = SQR((gdNumber(nTemp) * gdUncertainty(nTemp - 4)) ^ 2 + (gdNumber(nTemp - 4) * gdUncertainty(nTemp)) ^ 2)
         NumberFormat gdNumber(nTemp + 1), gdUncertainty(nTemp + 1), 51, srMsg, nrPrintPosition
         IF srMsg = "0" THEN srMsg = ""
      ELSE
         srMsg = ""
         gdNumber(nTemp + 1) = 0
         gdUncertainty(nTemp + 1) = 0
      END IF
      Src.txtVariable(nTemp + 1).Text = srMsg
                                          ' Percent yield
      IF gdNumber(nTemp - 2) > 0 THEN
         ' Percent yield is stored in elements 78, 79, 80.
         gdNumber(75 + N) = gdNumber(nTemp - 2) / gdNumber(nTemp) * 100#
         gdUncertainty(75 + N) = SQR(gdUncertainty(nTemp - 2) ^ 2 + (gdNumber(75 + N) * gdUncertainty(nTemp)) ^ 2) / gdNumber(nTemp) * 100#
         NumberFormat gdNumber(75 + N), gdUncertainty(75 + N), 51, srMsg, nrPrintPosition
         IF srMsg = "0" THEN srMsg = ""
      ELSE
         gdNumber(75 + N) = 0
         gdUncertainty(75 + N) = 0
         srMsg = ""
      END IF
      Src.lblPctYield(N - 3).Text = srMsg
   ELSE
      gdNumber(nTemp) = 0
      gdUncertainty(nTemp) = 0
      Src.txtVariable(nTemp).Text = ""
      gdNumber(nTemp + 1) = 0
      gdUncertainty(nTemp + 1) = 0
      Src.txtVariable(nTemp + 1).Text = ""
      gdNumber(75 + N) = 0
      gdUncertainty(75 + N) = 0
      Src.lblPctYield(N - 3).Text = ""
   END IF
NEXT N

EXIT SUB

ReactionError:
   ' Find the dirty villain.
   BEEP
   FOR N = 0 TO 5
      FOR N1 = 0 TO 2
         SELECT CASE N1
            CASE 0
               nTemp = 42 + 6 * N   ' Coefficients.
            CASE 1
               nTemp = nTemp - 3    ' Molar mass.
            CASE 2
               nTemp = nTemp + 2    ' Moles.
         END SELECT
         IF gdNumber(nTemp) > 1D+75 OR (gdNumber(nTemp) > 0 AND gdNumber(nTemp) < 1D+75) THEN
            IF gnConnected(nTemp) THEN Disconnect nTemp
            gdNumber(nTemp) = 0
            gdUncertainty(nTemp) = 0
            Src.txtVariable(nTemp).Text = ""
         END IF
      NEXT N1
   NEXT N
   FOR N = 0 TO 5
      nTemp = 40 + 6 * N   ' Grams.
      IF Src.cmdMolUse(N).Tag = "" THEN
         IF gdNumber(nTemp) > 1D+75 OR (gdNumber(nTemp) > 0 AND gdNumber(nTemp) < 1D+75) THEN
            IF gnConnected(nTemp) THEN Disconnect nTemp
            gdNumber(nTemp) = 0
            gdUncertainty(nTemp) = 0
            Src.txtVariable(nTemp).Text = ""
            CalcReaction nTemp, frmReaction
         END IF
      END IF
   NEXT N
   CalcReaction 41, frmReaction  ' Makes sure that CalcReaction is called at least once.
   EXIT SUB
END SUB

' Calculate moles and uncertainty.  Handle overflow errors.
'
'      Input:  nvIndexM       Molarity index
'              nvIndexV       Volume index
'              dvConvert      Volume conversion factor
'     Output:  drMoles        Moles
'              drMolesUnc     Mole uncertainty
'
'         Called by:  CalculateFormula
'  Global variables:  gdNumber, gdUncertainty
'
SUB CalculateMoles (BYVAL nvIndexM, BYVAL nvIndexV, BYVAL dvConvert AS DOUBLE, drMoles AS DOUBLE, drMolesUnc AS DOUBLE)
ON LOCAL ERROR GOTO MolesOverflow
drMoles = gdNumber(nvIndexM) * gdNumber(nvIndexV) * dvConvert
drMolesUnc = SQR((gdNumber(nvIndexV) * gdUncertainty(nvIndexM)) ^ 2 + (gdNumber(nvIndexM) * gdUncertainty(nvIndexV)) ^ 2) * dvConvert
EXIT SUB

MolesOverflow:
   ' Handle 2E308 moles or uncertainty.
   drMoles = 0
   drMolesUnc = 0
   BEEP
   EXIT SUB
END SUB

SUB ChooseReactant ()
DIM N
   ' Ask User which group to connect to.
   gnUserChoice = -3
   FOR N = 1 TO 3
      frmChoose.optVariable(N).Visible = TRUE
      frmChoose.optVariable(N).Caption = "reactant" + STR$(N) + " " + frmReaction.txtCompound(N - 1).Text
   NEXT N
   FOR N = 4 TO 6
      frmChoose.optVariable(N).Visible = TRUE
      frmChoose.optVariable(N).Caption = "product" + STR$(N - 3) + " " + frmReaction.txtCompound(N - 1).Text
   NEXT N
   frmChoose.Caption = "Reactions"
   frmChoose.SHOW 1
END SUB

' Disconnects a variable with a formula showing.  Replaces the
' formula with the variable's present value.
'  Input:   nvIndex    Index of the txtVariable that gets the text.
'
'  Called by:  EvaluateFormula3, DragDrop3, mnuMoveDisconnect, nOptionButton,
'              and others.
'  Module level variables:
'        Global variables:  gdNumber, gdUncertainty, gnConnected
'
SUB Disconnect (BYVAL nvIndex)
DIM N, srMsg AS STRING
' Can't just copy text of connected control over to this control because
' they may have different units (mL and L, for example).

NumberFormat gdNumber(nvIndex), gdUncertainty(nvIndex), 0, srMsg, N
N = gnConnected(nvIndex)
gnConnected(N) = 0
gnConnected(nvIndex) = 0

SELECT CASE nvIndex
   CASE IS < 4
      frmGramMole.txtVariable(nvIndex).Text = srMsg
      frmGramMole.lblVariable(nvIndex).BackColor = WHITE
   CASE IS < 7
'formGone      frmDensity.txtVariable(nvIndex).Text = srMsg
'formGone      frmDensity.lblVariable(nvIndex).BackColor = WHITE
   CASE IS < 10
'formgone      frmMolarity.txtVariable(nvIndex).Text = srMsg
'formgone      frmMolarity.lblVariable(nvIndex).BackColor = WHITE
   CASE IS < 13
      frmPercent.txtVariable(nvIndex).Text = srMsg
      frmPercent.lblVariable(nvIndex).BackColor = WHITE
      IF gnConnected(10) = 0 AND gnConnected(11) = 0 THEN
         Initialize_mnUnits 4
      END IF
   CASE IS < 18
'formGone      frmDilution.txtVariable(nvIndex).Text = srMsg
'formGone      frmDilution.lblVariable(nvIndex).BackColor = WHITE
   CASE IS < 26
      frmTitration.txtVariable(nvIndex).Text = srMsg
      frmTitration.lblVariable(nvIndex).BackColor = WHITE
   CASE IS < 34
      frmGas.txtVariable(nvIndex).Text = srMsg
      frmGas.lblVariable(nvIndex).BackColor = WHITE
   CASE IS < 39
      frmIdealGas.txtVariable(nvIndex).Text = srMsg
      frmIdealGas.lblVariable(nvIndex).BackColor = WHITE
   CASE IS < 81
      frmReaction.txtVariable(nvIndex).Text = srMsg
      frmReaction.lblVariable(nvIndex).BackColor = WHITE
END SELECT

' Make escape key return present string, not the previous formula.
' Without the following, escape displays, e.g., mole=g/M.M.  Also,
' when leave this txtVariable, it looks like the text has changed, so
' new text is processed, which introduces rounding error.
gsOriginalText(nGroupOfIndex(nvIndex)) = srMsg

SELECT CASE N
   CASE IS < 4
      frmGramMole.lblVariable(N).BackColor = WHITE
   CASE IS < 7
'formGone      frmDensity.lblVariable(N).BackColor = WHITE
   CASE IS < 10
'formgone      frmMolarity.lblVariable(N).BackColor = WHITE
   CASE IS < 13
      frmPercent.lblVariable(N).BackColor = WHITE
      IF gnConnected(10) = 0 AND gnConnected(11) = 0 THEN
         Initialize_mnUnits 4
      END IF
   CASE IS < 18
'formGone      frmDilution.lblVariable(N).BackColor = WHITE
   CASE IS < 26
      frmTitration.lblVariable(N).BackColor = WHITE
   CASE IS < 34
      frmGas.lblVariable(N).BackColor = WHITE
   CASE IS < 39
      frmIdealGas.lblVariable(N).BackColor = WHITE
   CASE IS < 81
      frmReaction.lblVariable(N).BackColor = WHITE
END SELECT

END SUB

' Determine if the formula is arranged correctly.
'  Input:  nvGroup   Group to which form belongs.
'          Src       Form to evaluate.
'
'  Called by:  DragDrop3
'  Module level variables:  mGroupDefine
'        Global variables:  gOccupantIndex, gbBadFormula, gbDontStopHere
'
SUB EvaluateFormula3 (BYVAL nvGroup, Src AS FORM)
DIM N, bUnrecognizedFormula, N1, N2, N3
DIM sTemp AS STRING, nResult, nPositionN1, nPositionN2, nPositionN3
STATIC snTries
   N1 = gGroupDefine(nvGroup).LowestIndex
   N2 = N1 + 1
   N3 = N1 + 2
   ' Get the present positions of the variables.
   nPositionN1 = VAL(Src.lblVariable(N1).Tag)
   nPositionN2 = VAL(Src.lblVariable(N2).Tag)
   nPositionN3 = VAL(Src.lblVariable(N3).Tag)
   IF gOccupantIndex(nvGroup, 0) = N1 THEN
      ' Molar mass on left side.
      IF nPositionN3 > 3 AND nPositionN2 < 4 THEN
         ' Moles on bottom      grams on top.
         ' Good formula.
         ' Run optDependent_Click (Index) in form module.  This way of doing
         ' this has the advantage that the txtVariable turns yellow before
         ' the move below, so user gets immediate positive feedback.
         ' If move before turn yellow, suggests user's actions weren't quite
         ' right.
         Src.lblDragDrop.Tag = STR$(N1)
         Src.lblDragDrop.Caption = "OB"  ' Calls optDependent_Click
         SwitchGeneral 1, 2, 25, nvGroup, Src
      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = N2 THEN
      ' Grams on left side.
      IF nPositionN1 < 4 AND nPositionN3 < 4 THEN
         '   Molar mass on top     moles on top.
         ' Good formula.
         Src.lblDragDrop.Tag = STR$(N2)
         Src.lblDragDrop.Caption = "OB"
         Src.lblX.Visible = TRUE
         Src.lblDivisionLine.Visible = FALSE
      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = N3 THEN
      ' Moles on left side.
      IF nPositionN1 > 3 AND nPositionN2 < 4 THEN
         ' Molar mass on bottom   grams on top.
         ' Good formula.
         Src.lblDragDrop.Tag = STR$(N3)
         Src.lblDragDrop.Caption = "OB"
         SwitchGeneral 1, 2, 25, nvGroup, Src
      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = -1 THEN
      bUnrecognizedFormula = TRUE
   END IF

   IF bUnrecognizedFormula THEN

      gbBadFormula(nvGroup) = TRUE

   ' A word about removing connections while rearranging formulas.  Any
   ' control that is dropped in
   ' a new position is disconnected, as is the control originally in
   ' the new position.  If the formula has been changed to a valid formula
   ' by one move (mole = g /M.W. ---> M.W. = g / mole), unmoved controls
   ' remain connected (grams, in the above example).
   '
   ' In recognized formulas, optDependent_Click is responsible for
   ' disconnecting the old dependent variable.

      snTries = snTries + 1
      FOR N = gGroupDefine(nvGroup).LowestIndex TO gGroupDefine(nvGroup).HighestIndex
         Src.txtVariable(N).BackColor = CYAN
         gbDontStopHere(N) = TRUE
      NEXT N
      Src.lblX.Visible = FALSE ' Two moves are required to go from grams to another recognized formula.
      Src.lblDivisionLine.Visible = TRUE
      IF nvGroup = 1 THEN
         frmGramMole.cmdUpdate.Enabled = FALSE  ' Set TRUE in call to optDependent_Click.
      END IF
      Src.cmdSize.Enabled = FALSE

   ELSE
      gbBadFormula(nvGroup) = FALSE
      Src.cmdSize.Enabled = TRUE
      CalculateFormula gOccupantIndex(nvGroup, 0)
      IF snTries > 4 THEN
         ' Play "Clair de lune" when get the formula right.
         PLAY "MBT160MLO2L8A-L2O3A-L4F.O2L8AO3E-FL2E-"
         PLAY "MLO2L8A-O3D-E-D-.L4F.L8D-.O2G-O3CD-L2C."
      END IF
      snTries = 0
   END IF

END SUB

' Determine if the formula in frmPercent is arranged correctly.
'  Input:  nvGroup        Group to which form belongs.
'          SourceForm     Form to evaluate.
'
'  Called by:  DragDrop4
'  Module level variables:  mGroupDefine
'        Global variables:  gOccupantIndex, gbBadFormula, gbDontStopHere
'
SUB EvaluateFormula4 (BYVAL nvGroup, SourceForm AS FORM)
DIM N, bUnrecognizedFormula, N1, N2, N3, N4
DIM sTemp AS STRING, nResult
DIM nPositionN1, nPositionN2, nPositionN3, nPositionN4
STATIC snTries
   IF nvGroup = 7 THEN
      SELECT CASE VAL(frmGas.cmdVariables.Tag)
         CASE 1
               ' T1-4     P1-1     P2*0     T2-2
            N1 = 32: N2 = 27: N3 = 28: N4 = 31
         CASE 2
               ' V2-4     P1-1     P2*0     V1-2
            N1 = 26: N2 = 27: N3 = 28: N4 = 29
         CASE 3
               ' n1-4     P1-1     P2*0     n2-2
            N1 = 30: N2 = 27: N3 = 28: N4 = 33
         CASE 4
               ' T1-4     V1-1     V2*0     T2-2
            N1 = 32: N2 = 29: N3 = 26: N4 = 31
         CASE 5
               ' T1-4     n2-1     n1*0     T2-2
            N1 = 32: N2 = 33: N3 = 30: N4 = 31
         CASE 6
               ' n1-4     V1-1     V2*0     n2-2
            N1 = 30: N2 = 29: N3 = 26: N4 = 33
      END SELECT
   ELSE
      N1 = gGroupDefine(nvGroup).LowestIndex    ' whole
      N2 = N1 + 1                                  ' part
      N3 = N1 + 2                                  ' percent
      N4 = N1 + 3                                  ' 100
   END IF
   ' Get the present positions of the variables.
   nPositionN1 = VAL(SourceForm.lblVariable(N1).Tag)
   nPositionN2 = VAL(SourceForm.lblVariable(N2).Tag)
   nPositionN3 = VAL(SourceForm.lblVariable(N3).Tag)
   nPositionN4 = VAL(SourceForm.lblVariable(N4).Tag)
   IF gOccupantIndex(nvGroup, 0) = N1 THEN
      '
      ' Whole =part * 100 / percent
      ' Vf = Vi * Mi / Mf
      IF nPositionN2 < 4 AND nPositionN4 < 4 AND nPositionN3 > 3 THEN
         '   part on top         100 on top     percent on bottom
         ' Good formula.
         SourceForm.lblDragDrop.Tag = STR$(N1)
         SourceForm.lblDragDrop.Caption = "OB"  ' Runs optDependent_Click
      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = N2 THEN
      ' Part on left side.
      ' Part = whole * percent / 100
      ' Mi = Mf * Vf / Vi
      IF nPositionN1 < 4 AND nPositionN3 < 4 AND nPositionN4 > 3 THEN
         '   Whole on top     percent on top.     100 on bottom.
         ' Good formula.
         SourceForm.lblDragDrop.Tag = STR$(N2)
         SourceForm.lblDragDrop.Caption = "OB"  ' "Click" option button.
      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = N3 THEN
   ' Percent on left side.
   ' Mf = Mi * Vi / Vf
      IF nPositionN1 > 3 AND nPositionN2 < 4 AND nPositionN4 < 4 THEN
         ' Whole on bottom      part on top         100 on top
         ' Good formula.
         SourceForm.lblDragDrop.Tag = STR$(N3)
         SourceForm.lblDragDrop.Caption = "OB"  ' "Click" option button.
      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = N4 THEN
   ' Vi.  This case never occurs for percent.
   ' Vi = Vf * Mf / Mi
   ' N4   N1   N3   N2
      IF nPositionN1 < 4 AND nPositionN3 < 4 AND nPositionN2 > 3 THEN
         '     Vf on top         Mf on top         Mi on bottom
         ' Good formula.
         SourceForm.lblDragDrop.Tag = STR$(N4)
         SourceForm.lblDragDrop.Caption = "OB"  ' "Click" option button.
      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = -1 THEN
      bUnrecognizedFormula = TRUE
   END IF
   IF bUnrecognizedFormula THEN

      gbBadFormula(nvGroup) = TRUE
      snTries = snTries + 1
      FOR N = gGroupDefine(nvGroup).LowestIndex TO gGroupDefine(nvGroup).HighestIndex
         SourceForm.txtVariable(N).BackColor = CYAN
         gbDontStopHere(N) = TRUE
      NEXT N
      SourceForm.cmdSize.Enabled = FALSE
      IF nvGroup = 7 THEN
         SourceForm.cmdVariables.Enabled = FALSE
      END IF
   ELSE
      gbBadFormula(nvGroup) = FALSE
      SourceForm.cmdSize.Enabled = TRUE
      IF nvGroup = 7 THEN
         SourceForm.cmdVariables.Enabled = TRUE
      END IF
      CalculateFormula gOccupantIndex(nvGroup, 0)
      IF snTries > 4 THEN
         ' Play "Clair de lune" when get the formula right.
         PLAY "MBT160MLO2L8A-L2O3A-L4F.O2L8AO3E-FL2E-"
         PLAY "MLO2L8A-O3D-E-D-.L4F.L8D-.O2G-O3CD-L2C."
      END IF
      snTries = 0
   END IF

END SUB

' Determine if the formula in frmPercent is arranged correctly.
'  Input:  nvGroup        Group to which form belongs.
'          Src     Form to evaluate.
'
'  Called by:  DragDrop4
'  Module level variables:  mGroupDefine
'        Global variables:  gOccupantIndex, gbBadFormula, gbDontStopHere
'
SUB EvaluateFormula6 (BYVAL nvGroup, Src AS FORM)
DIM N, bUnrecognizedFormula, N1, N2, N3, N4, N5, N6
DIM sTemp AS STRING, nResult, nTemp, nTmpIndex
DIM nPositionN1, nPositionN2, nPositionN3, nPositionN4
DIM nPositionN5, nPositionN6, nAdjust6
STATIC snTries
   N1 = gGroupDefine(nvGroup).LowestIndex    ' V2
   N2 = N1 + 1                                  ' M1
   N3 = N1 + 2                                  ' M2
   N4 = N1 + 3                                  ' V1
   N5 = N1 + 4                                  ' 2
   N6 = N1 + 5                                  ' 1

   ' Get the present positions of the variables.
   nPositionN1 = VAL(Src.lblVariable(N1).Tag)  ' V2
   nPositionN2 = VAL(Src.lblVariable(N2).Tag)  ' M1
   nPositionN3 = VAL(Src.lblVariable(N3).Tag)  ' M2
   nPositionN4 = VAL(Src.lblVariable(N4).Tag)  ' V1
   nPositionN5 = VAL(Src.lblVariable(N5).Tag)  ' 2
   nPositionN6 = VAL(Src.lblVariable(N6).Tag)  ' 1

   IF gOccupantIndex(nvGroup, 0) = N1 THEN
      ' V2 = M1 * V1 / M2 * 2 / 1
      IF nPositionN2 < 4 AND nPositionN4 < 4 AND nPositionN3 > 3 AND nPositionN5 < 4 AND nPositionN6 > 3 THEN
         '   M1 on top      V1 on top         M2 on bottom            2 on top             1 on bottom
         ' Good formula.
         ' Place 2 and 1 in far right positions.
         SwitchTop N5, nvGroup, Src
         SwitchBottom N6, nvGroup, Src
         Src.lblDragDrop.Tag = STR$(N1)
         Src.lblDragDrop.Caption = "OB"  ' Runs optDependent_Click
      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = N2 THEN
      ' M1 = M2 * V2 / V1 * 1 / 2  N2 = N3 * N1 * N6 / N4 / N5
      IF nPositionN3 < 4 AND nPositionN1 < 4 AND nPositionN4 > 3 AND nPositionN6 < 4 AND nPositionN5 > 3 THEN
         '   M2 on top      V2 on top         V1 on bottom            1 on top             2 on bottom
         ' Good formula.
         SwitchTop N6, nvGroup, Src
         SwitchBottom N5, nvGroup, Src
         Src.lblDragDrop.Tag = STR$(N2)
         Src.lblDragDrop.Caption = "OB"  ' "Click" option button.
      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = N3 THEN
      ' M2 = M1 * V1 / V2 * 2 / 1  N3 = N2 * N4 * N5 / N1 / N6
      IF nPositionN2 < 4 AND nPositionN4 < 4 AND nPositionN1 > 3 AND nPositionN5 < 4 AND nPositionN6 > 3 THEN
         '   M1 on top      V1 on top         V2 on bottom            2 on top             1 on bottom
         ' Good formula.
         SwitchTop N5, nvGroup, Src
         SwitchBottom N6, nvGroup, Src
         Src.lblDragDrop.Tag = STR$(N3)
         Src.lblDragDrop.Caption = "OB"  ' "Click" option button.
      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = N4 THEN
      ' V1 = M2 * V2 / M1 * 1 / 2   N4 = N3 * N1 * N6 / N2 / N5
      IF nPositionN3 < 4 AND nPositionN1 < 4 AND nPositionN2 > 3 AND nPositionN6 < 4 AND nPositionN5 > 3 THEN
         '   M2 on top      V2 on top         M1 on bottom            1 on top             2 on bottom
         ' Good formula.
         SwitchTop N6, nvGroup, Src
         SwitchBottom N5, nvGroup, Src
         Src.lblDragDrop.Tag = STR$(N4)
         Src.lblDragDrop.Caption = "OB"  ' "Click" option button.
      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = N5 THEN
      ' 2 = M2 * V2 / M1 / V1 * 1    N5 = N3 * N1 / N2 / N4 * N6
      IF nPositionN3 < 4 AND nPositionN1 < 4 AND nPositionN2 > 3 AND nPositionN4 > 3 AND nPositionN6 < 4 THEN
         '   M2 on top      V2 on top         M1 on bottom           V1 on bottom         1 on top
         ' Good formula.
         SwitchTop N5, nvGroup, Src
         SwitchBottom -1, nvGroup, Src
         Src.lblDragDrop.Tag = STR$(N5)
         Src.lblDragDrop.Caption = "OB"  ' "Click" option button.
         gbDontStopHere(23) = FALSE
      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = N6 THEN
      ' 1 = M1 * V1 / M2 / V2 * 2    N6 = N2 * N4 / N3 / N1 * N5
      IF nPositionN2 < 4 AND nPositionN4 < 4 AND nPositionN3 > 3 AND nPositionN1 > 3 AND nPositionN5 < 4 THEN
         '   M1 on top      V1 on top         M2 on bottom           V2 on bottom         2 on top
         ' Good formula.
         SwitchTop N6, nvGroup, Src
         SwitchBottom -1, nvGroup, Src
         Src.lblDragDrop.Tag = STR$(N6)
         Src.lblDragDrop.Caption = "OB"  ' "Click" option button.
         gbDontStopHere(22) = FALSE
      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = -1 THEN
      bUnrecognizedFormula = TRUE
   END IF

   IF bUnrecognizedFormula THEN

      gbBadFormula(nvGroup) = TRUE
      snTries = snTries + 1
      IF nvGroup = 6 THEN nAdjust6 = 4
      FOR N = gGroupDefine(nvGroup).LowestIndex TO gGroupDefine(nvGroup).HighestIndex - nAdjust6
         Src.txtVariable(N).BackColor = CYAN
         gbDontStopHere(N) = TRUE
      NEXT N
      FOR N = 0 TO 3
         frmTitration.txtCoef1(N).Enabled = FALSE
         frmTitration.txtCoef2(N).Enabled = FALSE
      NEXT N
      Src.cmdSize.Enabled = FALSE
   ELSE
      IF gOccupantIndex(nvGroup, 0) < 22 THEN
         FOR N = 0 TO 3
            frmTitration.txtCoef1(N).Enabled = TRUE
            frmTitration.txtCoef2(N).Enabled = TRUE
            frmTitration.txtCoef1(N).BackColor = CYAN
            frmTitration.txtCoef2(N).BackColor = CYAN
         NEXT N
         Src.lblDragDrop.Caption = "U"  ' Update the coefficients.
      ELSE
         ' Dependent variable is coefficient 1 or 2
         FOR N = 0 TO 3
            frmTitration.txtCoef1(N).Enabled = FALSE
            frmTitration.txtCoef2(N).Enabled = FALSE
            frmTitration.txtCoef1(N).BackColor = WHITE
            frmTitration.txtCoef2(N).BackColor = WHITE
         NEXT N
      END IF
      gbBadFormula(nvGroup) = FALSE
      Src.cmdSize.Enabled = TRUE
      CalculateFormula gOccupantIndex(nvGroup, 0)
      IF snTries > 4 THEN
         ' Play "Clair de lune" when get the formula right.
         PLAY "MBT160MLO2L8A-L2O3A-L4F.O2L8AO3E-FL2E-"
         PLAY "MLO2L8A-O3D-E-D-.L4F.L8D-.O2G-O3CD-L2C."
      END IF
      snTries = 0
   END IF

END SUB

' Determine if the formula in frmPercent is arranged correctly.
'  Input:  nvGroup        Group to which form belongs.
'          SourceForm     Form to evaluate.
'
'  Called by:  DragDropPct
'  Module level variables:  mGroupDefine
'        Global variables:  gOccupantIndex, gbBadFormula, gbDontStopHere
'
SUB EvaluateFormulaPct (BYVAL nvGroup, SourceForm AS FORM)
DIM N, bUnrecognizedFormula, N1, N2, N3, N4
DIM sTemp AS STRING, nResult
DIM nPositionN1, nPositionN2, nPositionN3, nPositionN4
STATIC snTries
   N1 = gGroupDefine(nvGroup).LowestIndex    ' whole
   N2 = N1 + 1                                  ' part
   N3 = N1 + 2                                  ' percent
   N4 = N1 + 3                                  ' 100
   ' Get the present positions of the variables.
   nPositionN1 = VAL(SourceForm.lblVariable(N1).Tag)
   nPositionN2 = VAL(SourceForm.lblVariable(N2).Tag)
   nPositionN3 = VAL(SourceForm.lblVariable(N3).Tag)
   nPositionN4 = VAL(SourceForm.lblVariable(N4).Tag)

   IF gOccupantIndex(nvGroup, 0) = N1 THEN
      ' Whole on left side.
      ' Whole =part * 100 / percent
      IF nPositionN2 < 4 AND nPositionN4 < 4 AND nPositionN3 > 3 THEN
         '   part on top         100 on top     percent on bottom
         ' Good formula.
         SourceForm.lblDragDrop.Tag = STR$(N1)
         SourceForm.lblDragDrop.Caption = "OB"  ' Runs optDependent_Click
         IF gOccupantIndex(nvGroup, 5) = N3 THEN
            ' Move from position 5 to position 4
            FOR N = 1 TO 12
               ' Move from 52, 0 to 27, 0.
               SourceForm.txtVariable(N3).MOVE 51 - 2 * N, 2
               SourceForm.lblVariable(N3).MOVE 51 - 2 * N + 19, 2
            NEXT N
            SourceForm.lblVariable(N3).Tag = "4"
            gOccupantIndex(nvGroup, 5) = -1
            gOccupantIndex(nvGroup, 4) = N3
         END IF
      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = N2 THEN
      ' Part on left side.
      ' Part = whole * percent / 100
      IF nPositionN1 < 4 AND nPositionN3 < 4 AND nPositionN4 > 3 THEN
         '   Whole on top     percent on top.     100 on bottom.
         ' Good formula.
         SourceForm.lblDragDrop.Tag = STR$(N2)
         SourceForm.lblDragDrop.Caption = "OB"  ' "Click" option button.
         IF gOccupantIndex(nvGroup, 5) = N4 THEN
            ' Move from position 5 to position 4
            FOR N = 1 TO 12
               ' Move from 52, 0 to 27, 0.
               SourceForm.txtVariable(N4).MOVE 63 - 2 * N, 2
               SourceForm.lblVariable(N4).MOVE 41 - 2 * N + 19, 2
            NEXT N
            SourceForm.lblVariable(N4).Tag = "4"
            gOccupantIndex(nvGroup, 5) = -1
            gOccupantIndex(nvGroup, 4) = N4
         END IF

      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = N3 THEN
   ' Percent on left side.
      IF nPositionN1 > 3 AND nPositionN2 < 4 AND nPositionN4 < 4 THEN
         ' Whole on bottom      part on top         100 on top
         ' Good formula.
         SourceForm.lblDragDrop.Tag = STR$(N3)
         SourceForm.lblDragDrop.Caption = "OB"  ' "Click" option button.
         IF gOccupantIndex(nvGroup, 5) = N1 THEN
            ' Move from position 5 to position 4
            FOR N = 1 TO 12
               ' Move from 52,0 to 27,0.
               SourceForm.txtVariable(N1).MOVE 51 - 2 * N, 2
               SourceForm.lblVariable(N1).MOVE 51 - 2 * N + 19, 2
            NEXT N
            SourceForm.lblVariable(N1).Tag = "4"
            gOccupantIndex(nvGroup, 5) = -1
            gOccupantIndex(nvGroup, 4) = N1
         END IF
      ELSE
         bUnrecognizedFormula = TRUE
      END IF
   ELSEIF gOccupantIndex(nvGroup, 0) = -1 THEN
      bUnrecognizedFormula = TRUE
   END IF

   IF bUnrecognizedFormula THEN

      gbBadFormula(4) = TRUE
      snTries = snTries + 1
      FOR N = gGroupDefine(nvGroup).LowestIndex TO gGroupDefine(nvGroup).HighestIndex
         SourceForm.txtVariable(N).BackColor = CYAN
         gbDontStopHere(N) = TRUE
      NEXT N
      SourceForm.cmdSize.Enabled = FALSE

   ELSE
      gbBadFormula(4) = FALSE
      SourceForm.cmdSize.Enabled = TRUE
      CalculateFormula gOccupantIndex(nvGroup, 0)
      IF snTries > 4 THEN
         ' Play "Clair de lune" when get the formula right.
         PLAY "MBT160MLO2L8A-L2O3A-L4F.O2L8AO3E-FL2E-"
         PLAY "MLO2L8A-O3D-E-D-.L4F.L8D-.O2G-O3CD-L2C."
      END IF
      snTries = 0
   END IF

END SUB

' Assigns positions for frmGas variables when form is small when option
' button is selected.
'
'      Input:  Index    Index of new dependent variable.
'              Src      The form frmGas.
'
'  Called By:  nOptionButton
'
SUB GasOption (BYVAL Index, Src AS FORM)
DIM nInitialPosition, nOriginalIndexAt0, nOriginalIndexAt1
DIM nOriginalIndexAt4or5, n4or5

nInitialPosition = VAL(Src.lblVariable(Index).Tag)
IF Index = gOccupantIndex(7, 0) THEN
   ' Do nothing if txtVariable is already in dependent position.
ELSEIF nInitialPosition = 4 OR nInitialPosition = 5 THEN
   ' Variable is in denominator; exchange with present dependent variable.
   nOriginalIndexAt0 = gOccupantIndex(7, 0)
   gOccupantIndex(7, 0) = Index
   gOccupantIndex(7, nInitialPosition) = nOriginalIndexAt0
   Src.lblVariable(gOccupantIndex(7, 0)).Tag = "0"
   Src.lblVariable(gOccupantIndex(7, nInitialPosition)).Tag = LTRIM$(STR$(nInitialPosition))

   ' To keep pairs of variable together, exchange position 1 and 2 variables.
   nOriginalIndexAt1 = gOccupantIndex(7, 1)
   gOccupantIndex(7, 1) = gOccupantIndex(7, 2)
   Src.lblVariable(gOccupantIndex(7, 1)).Tag = "1"
   gOccupantIndex(7, 2) = nOriginalIndexAt1
   Src.lblVariable(gOccupantIndex(7, 2)).Tag = "2"

ELSEIF nInitialPosition = 1 THEN
   ' Variable is in numerator
   '   X1 = X2 * X3  --> X2 = X1 * X4
   '           X4                X3
   ' First exchange "X1" and "X2", in locations 0 and 1.
   nOriginalIndexAt0 = gOccupantIndex(7, 0)
   gOccupantIndex(7, 0) = Index
   gOccupantIndex(7, 1) = nOriginalIndexAt0
   Src.lblVariable(gOccupantIndex(7, 0)).Tag = "0"
   Src.lblVariable(gOccupantIndex(7, nInitialPosition)).Tag = "1"

   ' Now exchange "X3" and "X4", in locations 2 and 4 (or 5).
   nOriginalIndexAt4or5 = gOccupantIndex(7, 4)
   n4or5 = 4
   IF nOriginalIndexAt4or5 = -1 THEN
      nOriginalIndexAt4or5 = gOccupantIndex(7, 5)
      n4or5 = 5
   END IF
   gOccupantIndex(7, n4or5) = gOccupantIndex(7, 2)
   Src.lblVariable(gOccupantIndex(7, n4or5)).Tag = LTRIM$(STR$(n4or5))
   gOccupantIndex(7, 2) = nOriginalIndexAt4or5
   Src.lblVariable(gOccupantIndex(7, 2)).Tag = "2"
ELSEIF nInitialPosition = 2 THEN
   ' Variable is in numerator
   '   X1 = X2 * X3  --> X3 = X4 * X1
   '           X4                X2
   nOriginalIndexAt0 = gOccupantIndex(7, 0)
   nOriginalIndexAt1 = gOccupantIndex(7, 1)
   nOriginalIndexAt4or5 = gOccupantIndex(7, 4)
   n4or5 = 4
   IF nOriginalIndexAt4or5 = -1 THEN
      nOriginalIndexAt4or5 = gOccupantIndex(7, 5)
      n4or5 = 5
   END IF

   gOccupantIndex(7, 0) = Index
   Src.lblVariable(gOccupantIndex(7, 0)).Tag = "0"
   gOccupantIndex(7, 1) = nOriginalIndexAt4or5
   Src.lblVariable(gOccupantIndex(7, 1)).Tag = "1"
   gOccupantIndex(7, 2) = nOriginalIndexAt0
   Src.lblVariable(gOccupantIndex(7, 2)).Tag = "2"
   gOccupantIndex(7, n4or5) = nOriginalIndexAt1
   Src.lblVariable(gOccupantIndex(7, n4or5)).Tag = LTRIM$(STR$(n4or5))
END IF
END SUB

' Evaluate text change in txtVariable.
'  Input    Index          Index of txtVariable that is losing focus.
'           nvGroup        Group of that txtVariable.
'           CallingForm    Form of that txtVariable.
'
'  Called by:  txtVariable_LostFocus
SUB GeneralLostFocus (BYVAL Index, BYVAL nvGroup, CallingForm AS FORM)

DIM bError, N
DIM nDependent
DIM dNumber AS DOUBLE
DIM dUncertainty AS DOUBLE
DIM srMsg AS STRING
DIM nrPrintPosition
' gdNumber(0), gdUncertainty(0)  refer to molar mass.
' gdNumber(1), gdUncertainty(1)  refer to grams.
' gdNumber(2), gdUncertainty(2)  refer to moles.
' This is related to display of starting formula as follows:
'        2  =  1 / 0

' Do nothing if formula is not arranged correctly.
IF gbBadFormula(nvGroup) THEN EXIT SUB

' Do nothing if leaving dependent variable.
nDependent = gOccupantIndex(nvGroup, 0)
IF Index = nDependent THEN EXIT SUB

' If text changed, analyze input, do calculation.
IF CallingForm.txtVariable(Index).Text <> gsOriginalText(nvGroup) THEN
   ' Text did change.
   IF gnConnected(Index) THEN
      ' Connected, but text changed, indicating need to break the connection.
      Disconnect Index
   END IF
   bError = bAnalyzeInput(CallingForm.txtVariable(Index).Text, dNumber, dUncertainty)
   IF bError THEN
      ' Unrecognized input; try again.
      CallingForm.txtVariable(Index).SETFOCUS
      EXIT SUB
   END IF
   NumberFormat dNumber, dUncertainty, 51, srMsg, nrPrintPosition

   ' Update the entry.
   ' Note that the entry is text.  If too small, VAL returns 0.
   CallingForm.txtVariable(Index).Text = srMsg
   gdNumber(Index) = dNumber
   gdUncertainty(Index) = dUncertainty
   CalculateFormula Index
END IF
' If text hasn't changed, do nothing but change color.
CallingForm.txtVariable(Index).BackColor = CYAN

END SUB

' This is needed to set the focus back to the calling form.
'
SUB InfoShow (LastControl AS CONTROL, BYVAL svInfo AS STRING, BYVAL nvColor)
   frmInfo.lblInfo.BackColor = nvColor
   frmInfo.lblInfo.Caption = svInfo
   LastControl.SETFOCUS
END SUB

'
' Pressing enter moves to next field.
SUB InterpretKeyPress (BYVAL Index, KeyAscii, nvGroup, Src AS FORM)
DIM Counter, N
' Allowed keys are enter, escape, backspace, delete, and ()eE-.0123456789.

IF gbDontStopHere(Index) THEN
   ' If the formula is invalid, or this is the dependent variable,
   ' then don't allow the user to enter anything here.

   ' The alternative is to set control.enabled = false, but this prevents
   ' rearranging an invalid formula with the keyboard, which users without
   ' a mouse need.
   KeyAscii = 0
ELSEIF KeyAscii = KEY_RETURN THEN
   ' Go to next entry.
   KeyAscii = 0
   N = nSetNextFocus(Index)
ELSEIF KeyAscii = KEY_ESCAPE THEN
   KeyAscii = 0
   Src.txtVariable(Index).Text = gsOriginalText(nvGroup)
ELSEIF KeyAscii = 40 OR KeyAscii = 41 OR KeyAscii = KEY_BACK OR KeyAscii = KEY_DELETE THEN
   ' Some key codes:  40 = (; 41 = ).
   ' Allowed keys; do nothing (this line simplifies the next).
ELSEIF (KeyAscii < 48 OR KeyAscii > 57) AND KeyAscii <> 45 AND KeyAscii <> 46 AND KeyAscii <> 69 AND KeyAscii <> 101 THEN
   ' Some key codes:  45 = -; 46 = .; 69 = E; 101 = e.
   KeyAscii = 0
   BEEP
END IF

END SUB

' Select a color indicating that an independent and dependent variable
' are connected.
'  Returns:  the next color to use
'
FUNCTION nGetConnectColor () AS INTEGER
STATIC nLastColor

' BLACK = 0                       ' Possible colors and conclusions.
' BLUE = 1     ' too dark
' GREEN = 2    ' works
' CYAN = 3
' RED = 4      ' hard to read, but usable
' MAGENTA = 5  ' hard to read, but usable
' BROWN = 6    ' works
' WHITE = 7    '
' GRAY = 8     ' hard to read, but usable
' BRIGHT_BLUE = 9    ' works
' BRIGHT_GREEN = 10  ' works
' BRIGHT_CYAN = 11   ' a bit bright
' BRIGHT_RED = 12    ' works
' PINK = 13          ' works
' YELLOW = 14        '
' BRIGHT_WHITE = 15  ' can't see white letters

nLastColor = (nLastColor + 1) MOD 6
SELECT CASE nLastColor
   CASE 0
      nGetConnectColor = GREEN
   CASE 1
      nGetConnectColor = BROWN
   CASE 2
      nGetConnectColor = BRIGHT_BLUE
   CASE 3
      nGetConnectColor = BRIGHT_GREEN
   CASE 4
      nGetConnectColor = BRIGHT_RED
   CASE 5
      nGetConnectColor = PINK
END SELECT

END FUNCTION

' Sets the focus to a txtVariable(Index) which may be on any form.
'  Input:      nvIndex   Index of variable of interest
'  Returns:    index to which focus was set.
FUNCTION nSetNextFocus (BYVAL nvIndex) AS INTEGER
DIM Counter
   NextFocus nvIndex, Counter
   SELECT CASE Counter
      CASE IS < 4
         'frmGramMole.SETFOCUS
         frmGramMole.txtVariable(Counter).SETFOCUS
      CASE IS < 7
'formGone         frmDensity.txtVariable(Counter).SETFOCUS
      CASE IS < 10
'formgone         frmMolarity.txtVariable(Counter).SETFOCUS
      CASE IS < 13
         frmPercent.txtVariable(Counter).SETFOCUS
      CASE IS < 18
'formGone         frmDilution.txtVariable(Counter).SETFOCUS
      CASE IS < 26
         frmTitration.txtVariable(Counter).SETFOCUS
      CASE IS < 34
         frmGas.txtVariable(Counter).SETFOCUS
      CASE IS < 39
         frmIdealGas.txtVariable(Counter).SETFOCUS
      CASE IS < 73
         frmReaction.txtVariable(Counter).SETFOCUS
   END SELECT
   nSetNextFocus = Counter
END FUNCTION

' Prints a number, such as 1.00756463, having a variance, such as .0002,
' as 1.0076(2).
'    number          variance        output
'  1543423122           200      1.54341(2)E9.
'  -.0000456384823(2)            -4.56384823(2)E-5.
'  0                    0        0
'  1510                 0        1510.000000          ' ten digits
'  .000154              0        .0001540000000
'     Input:
'        dNumber     Number to be formatted
'        dStdDev     Standard deviation of number.  NON-NEGATIVE!!!
'        TabStop     Desired position of decimal
'     Output:
'        srMsg       Formatted string to print
'        rPrintX     Where printing should begin so decimals line up.
'                    Exponential notation is presently distinguished from
'                    non-exponential notation by having the decimal one
'                    place further to the left.
'
SUB NumberFormat (BYVAL dNumber AS DOUBLE, BYVAL dStdDev AS DOUBLE, BYVAL TabStop, srMsg AS STRING, rPrintX AS INTEGER)
' Single precision numbers can be used in the call to this subroutine.
' They are automatically converted to double.

' The logic:
'  Exponential notation is required for large and small numbers.
'  The following covers all cases requiring exp. not. for large numbers.
'
'     Numbers larger than 1,234,567,890 are expressed in exp. notation,
'        because only ten digits of precision are allowed.  This is
'        regardless of the size of the uncertainty.

'     If the uncertainty is less than 1 part in 10^10, it can be assigned
'        to be 1 part in 10^10.  Assigned uncertainties are not displayed.
'        That is, instead of 1,234,567,890(1), they appear as 1,234,567,890.
'     Rounded uncertainty or assigned uncertainty greater than 1
'        requires exp. not, regardless of the magnitude of the number.
'
'  For small numbers, the choice of when to use exp. not. is arbitrary.
'
'     Because ten digits of precision can be displayed, one choice is
'        to use exp. not. if the uncertainty is less than 9.5E-10 and
'        greater than 0 (and number is less than 1).  This results in
'        printing nine zeroes, which are hard to read, .0000000001.
'        Another choice is to use exp. not. whenever the uncertainty
'        is less than 9.5E-7.  This results in the following:
'                       Input             Output
'                       .123456(2)       .123456(2)
'                       .1234567(2)      1.234567(2)e-1
'                       .012345(2)       .012345(2)
'                       .00123456(2)     1.23456(2)e-2
'                       .001234(2)       .001234(2)
'                       .0012345(2)      1.2345(2)e-3
'                       .000123(2)       .000123(2)
'                       .0001234(2)      1.234(2)e-4
'                       .000012(2)       .000012(2)
'                       .0000123(2)      1.23(2)e-5
'                       .000001(2)       .000001(2)
'                       .0000012(2)      1.2(2)e-6
'                       .0000001(2)      1.(2)e-7
'
'  Nonexponential notation is used for intermediate numbers.  How to
'  format them depends on whether they are greater or less than 1.

   DIM sNumber AS STRING, sStdDev AS STRING
   DIM DecimalStdDev, DecimalNumber, zero, LastZero
   DIM bNumberIsNegative
   DIM NumberExponent, StdDevExponent
   DIM DiffStdDev_Num_Exp
   DIM NumberLength
   DIM N, sNegativeCharacter AS STRING
   DIM nDecimalPosition

   IF dNumber < 0 THEN
      ' For negative numbers.
      dNumber = -dNumber            ' Make the number positive.
      TabStop = TabStop - 1         ' The - sign changes the decimal position.
      sNegativeCharacter = "-"      ' Add - to formated string at end of sub.
   END IF

   IF dNumber < dStdDev THEN
      NumberFormat dNumber, dNumber / 2#, TabStop, srMsg, rPrintX
      N = INSTR(srMsg, "(")
      srMsg = LEFT$(srMsg, N) + "big" + MID$(srMsg, N + 2)
      srMsg = sNegativeCharacter + srMsg
      EXIT SUB
   END IF

   ' How to format the number depends on the magnitude of the uncertainty.
   IF dStdDev = 0 THEN
      ' Show ten digits.
      IF dNumber = 0 THEN
         srMsg = "0"
         rPrintX = TabStop - 1
      ELSEIF dNumber >= 10000000000# OR dNumber < .00000095# THEN
         ' Use exponential format.
         srMsg = FORMAT$(dNumber, "0.000000000E-0")
         IF MID$(srMsg, 2, 1) = "0" THEN
            ' A bug with format$:  9.8E-5 is rounded to 10E-5, not 1E-4.
            ' Change 10.00000000e-x to 1.000000000e-(x-1)
            srMsg = "1.000000000E" + FORMAT$(VAL(MID$(srMsg, 14)) + 1)
         END IF
         rPrintX = TabStop - 2   ' Moved over 1 place, since exponential not.
      ELSEIF dNumber < 9.95 THEN
         ' Use non-exponential format for numbers less than 10.
         sNumber = FORMAT$(dNumber, "0.000000000E+0")
         srMsg = FORMAT$(VAL(sNumber))                ' back to non-exp. not.

         nDecimalPosition = INSTR(2, sNumber, gsIntDecimal)
         ' Print position:  to print 123.45 with the decimal at TabStop,
         ' start printing at TabStop - (DecimalNumber - 1)
         rPrintX = TabStop - nDecimalPosition + 1
      ELSE
         ' Use non-exponential format for large numbers.
         sNumber = FORMAT$(dNumber, "0.000000000E+0")
         srMsg = FORMAT$(VAL(sNumber))                ' back to non-exp. not.
         nDecimalPosition = INSTR(2, srMsg, gsIntDecimal)
         IF nDecimalPosition = 0 THEN
            ' No decimal.
            nDecimalPosition = LEN(srMsg) + 1   ' Where the decimal would be.
         END IF
         ' Add thousands separators, as in 1,430,345.
         AddThousandsSeparators srMsg, nDecimalPosition
         rPrintX = TabStop - nDecimalPosition + 1
      END IF
   ELSEIF dStdDev < .00000095# AND dNumber < 1 THEN

      ' ********************* Negative exponential format ******************
      ' Format the uncertainty, such as 1.543e-5, as 2e-5.  Then
      ' format the number, such as 1.234567890654e-1, with ten digits:
      ' 1.234567891e-1.  The location of the first nonsignificant digit is:
      '            absolute value of uncertainty exponent
      '          - absolute value of number exponent
      '          + 3
      ' 7 in this case.  If this digit is 5 or greater, round number
      ' up:  1.2346e-1.
      '                 ---Another example:  if number is 1.8e-6 and
      '                 ---uncertainty is 1e-6, look at the third digit of
      '                 ---number, the 8, to determine whether to round.
      '

      ' Prepare uncertainty string.
      sStdDev = FORMAT$(dStdDev, "0E+0")        ' Format with 1 digit & exp.
      IF MID$(sStdDev, 2, 1) = "0" THEN         ' A bug with format$.
         sStdDev = "1E" + FORMAT$(VAL(MID$(sStdDev, 4)) + 1)
      END IF
      ' sStdDev is now something like "8E-9"
      srMsg = "(" + LEFT$(sStdDev, 1) + ")E-"
      StdDevExponent = VAL(MID$(sStdDev, 4))   ' Start past negative sign on exponent.
      
      sNumber = FORMAT$(dNumber, "0.000000000E+0") ' Prepare number string.
      IF MID$(sNumber, 2, 1) = "0" THEN
         ' If get 10.00000000e-x, change to 1.000000000e-(x-1)
         sNumber = "1.000000000E" + FORMAT$(VAL(MID$(sNumber, 14)) + 1)
      END IF
      NumberExponent = VAL(MID$(sNumber, 14))   ' Start past negative sign on exponent.
      DiffStdDev_Num_Exp = StdDevExponent - NumberExponent

      IF DiffStdDev_Num_Exp > 9 THEN
         ' Precision of number exceeds ten digits; display without uncertainty.
         srMsg = sNumber
      ELSE
         ' Round number up, if necessary, then trim it.
         IF VAL(MID$(sNumber, DiffStdDev_Num_Exp + 3, 1)) > 4 THEN
            ' Never get here if have to adjust exponent, above.
            dNumber = dNumber + 1# * 10 ^ -StdDevExponent
            sNumber = FORMAT$(dNumber, "0.000000000E+0")
            NumberExponent = VAL(MID$(sNumber, 14))
         END IF
         srMsg = srMsg + MID$(sNumber, 14)  ' Get number exponent after rounding.
         ' Trim off unsignificant digits in number.
         sNumber = LEFT$(sNumber, StdDevExponent - NumberExponent + 2)
         srMsg = sNumber + srMsg
      END IF
      rPrintX = TabStop - 2   ' Decimal in exponent notation is moved left
                              ' 1 place so immediately obvious it isn't a
                              ' regular number.

   ELSEIF dStdDev < .95 THEN
      ' ********************* Non-exponential format **********************
      ' Rounded uncertainty is less than 1 (but greater than .00000095 or
      ' wouldn't get here).

      ' If precision of number is greater than 10 digits,
      ' display ten digits of dNumber without uncertainty.
      IF dNumber / dStdDev < 1E+10 THEN
         ' Fewer than 10 digits of precision; show uncertainty.

         ' Rounding procedure:
         '           Find the last zero before a digit in the
         '           uncertainty.  In 0.01, last zero is position 3.
         '           So the position of the least significant figure,
         '           relative to the decimal point, is last zero - 1.

         '           Find the decimal position in number.  In 123.021,
         '           decimal position is 4.

         '           The location of the least significant figure in number
         '           is given by decimal position + last zero - 1,
         '           in this case 6:      .01  uncertainty
         '                             123.026 number
         '                             123.03  rounded number

         ' Find the first non-zero character after the decimal in uncertainty.
         sStdDev = FORMAT$(dStdDev, "0.00000000000")

         LastZero = 2
         DO
            zero = INSTR(LastZero + 1, sStdDev, "0")
            IF zero > LastZero + 1 OR zero = 0 THEN
               ' Following character isn't zero.
               EXIT DO
            ELSE
               LastZero = zero
            END IF
         LOOP

         ' Round dStdDev
         IF VAL(MID$(sStdDev, LastZero + 2, 1)) > 4 THEN
            dStdDev = dStdDev + 1 * 10 ^ -(LastZero - 1)
            ' If this caused, say, .00095 to be rounded to .0010, then
            ' display 1 less digit in dNumber.
            sStdDev = FORMAT$(dStdDev, "0.00000000000")
            IF MID$(sStdDev, LastZero, 1) = "1" THEN
               LastZero = LastZero - 1
            END IF
         END IF
         sStdDev = "(" + MID$(sStdDev, LastZero + 1, 1) + ")"

         ' Round the number.
         ' Get the digit representing uncertainty in the last digit of dNumber.
         sNumber = FORMAT$(dNumber, "0.0000000000")
         DecimalNumber = INSTR(2, sNumber, gsIntDecimal)
         IF VAL(MID$(sNumber, DecimalNumber + LastZero, 1)) > 4 THEN
            dNumber = dNumber + 1# * 10 ^ -(LastZero - 1)
            sNumber = FORMAT$(dNumber, "0.000000000")
            DecimalNumber = INSTR(2, sNumber, gsIntDecimal)
         END IF
         srMsg = LEFT$(sNumber, DecimalNumber + LastZero - 1)  ' Trim excess figures.

         ' Add thousands separators, as in 1,430,345.
         AddThousandsSeparators srMsg, DecimalNumber

         srMsg = srMsg + sStdDev
         rPrintX = TabStop - DecimalNumber + 1
      ELSE
         ' More than ten digits of precision.
         ' Since the limit of precision for non-exponential notation is
         ' 1e-7, and more than ten digits are present, number must be
         ' greater than 1.
         sNumber = FORMAT$(dNumber, "0.000000000E+0") ' Round number.
            ' FORMAT$ without a format string trims zeroes beyond the decimal.
         srMsg = FORMAT$(VAL(sNumber), "0.0000000000")  ' back to non-exp. not.
         ' A number, such as, 1000.0000000(1) is now 1000 without decimal.
         srMsg = LEFT$(srMsg, 11)
         DecimalNumber = INSTR(2, srMsg, gsIntDecimal)
         AddThousandsSeparators srMsg, DecimalNumber
         rPrintX = TabStop - DecimalNumber + 1
      END IF
   ELSEIF dStdDev < 9.5 AND dNumber < 10000000000# THEN
      ' ************ Non-exponential format, 10 > uncertainty > 1 ***********
      ' Uncertainty is at least 1 and less than 10.  If Number was too large,
      ' would go to exp. not. for large numbers (next ELSE).

      sStdDev = "(" + FORMAT$(dStdDev, "0") + ")"

      ' Round dNumber
      sNumber = FORMAT$(dNumber, "0")
      nDecimalPosition = LEN(sNumber) + 1    ' Where the decimal would be.
      srMsg = sNumber + sStdDev
      AddThousandsSeparators srMsg, nDecimalPosition
      rPrintX = TabStop - nDecimalPosition + 1
   ELSE
      ' ****************** large number exponential format ******************
      ' The rounded uncertainty is greater than 10, or the number is greater
      ' than or equal to 10000000000.

      IF dNumber / dStdDev < 1E+10 THEN
         ' Fewer than 10 digits of precision; show uncertainty.
         ' Uncertainty is greater than 9.95, or would never get here.
         ' If uncertainty is less than than, then
         '     a)  number is large, in which case will go to the other part
         '         of this IF statement; or
         '     b)  number is small, in which case will have gone to one of
         '         the three preceeding parts of this sub.

         ' Round standard deviation.
         sStdDev = FORMAT$(dStdDev, "0")   ' Works, but may get a string 300 characters long.
         DecimalStdDev = LEN(sStdDev)
         IF VAL(MID$(sStdDev, 2, 1)) > 4 THEN
            ' Round up
            dStdDev = dStdDev + 1 * 10 ^ (DecimalStdDev - 1)
            sStdDev = FORMAT$(dStdDev, "0")
            DecimalStdDev = LEN(sStdDev)
         END IF
         sStdDev = LEFT$(sStdDev, 1)

         ' Round the Number.  Since uncertainty is 10 or greater,
         ' don't need to consider decimals in number.
         sNumber = FORMAT$(dNumber, "0")
         DecimalNumber = LEN(sNumber)
         IF VAL(MID$(sNumber, DecimalNumber - DecimalStdDev + 2, 1)) > 4 THEN
            dNumber = dNumber + 10 ^ (DecimalStdDev - 1)
            ' Rounding could have changed 99 to 100, so recheck the length.
            sNumber = FORMAT$(dNumber, "0")
            DecimalNumber = LEN(sNumber)
         END IF

         ' Compose the string to print.
         srMsg = LEFT$(sNumber, 1) + gsIntDecimal
         srMsg = srMsg + MID$(sNumber, 2, DecimalNumber - DecimalStdDev)
         srMsg = srMsg + "(" + sStdDev + ")E+" + FORMAT$(DecimalNumber - 1)
      ELSE
         ' More than ten digits of precision.
         ' Round dNumber, using exp. not.
         srMsg = FORMAT$(dNumber, "0.000000000E+0")
      END IF
      rPrintX = TabStop - 2   ' Decimal in exponent notation is moved left
                              ' 1 place so immediately obvious it isn't a
                              ' regular number.
   END IF
   
   srMsg = sNegativeCharacter + srMsg  ' Add negative sign to negative numbers.
END SUB

' Switchs position of coefficient 2 or 1 in titration on far right bottom,
' or places blank in far right bottom when nvFarRightIndex is -1.
'  Input:  nvFarRightIndex    Index of txtVariable, or, if negative, position
'                             containing txtVariable that goes on far right.
'          nvGroup            Group to which form belongs.
'          Src                Form to evaluate.
'
'  Called by:  EvaluateFormula6
'  Module level variables:
'        Global variables:  gOccupantIndex
'
SUB SwitchBottom (BYVAL nvFarRightIndex, BYVAL nvGroup, Src AS FORM)
DIM nTmpIndex, N
   IF gOccupantIndex(nvGroup, 4) = nvFarRightIndex THEN
      ' Exchange positions 4 and 6.
      nTmpIndex = gOccupantIndex(nvGroup, 6)
      FOR N = 2 TO 44 STEP 2
         ' Move from 23,3 to 67,3. 44
         IF nvFarRightIndex > -1 THEN
            Src.txtVariable(nvFarRightIndex).MOVE nvFarRightIndex + N, 3
            Src.lblVariable(nvFarRightIndex).MOVE nvFarRightIndex + N, 4
         END IF
         IF nTmpIndex > -1 THEN
            Src.txtVariable(nTmpIndex).MOVE 67 - N, 3
            Src.lblVariable(nTmpIndex).MOVE 67 - N, 4
         END IF
      NEXT N
      IF nvFarRightIndex > -1 THEN
         Src.lblVariable(nvFarRightIndex).Tag = "6"
      END IF
      gOccupantIndex(nvGroup, 6) = nvFarRightIndex
      IF nTmpIndex > -1 THEN
         Src.lblVariable(nTmpIndex).Tag = "4"
      END IF
      gOccupantIndex(nvGroup, 4) = nTmpIndex
   ELSEIF gOccupantIndex(nvGroup, 5) = nvFarRightIndex THEN
      ' Exchange positions 5 and 6.
      nTmpIndex = gOccupantIndex(nvGroup, 6)
      FOR N = 2 TO 22 STEP 2
         ' Move from 45,3 to 67,3.   22
         IF nvFarRightIndex > -1 THEN
            Src.txtVariable(nvFarRightIndex).MOVE 45 + N, 3
            Src.lblVariable(nvFarRightIndex).MOVE 45 + N, 4
         END IF
         IF nTmpIndex > -1 THEN
            Src.txtVariable(nTmpIndex).MOVE 67 - N, 3
            Src.lblVariable(nTmpIndex).MOVE 67 - N, 4
         END IF
      NEXT N
      IF nvFarRightIndex > -1 THEN
         Src.lblVariable(nvFarRightIndex).Tag = "6"
      END IF
      gOccupantIndex(nvGroup, 6) = nvFarRightIndex
      IF nTmpIndex > -1 THEN
         Src.lblVariable(nTmpIndex).Tag = "5"
      END IF
      gOccupantIndex(nvGroup, 5) = nTmpIndex
   END IF

END SUB

' Moves a variable from present position to empty position on same line.
'  Input:  nvToFill     Position to fill, if empty.
'          nvToEmpty    Position to empty, if a variable is there.
'          nvDistance   Distance between the two positions.
'          nvGroup      Group to which form belongs.
'          Src          Form to evaluate.
'
'  Called by:  EvaluateFormula5
'  Module level variables:
'        Global variables:  gOccupantIndex
'
SUB SwitchGeneral (BYVAL nvToFill, BYVAL nvToEmpty, BYVAL nvDistance, BYVAL nvGroup, Src AS FORM)
DIM nTmpIndex, N, nInitialLeftTxt, nInitialLeftLbl
' Do nothing if position already filled.
IF gOccupantIndex(nvGroup, nvToFill) = -1 THEN
   IF gOccupantIndex(nvGroup, nvToEmpty) > -1 THEN
      ' Exchange positions nvToFill and nvToEmpty
      nTmpIndex = gOccupantIndex(nvGroup, nvToEmpty)
      nInitialLeftTxt = Src.txtVariable(nTmpIndex).Left
      nInitialLeftLbl = Src.lblVariable(nTmpIndex).Left
      FOR N = 2 TO nvDistance - 1 STEP 2
         Src.txtVariable(nTmpIndex).MOVE nInitialLeftTxt - N
         Src.lblVariable(nTmpIndex).MOVE nInitialLeftLbl - N
      NEXT N
      ' A final move is needed for odd nvDistance.
      Src.txtVariable(nTmpIndex).MOVE nInitialLeftTxt - nvDistance
      Src.lblVariable(nTmpIndex).MOVE nInitialLeftLbl - nvDistance
      Src.lblVariable(nTmpIndex).Tag = STR$(nvToFill)
      gOccupantIndex(nvGroup, nvToFill) = nTmpIndex
      gOccupantIndex(nvGroup, nvToEmpty) = -1
   END IF
END IF
END SUB

' Switchs position of coefficient 2 or 1 in titration on far right top.
'  Input:  nvFarRightIndex    Index of txtVariable that goes on far right.
'          nvGroup            Group to which form belongs.
'          Src                Form to evaluate.
'
'  Called by:  EvaluateFormula6
'  Module level variables:
'        Global variables:  gOccupantIndex
'
SUB SwitchTop (BYVAL nvFarRightIndex, BYVAL nvGroup, Src AS FORM)
DIM nTmpIndex, N

   IF gOccupantIndex(nvGroup, 1) = nvFarRightIndex THEN
      ' Exchange positions 1 and 3.
      nTmpIndex = gOccupantIndex(nvGroup, 3)
      FOR N = 2 TO 44 STEP 2
         ' Move from 23,1 to 67,1. 44
         Src.txtVariable(nvFarRightIndex).MOVE 23 + N, 1
         Src.lblVariable(nvFarRightIndex).MOVE 23 + N, 0
         Src.txtVariable(nTmpIndex).MOVE 67 - N, 1
         Src.lblVariable(nTmpIndex).MOVE 67 - N, 0
      NEXT N
      Src.lblVariable(nvFarRightIndex).Tag = "3"
      gOccupantIndex(nvGroup, 3) = nvFarRightIndex
      Src.lblVariable(nTmpIndex).Tag = "1"
      gOccupantIndex(nvGroup, 1) = nTmpIndex
   ELSEIF gOccupantIndex(nvGroup, 2) = nvFarRightIndex THEN
      ' Exchange positions 2 and 3.
      nTmpIndex = gOccupantIndex(nvGroup, 3)
      FOR N = 2 TO nvFarRightIndex STEP 2
         ' Move from 45,1 to 67,1.   nvFarRightIndex
         Src.txtVariable(nvFarRightIndex).MOVE 45 + N, 1
         Src.lblVariable(nvFarRightIndex).MOVE 45 + N, 0
         Src.txtVariable(nTmpIndex).MOVE 67 - N, 1
         Src.lblVariable(nTmpIndex).MOVE 67 - N, 0
      NEXT N
      Src.lblVariable(nvFarRightIndex).Tag = "3"
      gOccupantIndex(nvGroup, 3) = nvFarRightIndex
      Src.lblVariable(nTmpIndex).Tag = "2"
      gOccupantIndex(nvGroup, 2) = nTmpIndex
   END IF

END SUB

