/*===================================================================
 rootfind.h

 Version 1.2

 Written by:
   Brent Worden
   WordenWare
   email:  brent@worden.org

 Copyright (c) 1999-2001 WordenWare

 Created:  May 09, 1999
 Revised:  March 26, 2001
===================================================================*/

#ifndef _ROOTFIND_H_
#define _ROOTFIND_H_

#include "numerics.h"

typedef double (*Function)(double);
typedef double (*FunctionWithState)(double, void*);

NUMERICS_EXPORT double bisection(Function func, double x0, double x1, double tol);
/*-------------------------------------------------------------------
 Returns the root of func bracketed between x0 and x1 using the
 bisection method.  The returned root is within tol of the true
 value of the root.
-------------------------------------------------------------------*/

NUMERICS_EXPORT double bisectionWithState(FunctionWithState func, void* state, double x0, double x1, double tol);
/*-------------------------------------------------------------------
 Returns the root of func (parameterized by state) bracketed between
 x0 and x1 using the bisection method.  The returned root is within
 tol of the true value of the root.
-------------------------------------------------------------------*/

NUMERICS_EXPORT BOOL bracketOut(Function func, double *x0, double *x1);
/*-------------------------------------------------------------------
 Given a function func and an initial range (x0, x1), the range is
 expanded until a root is bracketed by the returned values (x0, x1).
 If a root is bracketed, TRUE is returned.  Otherwise, FALSE is
 returned.
-------------------------------------------------------------------*/

NUMERICS_EXPORT BOOL bracketOutWithState(FunctionWithState func, void* state, double *x0, double *x1);
/*-------------------------------------------------------------------
 Given a function func (parameterized by state) and an initial range
 (x0, x1), the range is expanded until a root is bracketed by the
 returned values (x0, x1).  If a root is bracketed, TRUE is returned.
 Otherwise, FALSE is returned.
-------------------------------------------------------------------*/

NUMERICS_EXPORT double falsePosition(Function func, double x0, double x1, double tol);
/*-------------------------------------------------------------------
 Returns the root of func using the false position method with
 initial approximations x0 and x1.  The returned root is within tol
 of the true value of the root.
-------------------------------------------------------------------*/

NUMERICS_EXPORT double newton(Function func, Function der, double x0, double tol);
/*-------------------------------------------------------------------
 Returns the root of func with first derivative der using Newton's
 method with initial approximation x0.  The returned root is within
 tol of the true value of the root.
-------------------------------------------------------------------*/

NUMERICS_EXPORT double secant(Function func, double x0, double x1, double tol);
/*-------------------------------------------------------------------
 Returns the root of func using the secant method with initial
 approximations x0 and x1.  The returned root is within tol of the
 true value of the root.
-------------------------------------------------------------------*/

#endif

/*===================================================================
 Revision History

 Version 1.0 - 05/09/1999 - New.
 Version 1.1 - 01/11/1999 - Added bracketOut.
 Version 1.2 - 03/26/2001 - Added bracketOutWithState and
                            bisectionWithState.
===================================================================*/
