
 /* ----------------------------------------------------------------------- */

 #ifndef __MakeLib
   float TimeToJD(const unsigned short hour, const unsigned short min, const unsigned short sec)
 #else
   float __saveds __asm TimeToJD(register __d0 const unsigned short hour, register __d1 const unsigned short min, register __d2 const unsigned short sec)
 #endif

/*
******* Date/TimeToJD *******************************************************
*
*   NAME
*	TimeToJD -- Returns the JD for a time. (V33)
*
*   SYNOPSIS
*	jd = TimeToJD(hour,min,sec);
*	d0	       d0   d1  d2
*
*	float TimeToJD(const unsigned short hour, const unsigned short min,
*	    const unsigned short sec);
*
*   FUNCTION
*	Returns the JD for a specified time.
*
*   INPUTS
*	hour - hour of the time to convert
*	min  - minute of the time to convert
*	sec  - sec. of the time to convert
*
*   RESULT
*	jd - This is the JD time
*
*   EXAMPLE
*	...
*	jd = TimeToJD(16,33,0);
*	...
*
*   NOTES
*	none
*
*   BUGS
*	There is no check, if the specified time is a valid time!
*
*   SEE ALSO
*	JDToTime()
*
*****************************************************************************
*
*
*/

 {
  return((float)(((unsigned long)hour*3600+(unsigned int)min*60+sec) / 86400.0));
 }


 #ifndef __MakeLib
   #ifndef __cplusplus
     void JDToTime(float jd, unsigned short *const rhour, unsigned short *const rmin, unsigned short *const rsec)
   #else
     void JDToTime(float jd, unsigned short &rhour, unsigned short &rmin, unsigned short &rsec)
   #endif
 #else
   void __saveds __asm JDToTime(register __d0 float jd, register __a0 unsigned short *const rhour, register __a1 unsigned short *const rmin, register __a2 unsigned short *const rsec)
 #endif

/*
******* Date/JDToTime *******************************************************
*
*   NAME
*	JDToTime -- Returns the real time for a JD time. (V33)
*
*   SYNOPSIS
*	JDToTime(jd,rhour,rmin,rsec);
*		 d0  a0    a1   a2
*
*	void JDToTime(float jd, unsigned short *const rhour,
*	    unsigned short *const rmin, unsigned short *const rsec);
*
*	void JDToTime(float jd, unsigned short &rhour, unsigned short &rmin,
*	    unsigned short &rsec);
*
*   FUNCTION
*	Returns the real time for a JD time.
*
*   INPUTS
*	jd - JD time
*
*   RESULT
*	rhour - 24 hour real time
*	rmin  - real minutes
*	rsec  - real seconds
*
*   EXAMPLE
*	...
*	JDToTime(0.76543,&rhour,&rmin,&rsec);
*	...
*
*   NOTES
*	none.
*
*   BUGS
*	If jd is > 0 (including days) there will be occur arithmetic bugs!
*
*   SEE ALSO
*	TimeToJD()
*
*****************************************************************************
*
*
*/

 {
  unsigned long sec;

  if (jd > 0.0)
    jd -= (float)floor((double)jd);
  sec = (unsigned long)(jd * 86400.0);
  #ifndef __cplusplus
    *rhour = (unsigned short)(sec / 3600);
    sec -= (*rhour) * 3600;
    *rmin = (unsigned short)(sec / 60);
    sec -= (*rmin) * 60;
    *rsec = (unsigned short)sec;
  #else
    rhour = (unsigned short)(sec / 3600);
    sec -= rhour * 3600;
    rmin = (unsigned short)(sec / 60);
    sec -= rmin * 60;
    rsec = (unsigned short)sec;
  #endif
 }

 /* ----------------------------------------------------------------------- */

 #ifndef __MakeLib
   short TimeZoneFactor(const short degree)
 #else
   short __saveds __asm TimeZoneFactor(register __d0 const short degree)
 #endif

/*
******* Date/TimeZoneFactor *************************************************
*
*   NAME
*	TimeZoneFactor -- Returns the value you have to add to GMT time (V33)
*
*   SYNOPSIS
*	addhours = TimeZoneFactor(degrees);
*	   d0			    d0
*
*	short TimeZoneFactor(const short degree);
*
*   FUNCTION
*	This gives you the hours you have to add to GMT time,
*	specified on the fact, that a timezone is 15 degrees
*	and that GMT is centered on 0 degrees!
*
*   INPUTS
*	degrees - Position of timezone you live in
*	(from -180 east to +180 west)
*
*   RESULT
*	addhours - Time to add to GMT time to get your locale zone time
*	    (-12 to +12)
*
*   EXAMPLE
*	...
*	addhours = TimeZoneFactor(-8);
*	...
*
*   NOTES
*	none
*
*   BUGS
*	No errorcheck, if you put in valid degrees (-180 to +180)
*	Only full degrees are supportet, keep sure that you
*	round in the right way for 0.x degree places
*	I am not sure about the correct +/- behaviour!!!
*
*   SEE ALSO
*
*
*****************************************************************************
*
*
*/

 {
  if (degree >= 0)
    return((short)(degree / 15.0 + 0.5));
  else
    return((short)(degree / 15.0 - 0.5));
 }


 #ifndef __MakeLib
   long LMT(const unsigned long secs, const float meridiandegree, const float posdegree)
 #else
   long __saveds __asm LMT(register __d0 const unsigned long secs, register __d1 const float meridiandegree, register __d2 const float posdegree)
 #endif

/*
******* Date/LMT ************************************************************
*
*   NAME
*	LMT -- Calculates your local time in your timezone (V33)
*
*   SYNOPSIS
*	secs = LMT(secs,meridian,pos);
*	 d0	    d0     d1    d2
*
*	unsigned long LMT(const unsigned long secs,
*	    const float meridiandegree, const float posdegree);
*
*   FUNCTION
*	Calculates your Local Mean Time of your place!
*
*   INPUTS
*	secs     - Seconds of the running day (hours*3600+min*60+sec)
*	meridian - Degrees of your timezone-meridian
*	pos      - Degrees of your place
*
*   RESULT
*	secs - Local seconds of the running day
*
*   EXAMPLE
*	...
*	secs = LMT(76080,-15.0,-8.923055556);
*	...
*
*   NOTES
*	none
*
*   BUGS
*	No errorcheck, if you put in valid degrees (-180 to +180)
*
*   SEE ALSO
*
*
*****************************************************************************
*
*
*/

 {
  return((long)secs + (long)((meridiandegree / 15.0 - posdegree / 15.0)*3600.0));
 }

 /* ----------------------------------------------------------------------- */

 #ifndef __MakeLib
   unsigned long TimeToSec(const unsigned short hour, const unsigned short min, const unsigned short sec)
 #else
   unsigned long __saveds __asm TimeToSec(register __d0 const unsigned short hour, register __d1 const unsigned short min, register __d2 const unsigned short sec)
 #endif

/*
******* Date/TimeToSec ******************************************************
*
*   NAME
*	TimeToSec -- Returns the time in seconds (V33)
*
*   SYNOPSIS
*	secs = TimeToSec(hour,min,sec);
*	 d0		  d0   d1  d2
*
*	unsigned long TimeToSec(const unsigned short hour,
*	    const unsigned short min, const unsigned short sec);
*
*   FUNCTION
*	Gives you back the time in seconds
*
*   INPUTS
*	hour - hours you want (0-23)
*	min  - minutes you want (0-59)
*	sec  - seconds you want (0-59)
*
*   RESULT
*	secs - Time in seconds
*
*   EXAMPLE
*	...
*	secs = TimeToSec(21,15,00);
*	...
*
*   NOTES
*	Don't forget to convert AM/PM time to 24h time!
*
*   BUGS
*	No errorcheck, if you use a valid time
*
*   SEE ALSO
*	SecToTime()
*
*****************************************************************************
*
*
*/

 {
  return((unsigned long)hour*3600+(unsigned long)min*60+sec);
 }


 #ifndef __MakeLib
   #ifndef __cplusplus
     void SecToTime(unsigned long secs, unsigned short *const hour, unsigned short *const min, unsigned short *const sec)
   #else
     void SecToTime(unsigned long secs, unsigned short &hour, unsigned short &min, unsigned short &sec)
   #endif
 #else
   void __saveds __asm SecToTime(register __d0 unsigned long secs, register __a0 unsigned short *const hour, register __a1 unsigned short *const min, register __a2 unsigned short *const sec)
 #endif

/*
******* Date/SecToTime ******************************************************
*
*   NAME
*	SecToTime -- Returns the time from seconds (V33)
*
*   SYNOPSIS
*	SecToTime(secs,hour,min,sec);
*		   d0   a0  a1  a2
*
*	void SecToTime(unsigned long secs, unsigned short *const hour,
*	    unsigned short *const min, unsigned short *const sec);
*
*	void SecToTime(unsigned long secs, unsigned short &hour,
*	    unsigned short &min, unsigned short &sec);
*
*   FUNCTION
*	Gives you back the time from the specified seconds
*
*   INPUTS
*	secs - Time in seconds
*
*   RESULT
*	hour - hours (0-23)
*	min  - minutes (0-59)
*	sec  - seconds (0-59)
*
*   EXAMPLE
*	...
*	SecToTime(76860,&hour,&min,&sec);
*	...
*
*   NOTES
*	Don't forget to convert 24h time to AM/PM time if needed!
*
*   BUGS
*	No errorcheck, if you use a valid time
*
*   SEE ALSO
*	TimeToSec()
*
*****************************************************************************
*
*
*/

 {
  #ifndef __cplusplus
    *hour = (unsigned short)(secs / 3600);
    secs -= (unsigned long)(*hour) * 3600;
    *min = (unsigned short)(secs / 60);
    *sec = (unsigned short)(secs - (unsigned long)(*min) * 60);
  #else
    hour = (unsigned short)(secs / 3600);
    secs -= (unsigned long)(hour) * 3600;
    min = (unsigned short)(secs / 60);
    sec = (unsigned short)(secs - (unsigned long)(min) * 60);
  #endif
 }

 /* ----------------------------------------------------------------------- */

 #ifndef __MakeLib
   short Compare2Times(const unsigned short hour1, const unsigned short min1, const unsigned short sec1, const unsigned short hour2, const unsigned short min2, const unsigned short sec2)
 #else
   short __saveds __asm Compare2Times(register __d0 const unsigned short hour1, register __d1 const unsigned short min1, register __d2 const unsigned short sec1, register __d3 const unsigned short hour2, register __d4 const unsigned short min2, register __d5 const unsigned short sec2)
 #endif

/*
******* Date/Compare2Times **************************************************
*
*   NAME
*	Compare2Times -- Compares time1 with time2. (V33.100)
*
*   SYNOPSIS
*	compare = Compare2Times(hour1,min1,sec1,hour2,min2,sec2);
*	  d0			 d0    d1   d2   d3    d4   d5
*
*	short Compare2Times(const unsigned short hour1,
*	    const unsigned short min1, const unsigned short sec1,
*	    const unsigned short hour2, const unsigned short min2,
*	    const unsigned short sec2);
*
*   FUNCTION
*	Compare2Times compares time1 with time2 (24h format only).
*
*   INPUTS
*	hour1 - Hour of the first time.
*	min1  - Minute of the first time.
*	sec1  - Second of the first time.
*	hour2 - Hour of the second time.
*	min2  - Minute of the second time.
*	sec2  - Second of the second time.
*
*   RESULT
*	compare - -1 : time1 < time2
*		   0 : time1 = time2
*		   1 : time1 > time2
*
*   EXAMPLE
*	...
*	if (Compare2Times(13,10,0,9,0,0) == -1)
*	  printf("<\n");
*	else
*	  printf(">=\n");
*	...
*
*   NOTES
*	This compares two times of 24h format!
*
*   BUGS
*	There is no check if the times are valid times!
*
*   SEE ALSO
*	Compare2Dates()
*
*****************************************************************************
*
*
*/

 {
  if (hour1 < hour2)
    return(-1);
  else if (hour1 > hour2)
    return(1);
  else if (min1 < min2)
    return(-1);
  else if (min1 > min2)
    return(1);
  else if (sec1 < sec2)
    return(-1);
  else if (sec1 > sec2)
    return(1);
  else
    return(0);
 }

 /* ----------------------------------------------------------------------- */

 #ifndef __MakeLib
   bool ValidTime(const unsigned short hour, const unsigned short min, const unsigned short sec)
 #else
   bool __saveds __asm ValidTime(register __d0 const unsigned short hour, register __d1 const unsigned short min, register __d2 const unsigned short sec)
 #endif

/*
******* Date/ValidTime ******************************************************
*
*   NAME
*	ValidTime -- Checks if the time is a valid 24h-format time (V33.135)
*
*   SYNOPSIS
*	valid = ValidTime(hour,min,sec);
*	  d0		   d0   d1  d2
*
*	bool ValidTime(const unsigned short hour,
*	    const unsigned short min, const unsigned short sec);
*
*   FUNCTION
*	ValidTime checks if the time (24h format only) is valid.
*
*   INPUTS
*	hour - Hour of the time.
*	min  - Minute of the time.
*	sec  - Second of the time.
*
*   RESULT
*	valid - true  : The time is ok.
*		false : This is not a correct time!
*
*   EXAMPLE
*	...
*	if (ValidTime(25,0,0))
*	  printf("ok\n");
*	else
*	  printf("wrong time!!!\n");
*	...
*
*   NOTES
*	None.
*
*   BUGS
*	None.
*
*   SEE ALSO
*	ValidJulianDate(),ValidGregorianDate(),ValidHeisDate()
*
*****************************************************************************
*
*
*/

 {
  if ((hour < 24) && (min < 60) && (sec < 60))
    return(true);
  else
    return(false);
 }

 /* ----------------------------------------------------------------------- */

 #ifndef __MakeLib
   long TimeDiff(const unsigned short hour1, const unsigned short min1, const unsigned short sec1, const unsigned short hour2, const unsigned short min2, const unsigned short sec2)
 #else
   long __saveds __asm TimeDiff(register __d0 const unsigned short hour1, register __d1 const unsigned short min1, register __d2 const unsigned short sec1, register __d3 const unsigned short hour2, register __d4 const unsigned short min2, register __d5 const unsigned short sec2)
 #endif

/*
******* Date/TimeDiff *******************************************************
*
*   NAME
*	TimeDiff -- Returns the difference in seconds (V33)
*
*   SYNOPSIS
*	secs = TimeDiff(hour1,min1,sec1,hour2,min2,sec2);
*	 d0		 d0    d1   d2   d3    d4   d5
*
*	long TimeDiff(const unsigned short hour1, const unsigned short min1,
*	    const unsigned short sec1, const unsigned short hour2,
*	    const unsigned short min2, const unsigned short sec2);
*
*   FUNCTION
*	Gives you back the difference between the first and the second time
*	in seconds.
*
*   INPUTS
*	hour1 - hours of the first time
*	min1  - minutes of the first time
*	sec1  - seconds of the first time
*	hour2 - hours of the second time
*	min2  - minutes of the second time
*	sec2  - seconds of the second time
*
*   RESULT
*	secs - The difference betwen time1 and time1 in seconds.
*
*   EXAMPLE
*	...
*	secs = TimeDiff(21,15,00,22,0,0);
*	...
*
*   NOTES
*	Don't forget to convert AM/PM time to 24h time!
*	use SecToTime() to convert the seconds back to a hour,min,secs
*	format!
*
*   BUGS
*	No errorcheck, if you use a valid time
*
*   SEE ALSO
*	SecToTime(),TimeToSec()
*
*****************************************************************************
*
*
*/
  {
   return((long)(TimeToSec(hour1,min1,sec1)-TimeToSec(hour2,min2,sec2)));
  }

 /* ----------------------------------------------------------------------- */

 #ifndef __MakeLib
   #ifndef __cplusplus
     void DiffTime(const unsigned short hour, const unsigned short min, const unsigned short sec, long diffsecs, unsigned short *const rhour, unsigned short *const rmin, unsigned short *const rsec)
   #else
     void DiffTime(const unsigned short hour, const unsigned short min, const unsigned short sec, long diffsecs, unsigned short &rhour, unsigned short &rmin, unsigned short &rsec)
   #endif
 #else
   void __saveds __asm DiffTime(register __d0 const unsigned short hour, register __d1 const unsigned short min, register __d2 const unsigned short sec, register __d3 long diffsecs, register __a0 unsigned short *const rhour, register __a1 unsigned short *const rmin, register __a2 unsigned short *const rsec)
 #endif


/*
******* Date/DiffTime *******************************************************
*
*   NAME
*	DiffTime -- Returns the diff. time to another time. (V33)
*
*   SYNOPSIS
*	DiffTime(hour,min,sec,secs,rhour,rmin,rsec);
*		  d0  d1  d2  d3   a0    a1   a2
*
*	void DiffTime(const unsigned short hour, const unsigned short min,
*	    const unsigned short sec, long secs, unsigned short *const rhour,
*	    unsigned short *const rmin, unsigned short *const rsec);
*
*	void DiffTime(const unsigned short hour, const unsigned short min,
*	    const unsigned short sec, long secs, unsigned short &rhour,
*	    unsigned short &rmin, unsigned short &rsec);
*
*   FUNCTION
*	Returns the time which lies diffsecs before/after the specified time.
*
*   INPUTS
*	hour     - hour
*	min      - minute
*	sec      - second
*	diffsecs - difference to the time in seconds
*
*   RESULT
*	rhour - new hour
*	rmin  - new minute
*	rsec  - new second
*
*   EXAMPLE
*	...
*	DiffTime(12,19,0,2460,&hour,&min,&sec);
*	...
*
*   NOTES
*	Don't forget to convert AM/PM to 24h time!
*	Don't forget to convert 24h time to AM/PM time if needed!
*
*   BUGS
*	No errorcheck, if you use a valid time
*
*   SEE ALSO
*	TimeToSec(),SecToTime()
*
*****************************************************************************
*
*
*/

  {
   long secs;

   secs = (long)TimeToSec(hour,min,sec);
   secs += diffsecs;
   if (secs < 0)
    {
     secs += 86400L;
    }
   if (secs > 86400L)
    {
     secs -= 86400L;
    }
   SecToTime((unsigned long)secs,rhour,rmin,rsec);
  }

 /* ----------------------------------------------------------------------- */
