/* Copyright 1995-96 Jon Griffiths.  See the file "jlib.doc" for details. */

#include <stdlib.h>
#include <jlib.h>

/* private function */
void masked_copy(UBYTE *s,UBYTE *d, unsigned int len);

/*+------------------------------------------------------------------------+ */
/*|blit a portion of a buffer to another buffer without clipping           | */
/*+------------------------------------------------------------------------+ */
JINLINE void buff_blit_buff_toNC (buffer_rec * dbuff, int dbfx, int dbfy,
     buffer_rec * sbuff, int sbx1, int sby1, int sbx2, int sby2)
{
   int v_iterations, h_len;

   UBYTE *src = B_BUFF_PTR (sbuff), *dest = B_BUFF_PTR (dbuff);
   int sbuffx = B_X_SIZE (sbuff), dbuffx = B_X_SIZE (dbuff);

   JLIB_ENTER("buff_blit_buff_toNC");

   /* sort coordinates if needed */
   if (sbx2 < sbx1) {
      JLIB_SWAP(sbx1, sbx2);
   }
   if (sby2 < sby1) {
      JLIB_SWAP(sby1, sby2);
   }

   src += (sby1 *  sbuffx) + sbx1;
   dest += (dbfy * dbuffx) + dbfx;

   v_iterations = (sby2 - sby1)+1;        /* number of times to repeat horiz. blit */
   h_len = (sbx2 - sbx1)+1;       /* length of horizontal blit */

   switch (get_best_copy_method ((void *) src, (void *) dest, (unsigned int) h_len)) {

   case ALIGNMENT_STATE_1:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_1 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   case ALIGNMENT_STATE_2:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_2 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   case ALIGNMENT_STATE_3:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_3 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   case ALIGNMENT_STATE_4:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_4 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   case ALIGNMENT_STATE_5:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_5 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   case ALIGNMENT_STATE_6:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_6 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   case ALIGNMENT_STATE_7:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_7 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   case ALIGNMENT_STATE_8:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_8 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   default:
      jlib_exit("Unknown Copy Method.");
      break;
   }
 
 JLIB_LEAVE
}

/*+------------------------------------------------------------------------+ */
/*|blit a portion of a buffer to another buffer with clipping              | */
/*+------------------------------------------------------------------------+ */
JINLINE void buff_blit_buff_to (buffer_rec * dbuff, int dbfx, int dbfy,
     buffer_rec * sbuff, int sbx1, int sby1, int sbx2, int sby2)
{
   int v_iterations, h_len;
   UBYTE *src,*dest;
   int sbuffx,sbuffy,dbuffx,dbuffy;

   JLIB_ENTER("buff_blit_buff_to");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(sbuff);
   jlib_check_buffer(dbuff);
#endif

   src = B_BUFF_PTR (sbuff);
   dest = B_BUFF_PTR (dbuff);
   sbuffx = B_X_SIZE (sbuff);
   sbuffy = B_Y_SIZE(sbuff);
   dbuffx = B_X_SIZE (dbuff);
   dbuffy = B_Y_SIZE (dbuff);

   /* sort coordinates if needed */
   if (sbx2 < sbx1) {
      JLIB_SWAP(sbx1, sbx2);
   }
   if (sby2 < sby1) {
      JLIB_SWAP(sby1, sby2);
   }

   /* test dest coordinates for "don't draw" clipping */
   if ((dbfx >= dbuffx) || (dbfy >= dbuffy)) {
      JLIB_LEAVE;
      return;
   }

   if(dbfx < 0){
      sbx1 += (-dbfx);
      dbfx = 0;
      if(sbx1 > sbx2){
        JLIB_LEAVE;
	return;
      }
   }
   
   if(dbfy < 0){
      sby1 += (-dbfy);
      dbfy = 0;
      if(sby1 > sby2){
         JLIB_LEAVE;
	 return;
      }
   }

   /* test source coordinates for "don't draw" clipping */
   if ((sbx1 >= sbuffx) || (sby1 >= sbuffy)) {
      JLIB_LEAVE;
      return;
   }

   if ((sbx2 < 0) || (sby2 <0)) {
      JLIB_LEAVE;
      return;
   }

   /* clip source points to the source buffer */
   if(sbx1 < 0){
      sbx1 = 0;
   }
   if(sby1 < 0){
      sby1 = 0;
   }
   if(sbx2 >= sbuffx){
      sbx2 = sbuffx-1;
   }
   if(sby2 >= sbuffy){
      sby2 = sbuffy-1;
   }

   src += (sby1 * sbuffx) + sbx1;
   dest += (dbfy * dbuffx) + dbfx;

   v_iterations = (sby2 - sby1)+1; /* number of times to repeat horiz. blit */
   h_len = (sbx2 - sbx1)+1;        /* length of horizontal blit */

   /* test coordinates to clip area drawn */
   if (dbfx + h_len >= dbuffx) {
      h_len = dbuffx - dbfx;
   }
   if (dbfy + v_iterations >= dbuffy) {
      v_iterations = dbuffy - dbfy;
   }

   switch (get_best_copy_method ((void *) src, (void *) dest, (unsigned int) h_len)) {

   case ALIGNMENT_STATE_1:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_1 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   case ALIGNMENT_STATE_2:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_2 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   case ALIGNMENT_STATE_3:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_3 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   case ALIGNMENT_STATE_4:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_4 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   case ALIGNMENT_STATE_5:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_5 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   case ALIGNMENT_STATE_6:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_6 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   case ALIGNMENT_STATE_7:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_7 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   case ALIGNMENT_STATE_8:
      for (; v_iterations != 0; v_iterations--) {
	 MEM_COPY_ALIGN_8 (src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
      }
      break;

   default:
      jlib_exit("Unknown Copy Method.");
      break;
   }

   JLIB_LEAVE
}


/*+------------------------------------------------------------------------+ */
/*|blit a portion of an image to a buffer without clipping                 | */
/*+------------------------------------------------------------------------+ */
JINLINE void buff_blit_img_toNC (buffer_rec * dbuff, int dbfx, int dbfy,
	    image * img, int imx1, int imy1, int imx2, int imy2)
{
   JLIB_ENTER("buff_blit_img_toNC");

   /* do blit */
   buff_blit_buff_toNC (dbuff, dbfx, dbfy, img->buff, imx1, imy1, imx2, imy2);

   JLIB_LEAVE
}


/*+------------------------------------------------------------------------+ */
/*|blit a portion of an image to a buffer with clipping                    | */
/*+------------------------------------------------------------------------+ */
JINLINE void buff_blit_img_to (buffer_rec * dbuff, int dbfx, int dbfy,
	    image * img, int imx1, int imy1, int imx2, int imy2)
{
   JLIB_ENTER("buff_blit_img_to");

#ifndef JLIB_PRODUCTION
   if(img == NULL) {
      jlib_exit(jlib_msg(JLIB_ENULL));
   }
   jlib_check_buffer(dbuff);
#endif

   /* do blit */
   buff_blit_buff_to (dbuff, dbfx, dbfy, img->buff, imx1, imy1, imx2, imy2);

   JLIB_LEAVE
}




JINLINE void masked_copy(UBYTE *s,UBYTE *d, unsigned int len)
{
 while(len != 0){
    if(*s != 0){
       *d = *s; /* store non-zero pixels */
    }
    ++s;
    ++d;
    --len;
 }
}


/*+------------------------------------------------------------------------+ */
/*|blit a masked portion of a buffer to another buffer without clipping    | */
/*+------------------------------------------------------------------------+ */
JINLINE void buff_stencil_buff_toNC (buffer_rec * dbuff, int dbfx, int dbfy,
     buffer_rec * sbuff, int sbx1, int sby1, int sbx2, int sby2)
{
   int v_iterations,h_len;

   UBYTE *src = B_BUFF_PTR (sbuff), *dest = B_BUFF_PTR (dbuff);
   int sbuffx = B_X_SIZE (sbuff), dbuffx = B_X_SIZE (dbuff);

   JLIB_ENTER("buff_stencil_buff_toNC");

   /* sort coordinates if needed */
   if (sbx2 < sbx1) {
      JLIB_SWAP(sbx1, sbx2);
   }
   if (sby2 < sby1) {
      JLIB_SWAP(sby1, sby2);
   }

   src += (sby1 *  sbuffx) + sbx1;
   dest += (dbfy * dbuffx) + dbfx;

   v_iterations = (sby2 - sby1)+1; /* vertical length horiz. blit */
   h_len = (sbx2 - sbx1)+1;        /* length of horizontal blit */

   for(; v_iterations != 0; v_iterations--){
      masked_copy(src, dest, h_len);
      src += sbuffx;
      dest += dbuffx;
   }

   JLIB_LEAVE
}


/*+------------------------------------------------------------------------+ */
/*|blit a masked portion of a buffer to another buffer with clipping       | */
/*+------------------------------------------------------------------------+ */
JINLINE void buff_stencil_buff_to (buffer_rec * dbuff, int dbfx, int dbfy,
     buffer_rec * sbuff, int sbx1, int sby1, int sbx2, int sby2)
{
   int v_iterations, h_len;
   UBYTE *src,*dest;
   int sbuffx,sbuffy,dbuffx,dbuffy;

   JLIB_ENTER("buff_stencil_buff_to");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(sbuff);
   jlib_check_buffer(dbuff);
#endif

   src = B_BUFF_PTR (sbuff);
   dest = B_BUFF_PTR (dbuff);
   sbuffx = B_X_SIZE (sbuff);
   sbuffy = B_Y_SIZE(sbuff);
   dbuffx = B_X_SIZE (dbuff);
   dbuffy = B_Y_SIZE (dbuff);

   /* sort coordinates if needed */
   if (sbx2 < sbx1) {
      JLIB_SWAP(sbx1, sbx2);
   }
   if (sby2 < sby1) {
      JLIB_SWAP(sby1, sby2);
   }

   /* test dest coordinates for "don't draw" clipping */
   if ((dbfx >= dbuffx) || (dbfy >= dbuffy)) {
      JLIB_LEAVE;
      return;
   }

   if(dbfx < 0){
      sbx1 += (-dbfx);
      dbfx = 0;
      if(sbx1 > sbx2){
        JLIB_LEAVE;
	return;
      }
   }
   
   if(dbfy < 0){
      sby1 += (-dbfy);
      dbfy = 0;
      if(sby1 > sby2){
         JLIB_LEAVE;
	 return;
      }
   }


   /* test source coordinates for "don't draw" clipping */
   if ((sbx1 >= sbuffx) || (sby1 >= sbuffy)) {
      JLIB_LEAVE;
      return;
   }

   if ((sbx2 < 0) || (sby2 <0)) {
      JLIB_LEAVE;
      return;
   }

   /* clip source points to the source buffer */
   if(sbx1 < 0){
      sbx1 = 0;
   }
   if(sby1 < 0){
      sby1 = 0;
   }
   if(sbx2 >= sbuffx){
      sbx2 = sbuffx-1;
   }
   if(sby2 >= sbuffy){
      sby2 = sbuffy-1;
   }

   src += (sby1 * sbuffx) + sbx1;
   dest += (dbfy * dbuffx) + dbfx;

   v_iterations = (sby2 - sby1)+1; /* number of times to repeat horiz. blit */
   h_len = (sbx2 - sbx1)+1;        /* length of horizontal blit */

   /* test coordinates to clip area drawn */
   if (dbfx + h_len >= dbuffx) {
      h_len = dbuffx - dbfx;
   }
   if (dbfy + v_iterations >= dbuffy) {
      v_iterations = dbuffy - dbfy;
   }

   for (; v_iterations != 0; v_iterations--) {
	 masked_copy(src, dest, h_len);
	 src += sbuffx;
	 dest += dbuffx;
   }

   JLIB_LEAVE
}
