/***************************************************************

	map_dr.c        Map drawing routines
			for SFS Map utility

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023 
			Duke Station 
			Durham, NC  27706

			tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be 
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be 
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not 
	be used for illegal activities. 

***************************************************************/

#include "stdio.h"
#include "ctype.h"
#include "bw.h"
#include "gr.h"
#include "kb.h"
#include "ui.h"
#include "as.h"
#include "map.h"

#ifdef  __STDC__
#include "stdlib.h"
#endif

/***************************************************************

   dr_draw()

   This function draws the ground track screen.
   It is called by fo_draw().

***************************************************************/

dr_draw( uiwind, redraw_screen, s_start, s_end, pos )
   struct uiwindow *uiwind;
   int redraw_screen, pos;
   struct spj_pt *s_start, *s_end;
   {
   register int orbit;
   int x, y;

#ifdef  OLD_DEBUG
   sprintf( bw_ebuf, "dr_draw(): %d %d %d %d", uiwind->u_x1, uiwind->u_y1,
      uiwind->u_x2, uiwind->u_y2 );
   bw_debug( bw_ebuf );
#endif

   bw_message( MAP_DRAW );

   if ( redraw_screen == TRUE )
      {

      /* Prepare the screen */

      dr_prep( uiwind, pos );

      /* Draw the surface features */

      dr_plot( s_start, s_end, cl_surface, SOLID, pos );

     }

   }

/***************************************************************

   dr_prep()

   This routine draws the panel at the bottom of the screen
   which contains various parameters and is used when the
   ground track display is in effect.  The routine is called
   by dr_draw().

***************************************************************/

dr_prep( uiwind, pos )
   struct uiwindow *uiwind;
   int pos;
   {
   register int c;
   int inc;
   double x;
   int x_xmax, x_ymax;

   /***  Blank the entire window user area */

   ui_fbox( uiwind->u_x1, uiwind->u_y1,
      uiwind->u_x2, uiwind->u_y2, BLACK, SOLID );

   /***  First determine if x or y axis must be scaled */

   x_xmax = uiwind->u_x2 - uiwind->u_x1;
   x_ymax = uiwind->u_y2 - uiwind->u_y1;

   if ( ( x_xmax / 2 ) >
	( ( x_ymax * gr_pysize ) / gr_pxsize )
      )

      {                              /* TRUE = scale x */

#ifdef  OLD_DEBUG
      bw_debug( "Scaling x" );
#endif

      dr_ysize = x_ymax;
      dr_xsize = 2 * (( dr_ysize * gr_pysize ) / gr_pxsize );

      /***  Calculate dr_mapedge */

      dr_mapedge = uiwind->u_x1 +
	 ( ( x_xmax - dr_xsize ) / 2 );

      /***  Calculate dr_mapbase */

      dr_mapbase = uiwind->u_y1;

      }

   else                             /* FALSE = scale y */
      {

#ifdef  OLD_DEBUG
      bw_debug( "Scaling y" );
#endif

      dr_xsize = x_xmax;
      dr_ysize = (( dr_xsize * gr_pxsize ) / gr_pysize ) / 2;

      /***  Calculate dr_mapedge */

      dr_mapedge = uiwind->u_x1;

      /***  Calculate dr_mapbase */

      dr_mapbase = uiwind->u_y1 +
	 ( ( x_ymax - dr_ysize ) / 2 );

      }

   /***  Draw lines demarcating the display within the window */

   gr_line( GR_PRIMARY, dr_mapedge, dr_mapbase, dr_mapedge,
      dr_mapbase + dr_ysize, cl_grid, SOLID );

   gr_line( GR_PRIMARY, dr_mapedge + dr_xsize, dr_mapbase,
      dr_mapedge + dr_xsize, dr_mapbase + dr_ysize,
      cl_grid, SOLID );

   gr_line( GR_PRIMARY, dr_mapedge, dr_mapbase,
      dr_mapedge + dr_xsize,
      dr_mapbase, cl_grid, SOLID );

   gr_line( GR_PRIMARY, dr_mapedge, dr_mapbase + dr_ysize,
      dr_mapedge + dr_xsize, dr_mapbase + dr_ysize,
      cl_grid, SOLID );

#ifdef  OLD_DEBUG
   sprintf( bw_ebuf, "x_size %d, y_size %d, mapedge %d, mapbase %d",
      dr_xsize, dr_ysize, dr_mapedge, dr_mapbase );
   bw_debug( bw_ebuf );
#endif

   /***  Now draw the latitude and longitude lines. We must draw
	 only a quarter of a line at a time, or dr_line will
	 reject the request. */

   for ( x = -75.0; x < 90.0; x += 15.0 )               /* latitudes */
      {
      dr_line( x, -180.0, x, -90.0, cl_grid, GRID, pos );
      dr_line( x,  -90.0, x,   0.0, cl_grid, GRID, pos );
      dr_line( x,    0.0, x,  90.0, cl_grid, GRID, pos );
      dr_line( x,   90.0, x, 180.0, cl_grid, GRID, pos );
      }

   for ( x = -165.0; x < 180.0; x += 15.0 )             /* longitudes */
      {
      dr_line( -90.0, x, -45.0, x, cl_grid, GRID, pos );
      dr_line( -45.0, x,   0.0, x, cl_grid, GRID, pos );
      dr_line(   0.0, x,  45.0, x, cl_grid, GRID, pos );
      dr_line(  45.0, x,  90.0, x, cl_grid, GRID, pos );
      }

#ifdef  OLD_DEBUG
   bw_debug( "end of dr_prep() " );
#endif

   }

/***************************************************************

   dr_plot()

   This routine plots a set of coordinates on the ground
   track map.  It is called by dr_draw().

***************************************************************/

dr_plot( start, end, color, style, pos )
   struct spj_pt *start, *end;
   int color, style, pos;
   {
   struct spj_pt *current;
   double prevx, prevy;

   current = start->next;

#ifdef  OLD_DEBUG
   bw_debug( "DEBUG:  Plotting" );
#endif

   prevx = current->longitude;
   prevy = current->latitude;
   while ( current != end )
      {
      if ( current->code < 1000 )
	 {
	 dr_line( current->latitude, current->longitude,
	    prevy, prevx, color, style, pos );
	 }
      prevx = current->longitude;
      prevy = current->latitude;
      current = current->next;
      }

#ifdef  OLD_DEBUG
   bw_message( " " );
#endif
   }

/***************************************************************

   dr_line()

   This routine draws a single line on the ground
   track map.

***************************************************************/

dr_line( fromlat, fromlon, tolat, tolon, color, style, pos )
   double fromlat, fromlon, tolat, tolon;
   int color, style, pos;
   {
   int x1, y1, x2, y2;

   if ( ( fromlat < map_pos[ pos ][ 1 ] ) || ( fromlat > map_pos[ pos ][ 3 ] ))
      {
      /* sprintf( bw_ebuf, "dr_line received from latitude %.2lf", fromlat );
      bw_error( bw_ebuf ); */
      return BW_ERROR;
      }
   if ( ( tolat < map_pos[ pos ][ 1 ] ) || ( tolat > map_pos[ pos ][ 3 ] ))
      {
      /* sprintf( bw_ebuf, "dr_line received to latitude %.2lf", tolat );
      bw_error( bw_ebuf ); */
      return BW_ERROR;
      }
   if ( ( fromlon < map_pos[ pos ][ 0 ] ) || ( fromlon > map_pos[ pos ][ 2 ] ))
      {
      /* sprintf( bw_ebuf, "dr_line received from longitude %.2lf", fromlon );
      bw_error( bw_ebuf ); */
      return BW_ERROR;
      }
   if ( ( tolon < map_pos[ pos ][ 0 ] ) || ( tolon > map_pos[ pos ][ 2 ] ))
      {
      /* sprintf( bw_ebuf, "dr_line received to longitude %.2lf", tolon );
      bw_error( bw_ebuf ); */
      return BW_ERROR;
      }

#ifdef  OLD_DEBUG
   sprintf( bw_ebuf, "DEBUG from x = %.2lf, from y = %.2lf, to x = %.2lf, to y = %.2lf, color = %d, style = %d,",
      fromlon,
      fromlat,
      tolon,
      tolat,
      color, style );
   bw_debug( bw_ebuf );
#endif

   x1 = (( fromlon - (double) ( map_pos[ pos ][ 0 ] ) )
      * ( (double) dr_xsize / ( map_pos[ pos ][ 2 ] - (double) map_pos[ pos ][ 0 ] ) ));
   x2 = ((   tolon - (double) ( map_pos[ pos ][ 0 ] ) )
      * ( (double) dr_xsize / ( map_pos[ pos ][ 2 ] - (double) map_pos[ pos ][ 0 ] ) ));

   if ( abs( x1 - x2 ) > ( ( dr_xsize / 2 ) + 3 ) )
      {
#ifdef  OLD_DEBUG
      sprintf( sfs_tbuf, "DEBUG: dr_line() failed: x1 = %d, x2 = %d",
	 x1, x2 );
      bw_debug( sfs_tbuf );
      sprintf( sfs_tbuf, "DEBUG: dr_line() #2: abs() = %d, dr_xsize = %d, dr_xsize / 3 = %d",
	 abs( x1 - x2), dr_xsize, (dr_xsize / 3) );
      bw_debug( sfs_tbuf );
#endif
      return;
      }

   y1 = (( fromlat - (double) ( map_pos[ pos ][ 1 ] ))
      * ( (double) dr_ysize / ( map_pos[ pos ][ 3 ] - (double) map_pos[ pos ][ 1 ] ) ));
   y2 = ((   tolat - (double) ( map_pos[ pos ][ 1 ]  ))
      * ( (double) dr_ysize / ( map_pos[ pos ][ 3 ] - (double) map_pos[ pos ][ 1 ] ) ));

   if ( abs( y1 - y2 ) > ( ( dr_ysize / 2 ) + 3 ) )
      {
#ifdef  OLD_DEBUG
      sprintf( sfs_tbuf, "dr_line() failed: y1 = %d, y2 = %d",
	 y1, y2 );
      bw_debug( sfs_tbuf );
#endif
      return;
      }

#ifdef  OLD_DEBUG
   sprintf( sfs_tbuf, "DEBUG x1 = %d, y1 = %d, x2 = %d, y2 = %d, color = %d, style = %d,",
      dr_mapedge + x1,
      dr_mapbase + y1,
      dr_mapedge + x2,
      dr_mapbase + y2,
      color, style );
   bw_debug( sfs_tbuf );
#endif

   gr_line( GR_PRIMARY,
      dr_mapedge + x1,
      dr_mapbase + y1,
      dr_mapedge + x2,
      dr_mapbase + y2,
      color, style );
   }

