program Voxels;
{****************************************************************************
** Demonstration of displaying a voxel heightscape                         **
**  by Steven H Don                                                        **
**                                                                         **
** For questions, feel free to e-mail me.                                  **
**                                                                         **
**    shd@earthling.net                                                    **
**    http://shd.home.ml.org                                               **
**                                                                         **
****************************************************************************}

uses Crt, Dos;

const
  {This controls the maximum distance displayed. If you increase this,
  you will see more of the landscape, but it will also be slower}
  Depth = $7F;

type
  {We can only allocate 64K-1 bytes of memory, not 64K. So, for the extra
  ONE BYTE, we need to split up into two parts of 32K!!!!}
  ThirtyTwo = Array [0..32768] of Byte;
  SixtyFour = Array [0..1] of ^ThirtyTwo;
  {This is an off-screen frame buffer}
  Frame = Array [0..63999] of Byte;

var
  {Height map}
  HMap : SixtyFour;
  {Texture map}
  TMap : SixtyFour;
  {The buffer}
  Buffer : ^Frame;
  {The screen itself}
  Screen : Array [0..63999] of Byte absolute $A000:0000;
  {Cosine and Sine tables}
  CosT, SinT : Array [0..2047] of Integer;
  {Distance compensation table}
  DComp : Array [1..Depth + 1] of Integer;
  {Camera information}
  x, y, Angle : Word;
  Height : Byte;
  {Keyboard input}
  Key : Char;

{This sets the display to VGA 320x200 in 256 colours}
procedure VGAScreen; assembler;
asm
  mov ax, $13
  int $10
end;

{This resets the display to text mode}
procedure TextScreen; assembler;
asm
  mov ax, $3
  int $10
end;

{This sets a DAC register to a specific Red Green Blue-value}
procedure SetDAC (DAC, R, G, B : Byte);
begin
  Port [$3C8] := DAC;
  Port [$3C9] := R;
  Port [$3C9] := G;
  Port [$3C9] := B;
end;

{Draws a vertical line down the screen}
procedure DrawVLine (x1, y1, y2 : Integer; c : Byte);
var
  o : Word;

begin
  {o := y1 * 320 + x;}
  o := y1 shl 8 + y1 shl 6 + x1;
  {For every pixel along the line}
  for x1 := y1 to y2 do begin
    {Set pixel}
    Buffer^[o] := c;
    {Move down one pixel}
    inc (o, 320);
  end;
end;

{Casts a ray from specified position and renders result}
procedure Ray (a, x, y, sX : Word);
var
  deltax, deltay, p, o, d : Word;
  minY : Integer;
  h, y1 : LongInt;

begin
  deltax := CosT [a];
  deltay := SinT [a];
  minY := 200;
  d := 0;
  repeat
    inc (x, deltax);
    inc (y, deltay);
    {New distance}
    inc (d, 1);
    {Calculate offset into map}
    p := y shr 15;
    o := (y and $7F00) + hi (x);
    h := HMap [p]^[o] - Height;
    {Calculate height}
    y1 := DComp [d] - (h shl 5) div d;
    if y1 < minY then begin
      {Draw voxel}
      DrawVLine (sX, y1, minY - 1, TMap [p]^[o]);
      minY := y1;
    end;
  until d >= Depth;
end;

procedure DrawView;
var
  a, i : Integer;
  dd : Integer;

begin
  {Fill every column on-screen}
  for i := 0 to 319 do begin
    {Calculate ray angle depending on view angle}
    a := (Angle + i + 1888) and 2047;
    {Cast the ray}
    Ray (a, x, y, i);
  end;
end;

procedure InitMap;
var
  DAC, Red, Green, Blue : Byte;
  f : File;

begin
  {Assign 64K for a 256x256 texture map}
  New (TMap [0]);
  New (TMap [1]);
  {Assign 64K for a 256x256 height map}
  New (HMap [0]);
  New (HMap [1]);

  {Load in the file "GROUND.BMP"}
  Assign(f, 'GROUND.BMP');
  Reset(f, 1);
  {The palette starts at byte 54}
  Seek (f, 54);

  {Read palette}
  for DAC := 0 to 255 do begin
    {.BMP format saves 4 bytes per colour}
    BlockRead (f, TMap[0]^, 4);
    {In reverse order, and in the range of 0..255 instead of 0..63}
    Blue  := TMap [0]^[0] shr 2;
    Green := TMap [0]^[1] shr 2;
    Red   := TMap [0]^[2] shr 2;
    {Set the palette}
    SetDAC (DAC, Red, Green, Blue);
  end;

  {Load in both parts of the texture map}
  BlockRead(f, TMap [0]^, 32768);
  BlockRead(f, TMap [1]^, 32768);
  {And close the file}
  Close (f);

  {Load in the file "HEIGHT.BMP"}
  Assign(f, 'HEIGHT.BMP');
  Reset(f, 1);
  {Skip the palette for this one}
  Seek (f, 1078);
  BlockRead(f, HMap [0]^, 32768);
  BlockRead(f, HMap [1]^, 32768);
  Close (f);
end;

{This procedure calculates some lookup table}
procedure InitTables;
var
  a : Word;

begin
  for a := 0 to 2047 do begin
    {Precalculate cosine}
    CosT [a] := trunc(Cos (a * pi / 1024) * 256);
    {and sine}
    SinT [a] := trunc(Sin (a * pi / 1024) * 256);
  end;
  {Precalculate distance compensation table}
  for a := 1 to Depth + 1 do DComp [a] := 1000 div longint(a) + 100;
end;

begin
  {Precalculate sine and cosine}
  InitTables;
  {Set up player}
  x := $8000;
  y := $8000;
  Angle := 600;
  {Switch to graphics mode}
  VGAScreen;
  New (Buffer);
  {Create the map}
  InitMap;

  repeat

    {Adjust camera height according to height map}
    Height := HMap [y shr 15]^[(y and $7F00) + hi (x)];

    {Clear the buffer}
    fillChar (Buffer^, 64000, 0);
    {Draw a screen}
    DrawView;
    {Copy the buffer to the screen}
    move (Buffer^, Screen, 64000);

    {Get keyboard input}
    Key := ReadKey;

    if Key = Chr (0) then case Ord (ReadKey) of
      72 : begin {move forward}
             inc (x, CosT [Angle]);
             inc (y, SinT [Angle]);
           end;
      75 : begin {turn left}
             Angle := (Angle + 2032) and 2047;
           end;
      77 : begin {turn right}
             Angle := (Angle + 16) and 2047;
           end;
      80 : begin {move backward}
             dec (x, CosT [Angle]);
             dec (y, SinT [Angle]);
           end;
    end;

  until Key = Chr (27);
  {Switch back to text}
  TextScreen;
end.