#include <stdlib.h>
#include <stdio.h>
#include <io.h>
#include <fcntl.h>

#define BUF_SIZE 8192			// needs to be a power of 2

#define TRUE 1				// yes, affirmative, true
#define FALSE 0				// no, negative, false

#define addr(x) (&(x))			// address-of operation
typedef unsigned char BYTE;		// for an unsigned character
typedef unsigned int WORD;		// for an unsigned 16-bit word

//    Extension ROM Checksummer
//    -------------------------
//
//    Calling sequence is:
//
//    romxsum infile outfile
//

void main( int argc, char *argv[])
{

  int
    ihand,	    // handle for input
    ohand,	    // handle for output
    partial,	    // partial buffer amount
    i;		    // scratch

  BYTE
    sum,	    // checksum thus far
    old_sum,	    // last checksum
    *buf;	    // our buffer pointer

  long
    curpos,	    // current position in the file
    rom_size,	    // how big is the ROM code?
    total_size;	    // how big is the file

//  Grab a buffer and check the command-line arguments.

  if ( !(buf = malloc( BUF_SIZE)) )
  {
    fprintf( stderr, "Insufficient memory to run -- need about 48K\n");
    exit(1);
  }

  if (argc != 3)
  {
    fprintf( stderr, "Syntax is - %s infile outfile\n", argv[0]);
    exit(1);
  }

//  Open and create the output file.

  if ( ( ihand = open( argv[1], O_RDONLY+O_BINARY)) < 0)
  {
    fprintf( stderr, "Unable to open %s\n", argv[1]);
    exit(1);
  }

  if ( ( ohand = open( argv[2], O_CREAT+O_WRONLY+O_BINARY)) < 0)
  {
    fprintf( stderr, "Could not create %s\n", argv[2]);
    exit(1);
  }

//  Figure the size of the ROM file, as well as the size of the ROM code
//  itself.  Note that we copy the difference anyway.

  read( ihand, buf, 3);			// pick up the signature and length
  if ( (buf[0] != 0x55) || (buf[1] != 0xaa))
  {
    fprintf( stderr, "The ROM doesn\'t start with 55 AA!\n");
    close( ihand);
    close( ohand);
    exit(1);
  }
  rom_size = (long) (buf[2]) << 9;	  // get total ROM size
  lseek( ihand, rom_size-1L, SEEK_SET); // get to the checksum
  read( ihand, addr(old_sum), 1);	// read the old checksum
  total_size = lseek( ihand, 0L, SEEK_END); // get total file size
  lseek( ihand, 0L, SEEK_SET);	     // rewind it

  printf( "For the ROM extension file %s:\n\n"
	  "    Active ROM size:      %ld bytes\n"
	  "    Total file size:      %ld bytes\n"
	  "    Old checksum:         %02x (hex)\n",
	  argv[1], rom_size, total_size, old_sum);

//  Check for sizing errors.

  if ( total_size < rom_size)
  {
    printf( "\n ** WARNING ** ROM size is larger than actual file size.\n"
	    "               Will use file size as ROM size.\n");
    rom_size = total_size;
  }

  sum = 0;
  curpos = 0L;
  while (TRUE)
  {
    if ( (rom_size - curpos) > BUF_SIZE)
    { // a full buffer
      read( ihand, buf, BUF_SIZE);	// read/write a page
      write( ohand, buf, BUF_SIZE);

      for (  i = 0; i < BUF_SIZE; i++)
	sum += buf[i];

      curpos += BUF_SIZE;
    }
    else
    { // end of the file--partial buffer
      partial = (int) (rom_size - curpos) -1;
      read( ihand, buf, partial);
      write( ohand, buf, partial);	// read and write what's left
      for (  i = 0; i < partial; i++)
	sum += buf[i];			// figure the checksum on the remnant

      sum = 0 - sum;			// complement the checksum
      write( ohand, addr(sum), 1);	// write it out

      printf("    New checksum:         %02x (hex)\n", sum);

//  Now, copy the rest of the ROM area out.

      read( ihand, buf, 1);		// skip the old sum

      curpos += (long) (partial +1);
      while( curpos < total_size)
      { // keep filling
	partial = (total_size - curpos) > BUF_SIZE ? BUF_SIZE :
		  (int) (total_size - curpos);
	read( ihand, buf, partial);
	write( ohand, buf, partial);
	curpos += (long) partial;
      }
      break;

    } // end of the file
  } // while

  printf( "\n\nAll done!  File %s written.\n", argv[2]);

//  Close the files and quit.

  close( ihand);
  close( ohand);
  exit( 0);
} // main
