/**********************************************************************/
/*                               E G A C                              */
/*--------------------------------------------------------------------*/
/*    Description    : demonstrates the operation of the EGA-BIOS     */
/*                     functions.                                     */
/*--------------------------------------------------------------------*/
/*    Author         : MICHAEL TISCHER                                */
/*    developed on   : 08/30/1988                                     */
/*    last update    : 08/30/1988                                     */
/*--------------------------------------------------------------------*/
/*    (MICROSOFT C)                                                   */
/*    Creation       : CL /AS /c EGAC.C                               */
/*                     LINK EGAC EGACA;                               */
/*    Call           : EGAC                                           */
/*--------------------------------------------------------------------*/
/*    (BORLAND TURBO C)                                               */
/*    Before compiling/linking:                                       */
/*    1)  Set the following parameters in the Turbo C Options menu:   */
/*                   Compiler options - set Model option to Medium    */
/*                   Linker options - set Case-sensitive link to Off  */
/*    2)  Create project file made of the following lines:            */
/*                   EGAC                                             */
/*                   EGACA.OBJ                                        */
/**********************************************************************/

/*== Include files ===================================================*/

#include <dos.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <bios.h>

/*== Typedefs ========================================================*/

typedef unsigned char BYTE;                  /* make ourselves a byte */
typedef unsigned int WORD;
typedef BYTE BOOL;                          /* like BOOLEAN in Pascal */
typedef struct velb far * VP;    /* VP = FAR pointer in the Video-RAM */

/*== Definition of the function from the assembly language module ====*/

extern void chardef( BYTE ascii, BYTE table, BYTE lines,
                     BYTE number, BYTE far * buf );

/*== Structures ======================================================*/

struct velb {               /* describes a screen position as 2 bytes */
             BYTE character,                          /* the
ASCII code */
                  attribute/*attribute*/;                              /*
attribute */
            };

/*== Macros ==========================================================*/

/*-- MK_FP creates a FAR pointer to an object out of segment and -----*/
/*-- offset addresses                                            -----*/


#ifndef MK_FP                                  /* is MK_FP undefined? */
#define MK_FP(seg, ofs) ((void far *) ((unsigned long) (seg)<<16|(ofs)))
#endif

#define VOFS(x,y) ( 80 * ( y ) + ( x ) )   /* offset in the video RAM */
#define VPOS(x,y) (VP) ( vptr + VOFS( x, y ) )     /* pointer in VRAM */
#define GETCZ()   (vline)         /* returns the current cursor line */
#define GETCS()   (vcolumn)             /* returns the current column */

/*== Constants =======================================================*/

#define TRUE  1                    /* constants for working with BOOL */
#define FALSE 0

#define VIDEO_INT 0x10                        /* BIOS video interrupt */
#define LINE25 25                                   /* 25 line screen */
#define LINE43 43                                   /* 43 line screen */

#define MONO  0                           /* monitor types for GETMON */
#define COLOR 1
#define EGA   2

/*== Global variables ================================================*/

VP vptr;               /* pointer to the first character in video RAM */
BYTE vline/*vline*/,                    /* stores the current cursor
position */
     vcolumn/*vcolumn*/,
     anzline/*numline*/;                               /* number of
screen lines */
BOOL mono;                /* TRUE if a monochrome monitor is attached */

/***********************************************************************
*  Function         : C E M U L                                        *
**--------------------------------------------------------------------**
*  Aufgabe          : Switches the cursor emulation of the EGA card    *
*                     on or off.                                       *
*  Input parameters : - DOIT = TRUE : Emulation on.                    *
*                              FALSE: Emulation off                    *
*  Return value     : none                                             *
***********************************************************************/

void cemul( BOOL doit )

{
 /*-- Definition of the video-info byte at offset 0x87 within the    --
*/
 /*-- BIOS variable segment -------------------------------------------
*/

 #define VIO_INFO_BYTE ((BYTE far *) MK_FP(0x40, 0x87))

 if ( doit )                             /* Turn on cursor emulation? */
  *VIO_INFO_BYTE |= 1;                             /* YES, set bit 0  */
 else                                                           /* NO */
  *VIO_INFO_BYTE &= 254;                               /* clear bit 0 */
}


/***********************************************************************
*  Function         : G E T M O N                                      *
**--------------------------------------------------------------------**
*  Description      : Determines the type of monitor connected.        *
*  Input parameters : none                                             *
*  Return value     : the monitor type                                 *
*                       MONO = monochrome monitor                      *
*                       COLOR= color monitor                           *
*                       EGA  = EGA or Multisync monitor                *
***********************************************************************/

BYTE getmon()
 {
  union REGS regs;          /* processor registers for interrupt call */

  regs.h.ah = 0x12;       /* Function number: Determine configuration */
  regs.h.bl = 0x10;                             /* Subfunction number */
  int86(VIDEO_INT, &regs, &regs);        /* call BIOS video interrupt */
  if ( regs.h.cl == 0x0B )                      /* monchrome monitor? */
   return( MONO );                                             /* YES */
  if ( regs.h.cl == 0x08 )                          /* color monitor? */
   return( COLOR );                                            /* YES */
  else                                             /* no, must be EGA */
   return( EGA );
 }

/***********************************************************************
*  Function         : S E T C U R                                      *
**--------------------------------------------------------------------**
*  Description      : Sets the bilnking cursor screen cursor and the   *
*                     internal output position.                        *
*  Input parameters : - COLUMN   = the cursor column.                  *
*                     - LINE     = the cursor line.                    *
*  Return value     : none                                             *
***********************************************************************/

void setcur(BYTE column/*column*/, BYTE line/*line*/)
 {
  union REGS regs;          /* processor registers for interrupt call */

  regs.h.ah = 2;                                   /* function number */
  regs.h.bh = 0;                              /* access screen page 0 */
  regs.h.dh = vline = line;     /* save coordinates in global vars. */
  regs.h.dl = vcolumn = column;
  int86(VIDEO_INT, &regs, &regs);        /* call BIOS video interrupt */
 }

/***********************************************************************
*  Function         : S E T C O L                                      *
**--------------------------------------------------------------------**
*  Description      : Defines the contents of one of the 15 color      *
*                     registers on the EGA card.                       *
*  Input parameters : - REGNR = Color register number                  *
*                     - COLOR = Color value (0 - 15)                   *
*  Return values    : none                                             *
***********************************************************************/
void setcol(BYTE regnr, BYTE color)

{
 union REGS regs;           /* processor registers for interrupt call */

 regs.h.ah = 0x10;                 /* Ftn. no.: set colors/attributes */
 regs.h.al = 0;                                      /* subfunction 0 */
 regs.h.bl = regnr;                            /* set register number */
 regs.h.bh = color & 63;    /* Set color value (combine bits 6 and 7) */
 int86(VIDEO_INT, &regs, &regs);     /* BIOS-Video-Interrupt call     */
}

/***********************************************************************
*  Function         : S E T B O R D E R                                *
**--------------------------------------------------------------------**
*  Description      : Sets the color of the screen borders.            *
*  Input parameters : - COLOR = color value (0 to 15)                  *
*  Return value     : none                                             *
***********************************************************************/

void setborder( BYTE color )

{
 union REGS regs;           /* processor registers for interrupt call */

 regs.h.ah = 0x10;                 /* Ftn. no.: set colors/attributes */
 regs.h.al = 1;                                      /* subfunction 1 */
 regs.h.bh = color & 15;                           /* set color value */
 int86(VIDEO_INT, &regs, &regs);         /* call BIOS video interrupt */
}

/***********************************************************************
*  Function         : S E T L I N E S                                  *
**--------------------------------------------------------------------**
*  Description      : Sets the number of screen lines.                 *
*  Input parameters : - LINES = one of the two constants:              *
*                                LINE25 = 25 lines                     *
*                                LINE43 = 43 lines                     *
*  Return value     : none                                             *
***********************************************************************/

void setlines( BYTE lines )

{
 union REGS regs;           /* processor registers for interrupt call */

 regs.h.ah = 0x11;                   /* Ftn. no.: Character generator */
 regs.h.al = ((anzline = lines) == LINE25) ? 0x11 : 0x12;  /* subftn. */
 regs.h.bl = 0;                              /* use character table 0 */
 int86(VIDEO_INT, &regs, &regs);         /* call BIOS video interrupt */
}

/***********************************************************************
*  Function         : I S _ E G A                                      *
**--------------------------------------------------------------------**
*  Description      : Determines if an EGA card is installed.          *
*  Input paramters  : none                                             *
*  Return value     : TRUE, if an EGA or VGA card is installed, else   *
*                     FALSE.                                           *
***********************************************************************/

BOOL is_ega()
 {
  union REGS regs;          /* processor registers for interrupt call */

  regs.h.ah = 0x12;        /* Ftn. no.: determine video configuration */
  regs.h.bl = 0x10;                             /* subfunction number */
  int86(VIDEO_INT, &regs, &regs);       /* call BIOS video interrupt  */
  if ( regs.h.bl != 0x10 )                        /* EGA or VGA card? */
   {                                                           /* Yes */
    /*-- set pointer in VRAM according to the monitor being used    --*/
    vptr = (VP) MK_FP( (mono = regs.h.bh) ? 0xb000 : 0xb800, 0);
    setlines( LINE25 );                           /* set 25-line mode */
    setcur(0, 0);                  /* cursor in the upper left corner */

    /*-- Install EGA and VGA hardcopy routine ------------------------*/
    regs.h.ah = 0x12;               /* Function no.: Alternate Select */
    regs.h.bl = 0x20;     /* subftn.. 0x20 = install hardcopy routine */
    int86(VIDEO_INT, &regs, &regs);      /* call BIOS video interrupt */
   }
  return( regs.h.bl != 0x10 );           /* BL != 0x10 --> EGA or VGA */
 }

/***********************************************************************
*  Function         : P R I N T A T                                    *
**--------------------------------------------------------------------**
*  Description      : Output a string to the screen.                   *
*                                                                      *
*  Input parameters : - COLUMN   = the cursor column.                  *
*                     - LINE     = the output line.                    *
*                     - COLOR    = attribute for the character.        *
*                     - STRING   = pointer to the string.              *
*  Return value     : none                                             *
*  Information      : - Format characters as with printf are not       *
*                       recognized by this function.                   *
*                     - The screen is not scrolled when the end is     *
*                       reached.                                       *
***********************************************************************/

void printat(BYTE column, BYTE line, BYTE color, char * string)

 {
  register VP lptr;               /* running pointer to the video RAM */
  register BYTE i;                 /* counts the number of characters */
  unsigned neuofs;            /* used to calculate the new coordinate */

  lptr = VPOS(column, line);              /* set pointer to the VRAM */
  for (i=0 ; *string ; ++lptr, ++i)         /* run through the string */
   {
    lptr->character = *(string++);    /* the character in the video RAM */
    lptr->attribute = color;        /* set attribute for the character */
   }
  /*-- calculate new cursor coordinates ------------------------------*/

  vcolumn = (neuofs = ((unsigned) line * 80 + column + i)) % 80;
  vline = neuofs / 80;
 }

/***********************************************************************
*  Function         : P R I N T F A T                                  *
**--------------------------------------------------------------------**
*  Description      : Outputs a string like printf, but writes it      *
*                     directly to the video RAM.                       *
*  Input parameters : - COLUMN = the output column.                    *
*                     - LINE   = the output line.                      *
*                     - COLOR  = color of the characters.              *
*                     - STRING = pointer to the String.                *
*                     - ...    = the additional arguments.             *
*  Return value     : none                                             *
*  Information      : - The screen is not scrolled when the end is     *
*                       reached.                                       *
*                   : - The string can contain formatting characters   *
*                       as with printf.                                *
***********************************************************************/

void printfat(BYTE column, BYTE line, BYTE color, char * string,...)

 {
  va_list parameter;        /* stores parameter list for VA_... macros */
  char output[255];             /* the formatted string to be printed */

  va_start(parameter, string);  /* get parameters in the var PARAMETER */
  vsprintf(output, string, parameter);              /* convert string */
  printat(column, line, color, output);         /* output the string */
 }

/***********************************************************************
*  Function         : B L I N K I N G                                  *
**--------------------------------------------------------------------**
*  Description      : Defines the meaning of bit 7 in the attribute    *
*                     byte of the character in the text modes.         *
*  Input parameters : DOBLINK = TRUE  : blink.                         *
*                               FALSE : light background.              *
*  Return value     : none                                             *
***********************************************************************/

void blinking( BOOL doblink )
 {
  union REGS regs;          /* processor registers for interrupt call */

  regs.h.ah = 0x10;                /* Ftn. no.: set colors/attributes */
  regs.h.al = 0x3;                              /* subfunction number */
  regs.h.bl = doblink ? 1 : 0;                      /* BL = 1 : blink */
  int86(VIDEO_INT, &regs, &regs);        /* call BIOS video interrupt */
 }


/***********************************************************************
*  Function         : C L S                                            *
**--------------------------------------------------------------------**
*  Description      : Clears the screen, causing the video mode to be  *
*                     reset. (The contents of the palette registers    *
*                     and the character set will be reset.)            *
*  Input parameters : none                                             *
*  Return value     : none                                             *
***********************************************************************/

void cls()
 {
  union REGS regs;          /* processor registers for interrupt call */

  regs.h.ah = 0x0;                        /* Ftn. no.: set video mode */
  regs.h.al = ( mono ) ? 7 : 3;          /* 80*25 character text mode */
  int86(VIDEO_INT, &regs, &regs);        /* call BIOS video interrupt */
 }

/***********************************************************************
*  Function         : N O K E Y                                        *
**--------------------------------------------------------------------**
*  Description      : Tests to see if a key has been pressed.          *
*  Input parameters : none                                             *
*  Return value     : TRUE, if a key was pressed, else FALSE.          *
***********************************************************************/

BOOL nokey()

{
#ifdef __TURBOC__                                   /* using TURBO-C? */
 return( bioskey( 1 ) == 0 );          /* Yes, read keyboard via BIOS */
#else                                            /* using Microsoft C */
 return( _bios_keybrd( _KEYBRD_READY ) == 0 );       /* test via BIOS */
#endif
}

/***********************************************************************
*  Function         : E G A B I O S                                    *
**--------------------------------------------------------------------**
*  Description      : Demonstrates the special functions of the EGA    *
*                     BIOS.                                            *
*  Input parameters : none                                             *
*  Return value     : none                                            *
***********************************************************************/

void egabios()
 {
  static BYTE font[120][14] = {    /* Character definitions for LOGO */
   {  0,  0,255, 62, 28, 28, 28, 28, 28, 28, 28, 28, 28, 31},   /* E */
   {  0,  0,252,  7,  1,  1,  1,  1,  1,  1,  1,  1,  7,252},   /* A */
   {  0,  0,  0,  0,129,195,195,199,199,206,206,142, 14, 14},   /* C */
   {  0,  0, 62,193,128,128,  0,  0,  0,  0,  0,  0,  0,  0},   /* H */
   {  0,  0, 16,144,112, 48, 48, 16, 16,  0,  0,  0,  0,  0},   /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},   /* L */
   {  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},   /* I */
   {  0,  0,254,248,112,112,112,112,112,112,112,112,112,112},   /* N */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,252, 61, 30, 30, 28},   /* E */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,248,  6,  7,  3,  3},   /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  7,  0,  0,  0,128},   /* C */
   {  0,  0, 32, 96,224,224,224,224,224,254,224,224,224,224},   /* O */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  6, 12, 28, 24},   /* N */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,240, 28,  6,  7,  7},   /* T */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0, 63, 15,  7,  7,  7},   /* A */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0, 30, 39, 71,135,128},   /* I */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,126, 30, 15, 15, 14},   /* N */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,124,131,  3,  1,  1},   /* S */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,128,192},   /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},   /* T */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},   /* H */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},   /* E */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},   /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,127, 28, 14, 14,  7},   /* B */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,159,  3,  1,  2,  2},   /* I */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,193,  6, 12, 28, 24},   /* T */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,240, 28,  6,  7,  3},   /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0, 31,  7,  3,  3,  3},   /* P */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,159,160,192,192,128},   /* A */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,192,224, 96,112},   /* T */
   { 28, 28, 28, 28, 28, 28, 28, 28, 62,255,  0,  0,  0,  0},   /* T */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,  0,  0,  0},   /* E */
   { 14, 14, 14,  7,  7,  3,  3,  1,  0,  0,  0,  0,  0,  0},   /* R */
   {  0,  0,  0,  0,  0,  0,128,128,193, 62,  0,  0,  0,  0},   /* N */
   {  0,  0,  0,  0, 16, 16, 32, 64,128,  0,  0,  0,  0,  0},   /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},   /* F */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0},   /* O */
   {112,112,112,112,112,112,112,112,248,254,  0,  0,  0,  0},   /* R */
   { 28, 28, 28, 28, 28, 28, 28, 28, 62,255,  0,  0,  0,  0},   /*   */
   {  3,  3,  3,  3,  3,  3,  3,  3,  7,159,  0,  0,  0,  0},   /* O */
   {128,128,128,128,128,128,128,128,192,240,  0,  0,  0,  0},   /* N */
   {224,224,224,224,224,224, 96,112, 49, 30,  0,  0,  0,  0},   /* E */
   { 56, 63, 56, 56, 56, 24, 92, 76,134,  1,  0,  0,  0,  0},   /*   */
   {  7,255,  0,  0,  0,  0,  1,  2, 12,240,  0,  0,  0,  0},   /* C */
   {  7,  7,  7,  7,  7,  7,  7,  7, 15, 63,  0,  0,  0,  0},   /* H */
   {  0,  0,  0,  0,  0,  0,  0,  0,128,224,  0,  0,  0,  0},   /* A */
   { 14, 14, 14, 14, 14, 14, 14, 14, 31,127,  0,  0,  0,  0},   /* R */
   {  1,  1,  1,  1,  1,  1,  1,  1,  3,207,  0,  0,  0,  0},   /* A */
   {192,192,192,192,192,192,192,192,224,248,  0,  0,  0,  0},   /* C */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},   /* T */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},   /* E */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},   /* R */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},   /*   */
   {  7,  3,  3,  3,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0},   /* O */
   {  4,132,136,136,208,208,224,224, 96, 64,  0,  0,  0,  0},   /* F */
   { 56, 56, 56, 56, 56, 24, 28, 12,  6,  1,  0,  0,  0,  0},   /*   */
   {  3,  3,  3,  3,  3,  3,  7,  6, 28,240,  0,  0,  0,  0},   /* T */
   {131,131,131,131,131,  3,  3,  3,  7, 31,  0,  0,  0,  0},   /* H */
   {128,128,128,128,128,128,128,128,192,243,  0,  0,  0,  0},   /* E */
   {112,112,112,112,112,112,112,112,248,254,  0,  0,  0,  0},   /*   */
   {  0,  0,252, 60, 30, 30, 30, 23, 23, 23, 19, 19, 19, 17},   /* A */

   {  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,130,130,130,196},   /* S */
   {  0,  0,126,120,240,240,240,112,112,112,112,112,112,112},   /* C */
   {  0,  0, 28, 28, 28,  0,  0,  0,  0,252, 60, 28, 28, 28},   /* I */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  6, 12, 28, 24},   /* I */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,240, 12,  2,  7,  7},   /*   */
   {  0,  0, 63, 15,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7},   /* C */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0, 62, 65,129,128,  0},   /* H */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,192,192,224},   /* A */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0, 63, 64,224,224,224},   /* R */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,192, 96,112,112},   /* A */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  7, 24, 48,112, 96},   /* C */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,192,112, 24, 28, 28},   /* T */
   {  0,  0,252, 60, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28},   /* E */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},   /* R */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},   /*   */
   {  0,  0, 63, 56, 48, 48, 32, 32, 32,  0,  0,  0,  0,  0},   /* S */
   {  0,  0,255,112,112,112,112,112,112,112,112,112,112,112},   /* E */
   {  0,  0,225,225, 97, 32, 32, 32, 32, 15,  3,  1,  1,  1},   /* T */
   {  0,  0,192,192,192,  0,  0,  0,  0,192,193,195,195,195},   /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,252,  3,  0,  0,  0},   /* O */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0, 64, 65,195, 71, 70},   /* F */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,124,131,  0,  1,  1},   /*   */
   {  0,  0, 15,  3,  1,  1,  1,  1,  1,  1,  1,129,193,193},   /* T */
   {  0,  0,192,192,192,192,192,192,192,207,208,224,224,192},   /* H */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,128, 96,112, 48, 56},   /* E */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  3, 12, 24, 56, 48},   /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,224, 56, 12, 14, 14},   /* I */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,126, 30, 14, 15, 15},   /* B */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0, 60, 78,142, 14,  0},   /* M */
   { 17, 17, 16, 16, 16, 16, 16, 16, 48,254,  0,  0,  0,  0},   /* - */
   {196,196,232,232,232,112,112, 80, 32, 35,  0,  0,  0,  0},   /* P */
   {112,112,112,112,112,112,112,112,248,254,  0,  0,  0,  0},   /* C */
   { 28, 28, 28, 28, 28, 28, 28, 28, 62,255,  0,  0,  0,  0},
   { 56, 56, 56, 56, 56, 24, 28, 12,  6,129,  0,  0,  0,  0},
   {  7,  0,  0,  0,  0,  0,  1,  2, 12,240,  0,  0,  0,  0},
   {  7,  7,  7,  7,  7,  7,  7,  7, 15, 63,  0,  0,  0,  0},
   {  0,  0,  0,  0,  0,  0,  0,  0,129,231,  0,  0,  0,  0},
   {224,224,224,224,224,225,225,224,240,252,  0,  0,  0,  0},
   {  0,  3, 60,224,224,192,192,224,225, 62,  0,  0,  0,  0},
   {112,240,112,112,112,112,112,120,184, 14,  0,  0,  0,  0},
   {224,255,224,224,224, 96,112, 48, 24,  7,  0,  0,  0,  0},
   { 28,252,  0,  0,  0,  0,  4,  8, 48,192,  0,  0,  0,  0},
   { 28, 28, 28, 28, 28, 28, 28, 28, 62,255,  0,  0,  0,  0},
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,  0,  0,  0},

   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0},
   {112,112,112,112,112,112,112,112,248,254,  0,  0,  0,  0},
   {  1,  1,  1,  1,  1,  1,  1,  1,  3, 15,  0,  0,  0,  0},
   {193,193,192,192,192,194,195,195,227,250,  0,  0,  0,  0},
   {240,254,255, 15,  1,  0,  0,  0,129,126,  0,  0,  0,  0},
   { 14, 14,142,206,206,198, 71,195,129,  0,  0,  0,  0,  0},
   {  1,  0,  0,  0,  0,  0,  0,  0,131,124,  0,  0,  0,  0},
   {193,  1,  1,  1,  1,  1, 65,129,  3, 15,  0,  0,  0,  0},
   {192,192,192,192,192,192,192,192,224,249,  0,  0,  0,  0},
   { 56, 56, 56, 56, 56, 56, 56, 56,124,255,  0,  0,  0,  0},
   {112,127,112,112,112, 48, 56, 24, 12,  3,  0,  0,  0,  0},
   { 14,254,  0,  0,  0,  0,  2,  4, 24,224,  0,  0,  0,  0},
   { 14, 14, 14, 14, 14, 14, 14, 14, 31,127,  0,  0,  0,  0},
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,192,  0,  0,  0,  0}
  };

  union REGS regs;      /* processor registers for the interrupt call */
  unsigned i, j, k;                                   /* loop counter */

  /*-- prepare screen layout -----------------------------------------*/

  cls();                                          /* clear the screen */
  blinking( FALSE );          /* light background instead of blinking */
  chardef(128, 0, 14, 120, (BYTE far *) font);   /* define characters */

  for (i=0; i<250; ++i)                 /* run through loop 250 times */
   {                             /* write color bars to the video RAM */
    printfat(GETCS(), GETCZ(), ((i % 14) + 1) << 4, "     ");
    printfat(GETCS(), GETCZ(), 0, "   ");
   }
  for (i=10; i<16; ++i)                        /* make space for logo */
   printat(22, i, 0, "                                  ");
  for (k=128, i=0; i<4; ++i)        /* the logo consists of the ASCII */
   {                                /* characters 128 - 248           */
    for (j=0; j<30; ++j)
     printfat(j+24, i+11, 15, "%c", k++);
   }
  printat(1, 1, 15, " In spite of the logo, the most      ");
  printat(1, 2, 15, " important characters are still here ");
  printat(1, 3, 15, "                                 ");
  printat(1, 4, 15, " !\"#$%&'()*+-./0123456789:;<=>?@ ");
  printat(1, 5, 15, " ABCDEFGHIJKLMNOPQRSTUVXXYZ[\\]^_ ");
  printat(1, 6, 15, " `abcdefghijklmnopqrstuvxyz{|}~  ");
  printat(33, 21, 15, "                                           ");
  printat(33, 22, 15, " press a key to end the program            ");
  printat(33, 23, 15, "                                           ");
  setcur( 34, 22);

  /*-- change colors of the color bars -------------------------------*/

  i = 0;                       /* initial value of the color register */
  while ( nokey() )                    /* repeat until key is pressed */
   {
    ++i;           /* increment the color value of the first register */
    for (j=1; j<15; ++j)              /* run through register 1 to 14 */
     {
      setcol(j, i+j & 63);       /* write color value in the register */
      if ( !nokey() )                                 /* key pressed? */
       break;                           /* YES --> terminate loop now */
     }
   }
  cls();                                              /* clear screen */
 }


/**********************************************************************/
/**                           MAIN PROGRAM                           **/
/**********************************************************************/

void main()
{
 if ( is_ega() )                         /* is an EGA card installed? */
  {                                                            /* YES */
   if ( getmon() == EGA )             /* is an EGA monitor connected? */
    egabios();                                     /* yes, start demo */
   else
    { /* wrong Monitor */
     printf("This program can be used only\n");
     printf("with an EGA monitor!\n");
    }
  }
 else                                                 /* no EGA card */
  printf("No EGA card is installed!\n");
}
