{**********************************************************************}
{*                      F I X P A R T P . P A S                       *}
{*--------------------------------------------------------------------*}
{*    Task           : Display hard disk partitioning                 *}
{*--------------------------------------------------------------------*}
{*    Author         : MICHAEL TISCHER                                *}
{*    Developed on   : 04/26/1989                                     *}
{*    Last update    : 06/22/1989                                     *}
{*--------------------------------------------------------------------*}
{*    Call           : FIXPARTP [ Drive number ]                      *}
{*                       Default is drive 0 (drive C:)                *}
{**********************************************************************}

uses Dos;                                               { Add DOS unit }

{== Type declaration ==================================================}

type  SecPos    = record          { Describes the position of a sector }
                    Head : byte;                     { Read/write head }
                    SecCyl : word;        { Sector and cylinder number }
                  end;

      PartEntry = record                { Entry in the partition table }
                    Status   : byte;                { Partition status }
                    StartSec : SecPos;                  { First sector }
                    PartTyp  : byte;                  { Partition type }
                    EndSec   : SecPos;                   { Last sector }
                    SecOfs   : longint;    { Offset of the boot sector }
                    SecNum   : longint;            { Number of sectors }
                  end;

      PartSec   = record              { Describes the partition sector }
                    BootCode  : array [0..$1BD] of byte;
                    PartTable : array [1..4] of PartEntry;
                    IdCode    : word;                          { $AA55 }
                  end;

{**********************************************************************}
{*  ReadPartSec : Read a partition sector from the hard disk and      *}
{*                place in a buffer                                   *}
{**------------------------------------------------------------------**}
{*  Input  : - HrdDrive : BIOS code of the drive ($80, $81 etc.)      *}
{*           - Head     : Read/write head number                      *}
{*           - SecCyl   : Sector and cylinder number in BIOS format   *}
{*           - Buf      : Buffer into which sector should be loaded   *}
{**********************************************************************}

function ReadPartSec( HrdDrive, Head : byte;
                      SecCyl         : word;
                      var Buf        : PartSec ) : boolean;

var Regs : Registers;              { Processor regs for interrupt call }

begin
  Regs.AX := $0201;                { Function no. for "Read", 1 sector }
  Regs.DL := HrdDrive;                        { Load additional        }
  Regs.DH := Head;                            { parameters into the    }
  Regs.CX := SecCyl;                          { different registers    }
  Regs.ES := seg( Buf );
  Regs.BX := ofs( Buf );
  Intr( $13, Regs);                         { Call hard disk interrupt }
  ReadPartSec := ( Regs.Flags and 1 ) = 0;  {Carry flag indicates error}
end;

{**********************************************************************}
{*  GetSecCyl: Determines the combined sector/cylinder coding of BIOS *}
{*             sector and cylinder number                             *}
{**------------------------------------------------------------------**}
{*  Input  : SecCyl   : Value to be decoded                           *}
{*           Sector   : Reference to the sector variable              *}
{*           Cylinder : Reference to the cylinder variable            *}
{**********************************************************************}

procedure GetSecCyl( SecCyl : word; var Sector, Cylinder : integer );

begin
  Sector   := SecCyl and 63;                 { Exclude bits 6 and 7 }
  Cylinder := hi( SecCyl ) + ( lo( SecCyl) and 192 ) shl 2;
end;

{**********************************************************************}
{*  ShowPartition: Displays hard disk partitioning on the screen      *}
{**------------------------------------------------------------------**}
{*  Input  : DR : Number of the corresponding hard disk drive         *}
{*                (0, 1, 2 etc.)                                      *}
{**********************************************************************}

procedure ShowPartition( DR : byte );

var Head     : byte;                       { Head of current partition }
    SecCyl   : byte;        { Sector and cylinder of current partition }
    ParSec   : PartSec;                     { Current partition sector }
    Entry    : byte;                                    { Loop counter }
    Sector,                                         { Get sector and   }
    Cylinder : integer;                             { cylinder numbers }
    Regs     : Registers;          { Processor regs for interrupt call }

begin
  writeln;
  DR := DR + $80;                      { Prepare drive number for BIOS }
  if ReadPartSec( DR, 0, 1, ParSec ) then      { Read partition sector }
    begin                                         { Sector is readable }
      Regs.AH := 8;                                  { Read drive data }
      Regs.DL := DR;
      Intr( $13, Regs);                     { Call hard disk interrupt }
      GetSecCyl( Regs.CX, Sector, Cylinder );
      writeln(''+
              'ͻ');
                     { Upper left corner can be typed using <Alt><201> }
                     { Top horiz. line can be typed using <Alt><205>   }
                     { Upper right corner can be typed using <Alt><187>}
      writeln(' Drive    ', DR-$80, ':    ', Regs.DH+1:2,
              ' Heads with       ', Cylinder:5, ' cylinders and  ',
              Sector:3, ' sectors     ');
                           { Vert. lines can be typed using <Alt><186> }
      writeln(' Partition table  in  partition sector     '+
              '                              ');
                           { Vert. lines can be typed using <Alt><186> }
      writeln(''+
              '͹');
                               { Left T can be typed using <Alt><204>  }
                               { Top T can be typed using <Alt><209>   }
                               { Right T can be typed using <Alt><185> }
      writeln('                              Start    '+
              '     End      Dis.fr.       ');
            { First and last vert. lines can be typed using <Alt><186> }
                 { Remaining vert. lines can be typed using <Alt><179> }
      writeln('#.BootType               Head Cyl. Sec.'+
              'Head Cyl. Sec.BootsecNumber ');
            { First and last vert. lines can be typed using <Alt><186> }
                 { Remaining vert. lines can be typed using <Alt><179> }
      writeln(''+
              '͹');
                               { left T can be typed using <Alt><204>  }
                               { crosses can be typed using <Alt><216> }
                               { Right T can be typed using <Alt><185> }
 for Entry:=1 to 4 do                     { Execute table entries }
        with ParSec.PartTable[ Entry ] do
          begin
          write(' ', Entry, '');
           { Type first line using <Alt><186>, second using <Alt><179> }
          if Status = $80 then write ('YES ')
                               else write (' NO ');
          write('');
                               { Type thin vert. line using <Alt><179> }
          case PartTyp of                     { Display partition type }
            $00        : write('Not occupied       ');
            $01        : write('DOS, 12-bit FAT    ');
            $02        : write('XENIX              ');
            $03        : write('XENIX              ');
            $04        : write('DOS, 16-bit FAT    ');
            $05        : write('DOS, extd.partition');
            $DB        : write('Concurrent DOS     ');
            else         write('Unknown   (',PartTyp:3,')    ');
          end;
          GetSecCyl( StartSec.SecCyl, Sector, Cylinder );
          write('', StartSec.Head:2,' ',Cylinder:5,'  ',Sector:3 );
          GetSecCyl( EndSec.SecCyl, Sector, Cylinder );
                                    {Enter vert. line using <Alt><179> }
          write(' ', EndSec.Head:2,' ',Cylinder:5,'  ',Sector:3 );
                                    {Enter vert. line using <Alt><179> }
          writeln(' ', SecOfs:7,'', SecNum:7,'');
                { Enter first and second vert. lines using <Alt><179>, }
                { third line using <Alt><186>                          }
end;
      writeln(''+
              'ͼ'#13#10);
                          { Left angle can be typed using <Alt><200>   }
                          { Horiz. lines can be types using <Alt><205> }
                          { Bottom Ts can be typed using <Alt><207>    }
                          { Right angle can be typed using <Alt><188>  }
    end
  else
    writeln('Error during boot sector access!');
end;

{***********************************************************************
*                          M A I N     P R O G R A M                   *
***********************************************************************}

var HrdDrive,                              { Variables for converting  }
    DError    : integer;                   { given arguments           }

begin
  writeln( #13#10'-------------------------------- FIXPARTP - (c)',
           ' 1989 by MICHAEL TISCHER ---' );
  HrdDrive := 0;                          { Default is first hard disk }
  if ParamCount = 1 then                { Other drive specifier given? }
    begin                                                        { YES }
      val( ParamStr(1), HrdDrive, DError );            { ASCII/decimal }
      if DError <> 0 then                          { Conversion error? }
        begin                                                    { YES }
          writeln(#13#10'Illegal drive specifier!');
          exit;                                          { End program }
        end;
    end;
  ShowPartition( HrdDrive );                { Display partition sector }
end.

