PROGRAM SERVO;
{
 - Control R/C Model Servo Motors from a parallel printer port.
 - A Servo Motor is a DC motor with feedback.
 - All IBM / clone 8253 timers generates frequencies up to 1,193,180 Hertz
 - For R/C model servos, pulse width should be 0.5ms to 2.5ms.
 - Motors may oscillate due to delay increase cause by timer and other
   internal interrupt calls.  The effect is more noticeable on slower machines.
 - Many thanks to my good friend, Mike Mensch, who developed this program
}
USES DOS, CRT, My_Tpu;
CONST  PIA_8255B  =  $0061;    { 8255 port B address                        }
       TIM_8253C  =  $0043;    { 8253 timer control register                }
      TIM_8253T2  =  $0042;    { 8253 timer 2 register                      }

      Time_10_ms  =  $2e9a;    { 10 milli-seconds                           }
       time_2_ms  =  $0952;  { computed time for 2 millisecond delay        }
       time_1_ms  =  $04a9;  { computed time for 1 millisecond delay        }
    time_half_ms  =  $0255;  { computed time for half a millisecond delay   }
           Survo  :  BYTE = 0; { byte = unsigned char                       }

VAR                                     Ch : CHAR;
                        PBbits, r,y ,motor : BYTE;
    Lpt_Num, Lpt_Port_Address, Delay_Value : WORD;

PROCEDURE Do_Pulse( Delay_Value : WORD); ASSEMBLER;
{
 Send three copies of bit at three-phase zero crossings
 Waits for next zero crossing and restarts timer
 Gets bit value from global bitval
 Mashes BL and DX
}
LABEL  ReGet, L1, L2, L3, L4;    { must pre declare subroutine names in TP }
{
 NOTE: Jumps to lables L1, L2, L3, and L4 are not needed on XT machines
       XT machines do not have a pre-fetch.
}
ASM
           PUSH    bx
           MOV     bx,Delay_Value  { load the contents of Delay_Value into Bx }
           PUSH    bp
           MOV     bp,sp
           CLI
           IN      AL,PIA_8255B        {   get current port B bit          }
           OR      AL,$01              {   turn on timer gate              }
           JMP     L1                  {   flush CPU's internal buffer     }
L1:        OUT     PIA_8255B,AL        {   update timer gate bits          }
           STI                         {   get 16 bit timer count          }
ReGet:
           MOV     AL,10000000B        {   latch Timer 2                   }
           JMP     L2                  {   flush CPU's internal buffer     }
L2:        CLI
           OUT     TIM_8253C,AL        {   load 8255 control register w/AL }
           JMP     L3                  {   flush CPU's internal buffer     }
L3:        IN      AL,TIM_8253T2       {   get LSB from timer              }
           MOV     AH,AL
           JMP     L4                  {   flush CPU's internal buffer     }
L4:        IN      AL,TIM_8253T2       {   get MSB from timer              }
           STI (* shift this one down to after jump, JB *)
           XCHG    AH,AL               {   swap bits around,               }
           NOT     AX                  {   and flip bits to count upward   }
           CMP     AX,bx  { compare until count ( in AX ) >=  Delay_Value  }
           JB      reget
           (* try sti here instead of above *)
           POP     bp
           POP     bx
END; { do pulse }


PROCEDURE Init_Time; ASSEMBLER;
{
  Delay times measured in Timer-2 ticks These are found by computing
    (time/54.9ms)*64*1024 Delay required on AT after each I/O operation
  Sets up Timer2 Halts timer counting,
  sets FFFF reload value
  Sets Mode 2
  Clears "speaker enable" bit
  Does not start timer, leaves "gate speaker" bit low
}
LABEL  L1, L2, L3;       { must pre declare subroutine names in Turbo Pascal }
{
 NOTE: Jumps to lables L1, L2, and L3 are not needed on XT machines
       XT machines do not have a pre-fetch.
}
ASM        CLI                           {  clear control bits               }
           IN      AL,PIA_8255B          {  get current port B bits          }
           AND     AL, $FC               {  turn off timer gate and speaker  }
           JMP     L1                    {  flush CPU's internal buffer      }
L1:        OUT     PIA_8255B,AL          {  load Port B bits                 }
           STI
           CLI                           {  set Timer 2 to LSB/MSB mode 2    }
           MOV     AL,$B4                {  10110100B                        }
           OUT     TIM_8253C,AL          {  send control byte to 8253        }
           MOV     AL,$FF                {  set reload value to FFFF         }
           JMP     L2                    {  flush CPU's internal buffer      }
L2:        OUT     TIM_8253T2,AL
           JMP     L3                    {  flush CPU's internal buffer      }
L3:        OUT     TIM_8253T2,AL
           STI
           END; { asm }

PROCEDURE Delay_Between;
{
 set up for a ten millisecond delay between pulses sent to servo motor.
 may be eliminated if you are controlling only one motor
}
BEGIN
Init_Time;
Do_Pulse(time_10_ms);
END;

PROCEDURE Pulse(VAR r, MOTOR : BYTE; VAR Delay_Value : WORD);
{
 converts r value, (0 to 255), to appropriate pulse width, (0.5 to 2.5 ms)
}
BEGIN
Delay_Value := time_1_ms + ( 10 * r ) + 75;
Init_Time;
Do_Pulse(time_half_ms);
PORT[Lpt_Port_Address ] :=  motor;      { set printer port active bits HIGH }
Do_Pulse(Delay_Value);
PORT[Lpt_Port_Address] := $00;          { set printer port bits LOW         }
END; { pulse }

BEGIN { main }
Select_Printer_Port(Lpt_Num, Lpt_Port_Address); { in My_Tpu }
Ch := CHAR(32);
r := 128;
WHILE ORD(ch) <> 27 DO
  BEGIN { while }
  IF KeyPressed THEN
    BEGIN { if }
    Ch := ReadKey;
       CASE Ch OF
       '6', 'M' : if ( r < 255 ) THEN INC(R);
       '4', 'K' : if ( r > 0 ) THEN  DEC(r);
       '8', 'H' : if ( r > 250 ) THEN r := 255 ELSE R := R + 5;
       '2', 'P' : if ( r > 4 ) THEN r := R -5 else r := 0;
       '+' : r := 255;
       '-' : r := 1;
       END;  { case }
    WRITE('Ch = ',Ch); WRITELN(':   r value = ',r);writeln;
    END; { if }
  FOR y := 0 to 2 do
    BEGIN { for y }
    motor := $ff;
    Pulse(r,motor, Delay_Value);
    Delay_Between;
    END; { for y }
  END; { while }
END. { main }

