PROGRAM RS232;
{ Input / output data from IBM Serial Port, COM1 or COM2                      }

{ REF: Understanding Serial Communications, Peter W Gofton, Sybex Pub., Ch 15 }
{ Programmer's PC Sourcebook, by Tom Hogan, Microsoft Press 1991, 4-54..4-58  }
{ Turbo Pascal 6, by Stephen O'Brien, McGraw Hill Pub. Page 394..400          }

{ Programed by Paul Bergsman for use by Trenton Stat College to interface     }
{  thier electric car's killowatt meter to an IBM type serial port. 5/18/1993 }
{ Written with Borland's Turbo Pascal 6.                                      }

USES CRT, DOS;
TYPE Modem_Set_Type =
       RECORD
       Comm_Port_Num, BPS, Data_Bits, Stop_Bits : INTEGER;
                                         Parity : CHAR;
       END; { record }
      String80 = String[80];

CONST
  Main_Dseg : INTEGER = 0;
          Line_Offset = 5;
                   CR = #13;
    Max_Buffer_Length = 1024;
           Inter_Ctrl = $21;      { 8259 Interrupt Controller Port            }
VAR
                            Modem_Set : Modem_Set_Type;
                         Async_Vector : Pointer;
                                 Regs : Registers;
                                 Orig : CHAR;
                               Buffer : Array[1..Max_Buffer_Length] OF CHAR;
                          Data_String : String80;
                             Init_Num : BYTE;
   Chars_In_Buffer, Circ_out, Circ_In : WORD;
Intr_En_Reg, Line_Ctrl_Reg, Modem_Ctrl_Reg, Line_Status_Reg, 
             Modem_Ser_Reg, Data_Port : WORD;

PROCEDURE Select_Modem_Setup(VAR  Modem_Set : Modem_Set_Type);
 { user interface for serial port paramiter selection }
VAR Ch : CHAR;
BEGIN
ClrScr; GoToXY(1,5);
WRITELN { default settings match Solar Car's KiloWatt Meter Serial interface }
 ('Default settings are: COM1, 9600 baud, 8 data bits, 1 stop bit, No Parity');
WRITE(' Y / N: '); READLN(Ch);
WITH Modem_Set DO
 BEGIN { with }
 IF UPCASE(Ch) = 'Y' THEN
  BEGIN { if upcase }
  Comm_Port_Num := 1; BPS := 9600; 
  Data_Bits := 8; Stop_Bits := 1; 
  Parity := 'N';
  END { if upcase }
 ELSE
  BEGIN { else }
   REPEAT { getting user's parameters }
   ClrScr; WRITELN;
    REPEAT
    WRITELN; WRITE ('Select COMM PORT: 1 / 2: '); READLN(Comm_Port_Num);
    IF Comm_Port_Num IN [1, 2] THEN BEGIN END
      ELSE WRITELN('You selecte COMM PORT ',Comm_Port_Num,'.  Try again.');
    UNTIL Comm_Port_Num IN [1, 2];

    REPEAT
    WRITELN; WRITE('Select BAUD; 300, 600, 1200, 2400, 4800, 9600: ');
    READLN(BPS);
    IF ( BPS DIV 100 ) IN [ 3, 6, 12, 24, 48, 96] THEN BEGIN END
      ELSE WRITELN('You have selected ',BPS,' Bits Per Second. Try again');
    UNTIL ( BPS DIV 100 ) IN [ 3, 6, 12, 24, 48, 96];

    REPEAT
    WRITELN; WRITE ('Select Data Bits; 5, 6, 7, or 8: '); READLN(Data_Bits);
    IF Data_Bits IN [5..8] THEN BEGIN END
      ELSE  WRITELN('You have selected ',Data_Bits,' Data Bits.  Try again');
    UNTIL Data_Bits IN [5..8];

    REPEAT
    WRITELN; WRITE('Select Stop Bits; 1 / 2: '); READLN(Stop_Bits);
    IF Stop_Bits IN [1..2] THEN BEGIN END
      ELSE WRITELN('You have selected ',Stop_Bits,' Stop Bits.  Try Again');
    UNTIL Stop_Bits IN [1..2];

    REPEAT
    WRITELN; WRITE('Select Parity; (N)one, (O)dd, (E)ven: ');
    READLN(Parity); Parity := UpCase(Parity);
    IF Parity IN ['N', 'O', 'E'] THEN BEGIN END
      ELSE WRITELN('You have selected ',Parity,' paritey.  Try again.');
    UNTIL UpCase(Parity) IN ['N', 'O', 'E'];

   WRITELN; WRITE('SELECTED PARAMETERS ARE: COM', Comm_Port_Num,', ');
   WRITE(BPS,' Baud, ',Data_Bits,' Data Bits, ', Stop_Bits,' Stop Bits, ');
   CASE Parity OF
         'N' : WRITELN('NO Parity.');
         'O' : WRITELN('ODD Parity.');
         'E' : WRITELN('EVEN Parity.');
         END; { case }
   WRITELN; WRITE('Are these parameters correct Y/N: '); READLN(Ch);
   UNTIL UpCase(Ch) = 'Y';
  END; { else }
  WRITELN;
  END; { with }
END; { modem setup }

PROCEDURE Async_Int; INTERRUPT;
BEGIN
INLINE($FB);  { STI = set interrupt enable }
IF (Chars_In_Buffer < Max_Buffer_Length) THEN
  BEGIN
  Buffer[Circ_In] := Char(Port[Data_Port]);
  IF ( Circ_In < Max_Buffer_Length) THEN INC(Circ_In,1)
    ELSE Circ_In := 1;
  INC(Chars_In_Buffer,1);
  END;
INLINE($FA);   { CLI = clear interrupt enable }
PORT[$20] := $20;
END; { async init }

PROCEDURE Set_Up_Serial_Port( VAR Modem_Set : Modem_Set_Type );

VAR Regs_AL : Byte;

  (* sub *) PROCEDURE Clear_Buffer;
            BEGIN
            Circ_In := 1;
            Circ_Out := 1;
            Chars_In_Buffer := 0;
            FillChar(Buffer, SizeOf(Buffer),0);
            END; { clear buffer }

  (* sub *) PROCEDURE Enable_Ports;
            CONST Enable_Ready = $01;      { Initial value for Port[IRR]       }
            BEGIN
            Main_Dseg := Dseg;
            Clear_Buffer;
            GetIntVec(Init_Num, Async_Vector);
            SetIntVec(Init_Num, @Async_Int);
            PORT[Inter_Ctrl] := PORT[Inter_Ctrl] AND $0EF;
            PORT[Line_Ctrl_Reg] := PORT[Line_Ctrl_Reg] AND $7F;
            PORT[Intr_En_Reg] := Enable_Ready;
            PORT[Modem_Ctrl_Reg] := $0B;
            PORT[Modem_Ser_Reg] := $08;
            PORT[$20] := $20;
            END; { enable port }

BEGIN { set serial port }
WITH MODEM_SET DO
 BEGIN { with }
   { identify ISN8250 UART Register locations. }
 IF Comm_Port_Num = 1
   THEN BEGIN Data_Port := $03F8;Init_Num := $0C; END
     ELSE BEGIN Data_Port := $02F8; Init_Num := $0B; END;
 Intr_En_Reg := Data_Port + 1;     { Enables Serial Port when set to 1        }
 Line_Ctrl_Reg := Data_Port + 3;   { Sets communications parameters           }
 Modem_Ctrl_Reg := Data_Port + 4;  { Bits 1, 2, and 4 set HIGH to ready modem }
 Line_Status_Reg := Data_Port + 5; { Bit 6 HIGH = safe to send data           }
 Modem_Ser_Reg := Data_Port + 6;   { To start, initialize with 80h            }
 CASE BPS OF
    100 : BPS := 0;
    150 : BPS := 1;
    300 : BPS := 2;
    600 : BPS := 3;
   1200 : BPS := 4;
   2400 : BPS := 5;
   4800 : BPS := 6;
   9600 : BPS := 7;
   END; { baud case }
 Stop_Bits := Stop_Bits - 1;
 CASE Data_Bits OF
  5 : Data_Bits := 0;
  6 : Data_Bits := 1;
  7 : Data_Bits := 2;
  8 : Data_Bits := 3;
  END; { data bit case }
 Regs_AL := (BPS SHL 5) + (Stop_Bits SHL 2) + Data_Bits;
 CASE Parity OF
   'N' : Regs_AL := Regs_AL + 0;   { or 16 }
   'E' : Regs_AL := Regs_AL + 24;
   'O' : Regs_AL := Regs_AL + 8;
   END; { parity case }
 Regs.DX := Comm_Port_Num - 1;
 Regs.AH := 0;
 Regs.AL := Regs_AL;
 Regs.Flags := 0;
 INTR($14,Regs);
 Enable_Ports;
 END; { with }
END; { set serial port }

FUNCTION Get_Char_In_Buffer : CHAR;
 { if character is in buffer, process it }
BEGIN
IF Chars_In_Buffer > 0 THEN
  BEGIN
  Get_Char_In_Buffer := Buffer[Circ_Out];
  If Circ_Out < Max_Buffer_Length THEN INC(Circ_Out, 1) ELSE Circ_Out := 1;
  DEC(Chars_In_Buffer,1);
  END; { if }
END; { get char in buffer }

PROCEDURE Send_Char(Out_Char : CHAR);
{ send character out serial port }
BEGIN
WHILE ((PORT[Line_Status_Reg] AND $20) <> $20) DO BEGIN END;
PORT[Data_Port] := ORD(Out_Char);
END; { send character }

PROCEDURE Set_Up_Display_Screen;
VAR N : INTEGER;
BEGIN
N := Line_Offset;
GoToXY(1,1);     Write('TRENTON STATE, POTENTIAL DIFFERENCE');
GoToXY(1,N);     Write('TIME (sec): ');
GoToXY(1,N + 2); Write('      AMPS: ');
GoToXY(1,N + 4); Write('     VOLTS: ');
GoToXY(1,N + 6); Write('    KW Hrs: ');
END; { set up display screen }

PROCEDURE Display_Data(Data_String : String80);
 { display electric car's wattmeter info on laptop's display }

VAR P, N : INTEGER;
    Volts, Amps, Watts, Time : String[10];
BEGIN
  (* separate data string into Time, Watts, Amps, and Volts *)
  (* elements are separated by commas                       *)
P := POS(',', Data_String); Time := COPY(Data_String, 1, P-1);
Data_String := Copy(Data_String,P+1,LENGTH(Data_String));
P := POS(',', Data_String); WATTS := COPY(Data_String, 1, P-1);
Data_String := Copy(Data_String,P+1,LENGTH(Data_String));
P := POS(',', Data_String); AMPS := COPY(Data_String, 1, P-1);
Data_String := Copy(Data_String,P+1,LENGTH(Data_String));
P := LENGTH(Data_String); VOLTS := COPY(Data_String, 1, P);
Data_String := Copy(Data_String,P+1,LENGTH(Data_String));
  (* display data *)
N := Line_Offset;
GoToXY(1,1);     Write('TRENTON STATE, POTENTIAL DIFFERENCE');
GoToXY(13,N);     Write('TIME (sec): ', Time:7);
GoToXY(13,N + 2); Write('      AMPS: ', Amps:7);
GoToXY(13,N + 4); Write('     VOLTS: ', Volts:7);
GoToXY(13,N + 6); Write('    KW Hrs: ', Watts:7);
END;

PROCEDURE Start_Communicating;
 { transmit characters you type, and display characters received by UART      }
VAR Out_Char, In_Char : CHAR;

BEGIN
Data_String := '';
  REPEAT
  IF (Chars_In_Buffer > 0) THEN
    BEGIN
    In_Char := Get_Char_In_Buffer;
    IF In_Char IN [#32..#128] THEN Data_String := Data_String + In_Char
      ELSE IF (In_Char = CR ) THEN
        BEGIN
        Display_Data(Data_String);
        Data_String := '';
        END;
    END;  { if }
  IF KeyPressed THEN
    BEGIN
    Out_Char := ReadKey;
    IF ORD(Out_Char) <> 27 THEN
      BEGIN
      Send_Char(Out_Char);
      IF Out_Char = CR THEN WRITELN ELSE WRITE(Out_Char);
      END; { if ord }
    END; { if keypressed }
  UNTIL  ORD(Out_Char) = 27;
END; { start communicating }

PROCEDURE  Disable_Ports;
 { before exiting program, leave the port as you found it                    }
 { install original intercepts in Interrupt Vector Table and Reset UART      }
BEGIN
 { turn off COMx communication interrupt }
PORT[Inter_Ctrl] := PORT[Inter_Ctrl] OR $10;
 { disable 8250 Data Ready Interrupt }
PORT[Line_Ctrl_Reg] := PORT[Line_Ctrl_Reg] AND $7F;
PORT[Intr_En_Reg] := $0;

 { DISABLE OUT2 on 8250 }
PORT[Modem_Ctrl_Reg] := $0;
PORT[$20] := $20;
SetIntVec(Init_Num, Async_Vector);
END; { disable ports }

BEGIN { main }
Select_Modem_Setup( Modem_Set);
WITH Modem_Set DO Set_Up_Serial_Port( Modem_Set );
ClrScr;
Set_Up_Display_Screen;
Start_Communicating; { exit when ESC Key is pressed }
WRITELN('Ending program.  Disabling Ports');
WRITE('Press RETURN to continue: '); READLN;
Disable_Ports;
TextMode(Lo(LastMode));
END. { main }




