UNIT My_TPU;
{ A TPU unit of procedures I repeatedly use                                  }
INTERFACE

USES CRT, DOS, GRAPH;

TYPE Set_Of_Char = Set Of CHAR;
     Digit_Type  = ARRAY[0..10,0..10] OF STRING[15];
     Line_Info_Type =
       RECORD
       X1, Y1, X2, Y2 : INTEGER;
       END; { line info type }

VAR  OK_Set                  : Set_Of_Char;
     Digit                   : Digit_Type;
     X, Y, Color, Xdir, Ydir : INTEGER;
     Lines                    : Line_Info_Type;

PROCEDURE Cursor_Off;
 { Hide the cursor.  Use BIOS interrupt $10, Function 1 to hide the curser   }
PROCEDURE Cursor_ON;
 { Determine the active vidio card.  This is not a trival task               }
PROCEDURE Clear_Kbd_Buffer;
 { Clear contents of keyboard buffer.                                        }
PROCEDURE Get_Character( OK_Set : Set_Of_Char; VAR Ch : CHAR);
 { Read a character from the keyboard.  Print the character's ASCII code at  }
 { the cursor.  Return the ASCII code to program via variable Ch             }
PROCEDURE Select_Printer_Port(VAR Lpt_Num, Lpt_Port_Address : WORD);
 { check for valid port selection  ( 1, 2, or 3 )                            }
 { check that selected printer port is actually installed on machine         }
 { return port address from BIOS variable segment, ( Lpt_Port_Address )      }
PROCEDURE Find_Lpt1(VAR Lpt1_Base_Address : INTEGER);
 { find first printer port, LPT1 }
FUNCTION Find_LPTx(Num : INTEGER) : WORD;
 { return port address of printer port 1, 2, or 3, selected by variable Num. }
PROCEDURE Init_Printer_Port(Lpt_Num, Lpt_Port_Address : WORD);
 { Initilize selected printer port;                                          }
 { Tell ROM-BIOS service routine $17, function $01, to initilize the         }
 { selected printer port, Lpt_Num.                                           }
PROCEDURE Generate_Characters( Digit : Digit_Type );
 { generate large numbers from extended ASCII characters }

IMPLEMENTATION
VAR         Regs : REGISTERS;

PROCEDURE Clear_Kbd_Buffer;
  { Clear contents of keyboard buffer.                                       }
  { From PC Intern, by Michael Tischer, Abacus Press, 1992, page 462         }
BEGIN
INLINE ( $FA );                  { CLI: disable hardware interrupts          }
MemW[$40:$1A] := MemW[$40:$1C];  { No more characters in Buffer              }
INLINE ( $1B );                  { STI: enable hardware interrupts           }
END;

PROCEDURE Get_Character( OK_Set : Set_Of_Char; VAR Ch : CHAR);
{ Read a character from the keyboard.  Print the character's ASCII code at   }
{ the cursor.  Return the ASCII code to program via variable Ch              }
BEGIN
  REPEAT
  REPEAT UNTIL KEYPRESSED;
  Ch := ReadKey;
  IF ORD(Ch) = 0 THEN { it's an extended character code, between 128 and 255 }
    BEGIN
    Ch := ReadKey;
    Ch := CHAR(ORD(Ch) + 127);
    END;
  IF (Ch IN OK_Set) AND (Ch IN [CHAR(32)..CHAR(255)]) THEN WRITE(Ch);
  IF (Ch IN OK_Set) THEN BEGIN END ELSE WRITE(CHAR(7));
  UNTIL Ch IN OK_Set;
END; { get character }

PROCEDURE Select_Printer_Port(VAR Lpt_Num, Lpt_Port_Address : WORD);
VAR Ch : CHAR;
  { check for valid port selection  ( 1, 2, or 3 )                           }
  { check that selected printer port is actually installed on machine        }
  { return port address from BIOS variable segment, ( Lpt_Port_Address )     }
  {   $0408 contains port address of LPT1                                    }
  {   $040A contains port address of LPT2                                    }
  {   $040C contains port address of LPT3                                    }
  { The Programmer's PC Sourcebook, MicroSoft Press, 1991, Page 7-75,        }
  { Compute's Mapping the IBM PC, Compute Books, 1985, page 234              }

BEGIN
  REPEAT
  ClrScr; Ch := ' ';   Lpt_Port_Address := 0;
  GOTOXY(1,10); WRITE('Which printer port do you want to use, 1, 2, or 3? ');
  OK_Set := ['1'..'3'];
  Get_Character( OK_Set, Ch);
  Lpt_Num := ORD(Ch) - 48;
  Lpt_Port_Address := MemW[ 0: $0400 + 6 + ( Lpt_Num * 2 ) ];
  IF Lpt_Port_Address = 0 THEN
    BEGIN
    WRITELN;
    WRITELN('Port selected is not installed on this machine.');
    WRITELN; WRITELN('PRESS ANY KEY TO CONTINUE:');
    REPEAT UNTIL KEYPRESSED;     Ch := Readkey;
    END; { else }
  UNTIL Lpt_Port_Address <> 0;
END; { select printer port }

PROCEDURE Find_Lpt1(VAR Lpt1_Base_Address : INTEGER);
CONST Base_Address = 1024;      Offset = 8;
  { use 8 for LPT1, use 10 to find LPT2, and 12 to find LPT3 }
BEGIN
Lpt1_Base_Address := MemW[0: Base_Address + Offset];
END;

FUNCTION Find_LPTx(Num : INTEGER) : WORD;
{
return port address of printer port, 1, 2,  or 3, selected by variable Num.
From:
  Compute's Mapping The IBM PC & PCjr, Russ Cavies, Compute Books, 1985, P 234
  The Programmer's PC Source Book, 1991, Thom Hogan, Microsoft Press, Page 7-75
}
CONST Base_Address = 1024;  { base address for IBM printer ports }
VAR Offset : INTEGER;
BEGIN
Offset := (2 * Num) + 6; { 8 = LPT1, 10 = LPT2, and 12 = LPT3 }
Find_LPTX := MemW[0: Base_Address + Offset];
END; { find lpt1 }

PROCEDURE Init_Printer_Port(Lpt_Num, Lpt_Port_Address : WORD);
{ Initilize selected printer port;                                           }
{ Tell ROM-BIOS service routine $17, function $01, to initilize the selected }
{   printer port, Lpt_Num.                                                   }
{ Referances:                                                                           }
{ PC Interrupts, Addison-Wesley, 1991, page 3-31, &  Programmer's Reference  }
{ Manual for IBM Personal Computers, Dow Jones Pub, 1986 Pub. Page 678       }
BEGIN
WITH Regs DO  { from Mastering Turbo Pascal 6, (c) 1991, Tom Swan, Page 889  }
  BEGIN
  AH := $01;     { function number }
  DX := Lpt_Num; { printer port number }
  END; { with }
INTR($17, Regs);       
END; { init printer port }

PROCEDURE Cursor_Off;
{ Hide the cursor.  Actually, sets cursor off the screen                     }
{ Uses BIOS interrupt $10, Function 1 to hide the curser                     }
{ Restores the normal default cursor setings.                                }
{ Uses BIOS interrupt $10, Function 1 to hide the curser                     }
{ Turbo Pascal 6, Comp. Ref; Borland/Osborne Pub; Stephen. O'Brien, Page 416 }
{ PC Interrupts, Addison Wesley Pub. 1991, by Ralf Brown; Page 5-12          }
{ Programmer's Ref Man for IBM Per. Comp. Dow Jones Irwin Pub, 1986; P.522   }
{ Programming the IBM User Interface; Addison-Wesley 1989; Ben Ezzell; P.223 }

BEGIN
WITH Regs DO
  BEGIN
  ah := $01;  { set cursor shape           }
  ch := $20;  { hide cursor, starting row  }
  cl := $02;  { hide cursor, ending row    }
  END; { with }
INTR($10, Regs);
END; { cursor off }

PROCEDURE Cursor_ON;
{ Find out if a graphics card is being used.  Use apropreate procedure to    }
{ turn on the cursor. Hercules and MDA video cards are different from CGA    }
{ Turbo Pascal 6 System Programming, Abacus 1991, by Michal Tischer, Pg 414  }

TYPE Video_Cards = (MDA, CGA, HERC, EGA, VGA, Unknown );
     VioSaveBuf = ARRAY [$0000..$3FFF] OF BYTE;
     VSPtr = ^VioSaveBuf;
VAR                          Video_Card : Video_Cards;
                                Vio_Ptr : VSPtr;  { pointer to video RAM }
                              IN_Graphc : BOOLEAN;
                            Active_Mode : String[10];
                  Video_Mode, Scr_Lines : BYTE;
                             Vidio_Mode : BYTE;
               Graph_Driver, Graph_Mode : INTEGER;

PROCEDURE Graphics_Cursor_ON;
{ Uses BIOS interrupt $10, Function 1 to restore the normal cursor           }
{ Restores the normal default cursor setings to any color graphics card      }
{ Programming the IBM User Interface; Addison-Wesley, 1989; P.223            }
BEGIN
WITH Regs DO
  BEGIN
  ah := $01;
  ch := $06;
  cl := $07;
  END; { with }
INTR($10, Regs);
END; { graphics cursor on }

PROCEDURE Mono_Cursor_ON;
{ Uses BIOS interrupt $10, Function 1 to restore the normal cursor. Restores }
{ the normal default cursor to monochrome and Hercules graphics cards.       }
{ Programming the IBM User Interface; Addison-Wesley, 1989; P.223            }
BEGIN
WITH Regs DO
  BEGIN
  ah := $01;
  ch := $09;
  cl := $0A;
  END; { with }
 INTR($10, Regs);
END; { mono cursor on }

PROCEDURE Get_Video_Card;
{ Determine the active vidio card.  This is not a trival task               }
{ Turbo Pascal 6 System Programming, Abacus 1991, Michael Tischer,Pg 410-15 }
Const VGA_Vidio_Tab : ARRAY[0..12] OF Video_Cards =
                   (Unknown, MDA, CGA, Unknown, EGA, EGA,
                    Unknown, VGA, VGA, Unknown, CGA, CGA, CGA );
      Mono_Adr_Reg = $3b4;                   { monochrome card port address }
      Mono_Status = $3ba;           { monochrome card's status port address }
VAR   Regs : Registers;
         i : INTEGER;
    Status : BYTE;                             { MDA-/ Hercules status port }

BEGIN
Video_Card := Unknown;
Regs.AH := $0F;                          { Function to determine vidio mode }
INTR($10, Regs);
Video_Mode := Regs.AL and $7f;                           { store vidio mode }

Regs.AX :=$1a00;                        { Function to determine if only VGA }
INTR($10,Regs);
IF (Regs.AL = $1a) THEN                            { is function available? }
  BEGIN
  IF (Regs.BL <> $1a) THEN          { yes, VGA, code of current card in  BL }
    Video_Card := VGA_Vidio_Tab[Regs.BL]         { yes, get code from table }
  END
ELSE                                              { not VGA, is it an EGA ? }
  BEGIN
  Regs.AH := $12;                                       { call function $12 }
  Regs.BL := $10;                                       { sub function $10  }
  INTR($10, Regs);                                      { call video BIOS   }
  If (Regs.BL <> $10) THEN Video_Card := EGA;           { EGA  is installed }
  END; { else }
IF (Video_Card = Unknown) THEN                 { vidio card  still unknown? }
  IF  (MemW[$0040:$0063] = Mono_Adr_Reg ) THEN          { monochrome card ? }
    BEGIN                                    { yes, must be MDA or Hercules }
     { if card is Hercules, then bit 7 in CRT status register will change,  }
     { otherwise, the card is a monochrome display                          }
    Vio_Ptr := PTR($B000, $0000 );
    Status := PORT[Mono_Status] AND $80;             { read CRT status port }
    i := 0;
    WHILE ( PORT[Mono_Status] AND $80 = Status) AND (i < 32767) DO INC(I);
    IF ( I = 32767 ) THEN Video_Card := MDA ELSE Video_Card := Herc;
    END { if mem }
ELSE Video_Card := CGA;
END;

PROCEDURE Error_Message;
BEGIN
ClrScr;
WRITELN;
WRITELN('Unable to restore your cursor. ');
WRITELN('Your type of vidio display is "UNKNOWN" to this software');
WRITELN; WRITE('PRESS ANY KEY TO CONTINUE: ');READLN;
END; { case }

BEGIN { cursor on }
Get_Video_Card;
CASE Video_Card OF
         MDA, HERC : Mono_Cursor_ON;
     CGA, EGA, VGA : Graphics_Cursor_ON;
           UNKNOWN : Error_Message;
                END; { case }
END; { cursor on }

PROCEDURE Generate_Characters( Digit : Digit_Type );
 { generate large numbers from extended ASCII characters }
BEGIN
DIGIT[0,1] := ('  ');
DIGIT[0,2] := ('     ');
DIGIT[0,3] := ('       ');
DIGIT[0,4] := ('       ');
DIGIT[0,5] := ('       ');
DIGIT[0,6] := ('       ');
DIGIT[0,7] := ('       ');
DIGIT[0,8] := ('     ');
DIGIT[0,9] := ('  ');

DIGIT[1,1] := ('        ');
DIGIT[1,2] := ('        ');
DIGIT[1,3] := ('        ');
DIGIT[1,4] := ('        ');
DIGIT[1,5] := ('        ');
DIGIT[1,6] := ('        ');
DIGIT[1,7] := ('        ');
DIGIT[1,8] := ('        ');
DIGIT[1,9] := ('        ');

DIGIT[2,1] := ('  ');
DIGIT[2,2] := ('     ');
DIGIT[2,3] := ('        ');
DIGIT[2,4] := ('       ');
DIGIT[2,5] := ('  ');
DIGIT[2,6] := ('       ');
DIGIT[2,7] := ('        ');
DIGIT[2,8] := ('        ');
DIGIT[2,9] := ('');

DIGIT[3,1] := ('  ');
DIGIT[3,2] := ('     ');
DIGIT[3,3] := ('        ');
DIGIT[3,4] := ('       ');
DIGIT[3,5] := ('   ');
DIGIT[3,6] := ('       ');
DIGIT[3,7] := ('        ');
DIGIT[3,8] := ('     ');
DIGIT[3,9] := ('  ');

DIGIT[4,1] := ('       ');
DIGIT[4,2] := ('       ');
DIGIT[4,3] := ('       ');
DIGIT[4,4] := ('       ');
DIGIT[4,5] := ('');
DIGIT[4,6] := ('        ');
DIGIT[4,7] := ('        ');
DIGIT[4,8] := ('        ');
DIGIT[4,9] := ('        ');
write('5');
DIGIT[5,1] := ('');
DIGIT[5,2] := ('        ');
DIGIT[5,3] := ('        ');
DIGIT[5,4] := (' ');
DIGIT[5,5] := ('       ');
DIGIT[5,6] := ('        ');
DIGIT[5,7] := ('        ');
DIGIT[5,8] := ('     ');
DIGIT[5,9] := ('  ');

DIGIT[6,1] := ('       ');
DIGIT[6,2] := ('       ');
DIGIT[6,3] := ('        ');
DIGIT[6,4] := (' ');
DIGIT[6,5] := ('     ');
DIGIT[6,6] := ('       ');
DIGIT[6,7] := ('       ');
DIGIT[6,8] := ('     ');
DIGIT[6,9] := ('  ');

DIGIT[7,1] := ('');
DIGIT[7,2] := ('        ');
DIGIT[7,3] := ('        ');
DIGIT[7,4] := ('        ');
DIGIT[7,5] := ('        ');
DIGIT[7,6] := ('        ');
DIGIT[7,7] := ('        ');
DIGIT[7,8] := ('        ');
DIGIT[7,9] := ('        ');

DIGIT[8,1] := ('  ');
DIGIT[8,2] := ('     ');
DIGIT[8,3] := ('       ');
DIGIT[8,4] := ('     ');
DIGIT[8,5] := ('  ');
DIGIT[8,6] := ('     ');
DIGIT[8,7] := ('       ');
DIGIT[8,8] := ('     ');
DIGIT[8,9] := ('  ');

DIGIT[9,1] := ('  ');
DIGIT[9,2] := ('     ');
DIGIT[9,3] := ('       ');
DIGIT[9,4] := ('     ');
DIGIT[9,5] := (' ');
DIGIT[9,6] := ('        ');
DIGIT[9,7] := ('        ');
DIGIT[9,8] := ('        ');
DIGIT[9,9] := ('        ');
END; { generate characters }

END. { my TPU }

