package org.apache.log4j.appender;

import java.util.Calendar;
import java.util.Locale;
import java.util.TimeZone;

import junit.framework.Assert;
import junit.framework.TestCase;

public class TestTimeBasedRollStrategy extends TestCase {

  private AppenderRollingProperties rollingProperties;

  protected void setUp() throws Exception {
    this.rollingProperties = new AppenderRollingProperties();
    this.rollingProperties.setDatePatternLocale(Locale.UK);
  }

  protected void tearDown() throws Exception {
    this.rollingProperties = null;
  }

  public void testDateTimePatternAsPerDRFA() {
    this.rollingProperties.setDatePattern("yyyy-MM-dd.'log'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);

    this.rollingProperties.setDatePattern("yyyy-MM-dd mm.'log'");
    strategy = TimeBasedRollStrategy.findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);

    this.rollingProperties.setDatePattern("yyyy-MM-dd a.'log'");
    strategy = TimeBasedRollStrategy.findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);

    this.rollingProperties.setDatePattern("yyyy-MM-dd HH.'log'");
    strategy = TimeBasedRollStrategy.findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);

    this.rollingProperties.setDatePattern("yyyy-MM.'log'");
    strategy = TimeBasedRollStrategy.findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);

    this.rollingProperties.setDatePattern("'log'HH'log'");
    strategy = TimeBasedRollStrategy.findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  // This code taken from the Log4J DRFATestCase.
  public void testRollEachDayAsPerDRFA() {
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy.ROLL_EACH_DAY;

    Calendar c = Calendar.getInstance();

    // jan, mar, may, july, aug, oct, dec have 31 days
    int[] M31 = { 0, 2, 4, 6, 7, 9, 11 };

    for (int i = 0; i < M31.length; i++) {
      for (int d = 1; d <= 31; d++) {
        for (int h = 0; h < 23; h++) {
          c.clear();
          c.set(Calendar.YEAR, 20);
          c.set(Calendar.MONTH, Calendar.JANUARY + M31[i]);
          c.set(Calendar.DAY_OF_MONTH, d);
          c.set(Calendar.HOUR_OF_DAY, h);
          c.set(Calendar.MINUTE, 10);
          c.set(Calendar.SECOND, 10);
          c.set(Calendar.MILLISECOND, 88);

          c.setTime(strategy.nextRollDate(c.getTime()));
          if (d == 31) {
            assertEquals(c.get(Calendar.MONTH),
                (Calendar.JANUARY + M31[i] + 1) % 12);
            assertEquals(c.get(Calendar.DAY_OF_MONTH), 1);
          } else {
            assertEquals(c.get(Calendar.MONTH), Calendar.JANUARY + M31[i]);
            assertEquals(c.get(Calendar.DAY_OF_MONTH), d + 1);
          }
          assertEquals(c.get(Calendar.HOUR_OF_DAY), 0);
          assertEquals(c.get(Calendar.MINUTE), 0);
          assertEquals(c.get(Calendar.SECOND), 0);
          assertEquals(c.get(Calendar.MILLISECOND), 0);
        }
      }
    }
  }

  // This code taken from the Log4J DRFATestCase.
  public void testRollEachHourAsPerDRFA() {
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy.ROLL_EACH_HOUR;

    Calendar c = Calendar.getInstance();
    TimeZone tz = c.getTimeZone();

    // jan, mar, may, july, aug, oct, dec have 31 days
    int[] M31 = { 0, 2, 4, 6, 7, 9, 11 };

    for (int i = 0; i < M31.length; i++) {
      for (int d = 1; d <= 31; d++) {
        for (int h = 0; h < 23; h++) {
          for (int m = 0; m <= 59; m++) {
            c.clear();
            c.set(Calendar.YEAR, 20);
            c.set(Calendar.MONTH, Calendar.JANUARY + M31[i]);
            c.set(Calendar.DAY_OF_MONTH, d);
            c.set(Calendar.HOUR_OF_DAY, h);
            c.set(Calendar.MINUTE, m);
            c.set(Calendar.SECOND, 12);
            c.set(Calendar.MILLISECOND, 88);

            boolean dltState0 = c.getTimeZone().inDaylightTime(c.getTime());
            c.setTime(strategy.nextRollDate(c.getTime()));
            boolean dltState1 = c.getTimeZone().inDaylightTime(c.getTime());

            assertEquals(c.get(Calendar.MILLISECOND), 0);
            assertEquals(c.get(Calendar.SECOND), 0);
            assertEquals(c.get(Calendar.MINUTE), 0);

            if (dltState0 == dltState1) {
              assertEquals(c.get(Calendar.HOUR_OF_DAY), (h + 1) % 24);
            } else {
              // returning to standard time
              if (dltState0) {
                assertEquals(c.get(Calendar.HOUR_OF_DAY), h);
              } else { // switching to day light saving time
                // System.err.println("m="+m+", h="+h+", d="+d+", i="+i);
                // if(h==2) {
                // System.err.println(c);
                // }
                // assertEquals(c.get(Calendar.HOUR_OF_DAY), (h+2)%24);
              }
            }

            if (h == 23) {
              assertEquals(c.get(Calendar.DAY_OF_MONTH), (d + 1) % 32);
              if (d == 31) {
                assertEquals(c.get(Calendar.MONTH),
                    (Calendar.JANUARY + M31[i] + 1) % 12);
              } else {
                assertEquals(c.get(Calendar.MONTH), Calendar.JANUARY + M31[i]);
              }
            } else {
              assertEquals(c.get(Calendar.DAY_OF_MONTH), d);
              assertEquals(c.get(Calendar.MONTH), Calendar.JANUARY + M31[i]);
            }
          }
        }
      }
    }
  }

  // This code taken from the Log4J DRFATestCase.
  public void testRollEachMinuteAsPerDRFA() {

    TimeBasedRollStrategy strategy = TimeBasedRollStrategy.ROLL_EACH_MINUTE;

    int[] S = { 0, 1, 5, 10, 21, 30, 59 };
    int[] M = { 0, 1, 5, 10, 21, 30, 59 };
    Calendar c = Calendar.getInstance();

    // jan, mar, may, july, aug, oct, dec have 31 days
    int[] M31 = { 2, 9, 0, 4, 6, 7, 11 };

    for (int i = 0; i < M31.length; i++) {
      for (int d = 1; d <= 31; d++) {
        for (int h = 0; h < 23; h++) {
          for (int m = 0; m < M.length; m++) {
            for (int s = 0; s < S.length; s++) {
              c.clear();
              c.set(Calendar.YEAR, 20);
              c.set(Calendar.MONTH, Calendar.JANUARY + M31[i]);
              c.set(Calendar.DAY_OF_MONTH, d);
              c.set(Calendar.HOUR_OF_DAY, h);
              c.set(Calendar.MINUTE, M[m]);
              c.set(Calendar.SECOND, S[s]);
              c.set(Calendar.MILLISECOND, 88);
              c.add(Calendar.MILLISECOND, 1);

              boolean dltState0 = c.getTimeZone().inDaylightTime(c.getTime());

              c.setTime(strategy.nextRollDate(c.getTime()));
              c.add(Calendar.MILLISECOND, 0);
              boolean dltState1 = c.getTimeZone().inDaylightTime(c.getTime());

              assertEquals(c.get(Calendar.MILLISECOND), 0);
              assertEquals(c.get(Calendar.SECOND), 0);
              assertEquals(c.get(Calendar.MINUTE), (M[m] + 1) % 60);

              if (M[m] == 59) {
                if (dltState0 == dltState1) {
                  assertEquals(c.get(Calendar.HOUR_OF_DAY), (h + 1) % 24);
                }
                if (h == 23) {
                  assertEquals(c.get(Calendar.DAY_OF_MONTH), (d + 1) % 32);
                  if (d == 31) {
                    assertEquals(c.get(Calendar.MONTH), (Calendar.JANUARY
                        + M31[i] + 1) % 12);
                  } else {
                    assertEquals(c.get(Calendar.MONTH), Calendar.JANUARY
                        + M31[i]);
                  }
                } else {
                  assertEquals(c.get(Calendar.DAY_OF_MONTH), d);
                }
              } else {
                // allow discrepancies only if we are switching from std to dls
                // time
                if (c.get(Calendar.HOUR_OF_DAY) != h) {
                  c.add(Calendar.HOUR_OF_DAY, +1);
                  boolean dltState2 = c.getTimeZone().inDaylightTime(
                      c.getTime());
                  if (dltState1 == dltState2) {
                    fail("No switch");
                  }
                }
                assertEquals(c.get(Calendar.DAY_OF_MONTH), d);
                assertEquals(c.get(Calendar.MONTH), Calendar.JANUARY + M31[i]);
              }
            }
          }
        }
      }
    }
  }

  public void testDateTimePatternMinutes() {
    this.rollingProperties.setDatePattern("mm");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);
  }

  public void testDateTimePatternMinutesQuotedBefore() {
    this.rollingProperties.setDatePattern("'.'mm");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);
  }

  public void testDateTimePatternMinutesQuotedAfter() {
    this.rollingProperties.setDatePattern("mm'.log'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);
  }

  public void testDateTimePatternMinutesQuotedAround() {
    this.rollingProperties.setDatePattern("'.'mm'.log'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);
  }

  public void testDateTimePatternMinutesQuotedContainingUnit() {
    this.rollingProperties.setDatePattern("mm'm'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);
  }

  public void testDateTimePatternMinutesQuotedContainingSmallerUnit() {
    this.rollingProperties.setDatePattern("mm'ss'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);
  }

  public void testDateTimePatternMinutesSmallerUnitBefore() {
    this.rollingProperties.setDatePattern("ssmm");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);
  }

  public void testDateTimePatternMinutesSmallerUnitAfter() {
    this.rollingProperties.setDatePattern("mmss");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);
  }

  public void testDateTimePatternMinutesLargerUnitBefore() {
    this.rollingProperties.setDatePattern("HHmm");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);
  }

  public void testDateTimePatternMinutesLargerUnitAfter() {
    this.rollingProperties.setDatePattern("mmHH");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);
  }

  public void testDateTimePatternMinutesLargerUnitBeforeSmallerUnitAfter() {
    this.rollingProperties.setDatePattern("HHmmss");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);
  }

  public void testDateTimePatternMinutesSmallerUnitBeforeLargerUnitAfter() {
    this.rollingProperties.setDatePattern("ssmmHH");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);
  }

  public void testDateTimePatternMinutesQuoted() {
    this.rollingProperties.setDatePattern("'mm'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_ERROR, strategy); // default
  }

  public void testDateTimePatternMinutesLargerUnitAndQuotedBefore() {
    this.rollingProperties.setDatePattern("HH'.'mm");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);
  }

  public void testDateTimePatternMinutesQuotedAndLargerUnitAfter() {
    this.rollingProperties.setDatePattern("mm'_'HH");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);
  }

  public void testDateTimePatternMinutesUninterpretedChars() {
    this.rollingProperties.setDatePattern("_mm.log");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MINUTE, strategy);
  }

  public void testDateTimePatternHoursInDay0_23() {
    this.rollingProperties.setDatePattern("HH");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursInDay1_24() {
    this.rollingProperties.setDatePattern("kk");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursInDay0_11() {
    this.rollingProperties.setDatePattern("KK");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursInDay1_12() {
    this.rollingProperties.setDatePattern("hh");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursQuotedBefore() {
    this.rollingProperties.setDatePattern("'.'HH");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursQuotedAfter() {
    this.rollingProperties.setDatePattern("hh'.log'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursQuotedAround() {
    this.rollingProperties.setDatePattern("'.'kk'.log'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursQuotedContainingUnit() {
    this.rollingProperties.setDatePattern("HH'K'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursQuotedContainingSmallerUnit() {
    this.rollingProperties.setDatePattern("HH'mm'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursSmallerUnitBefore() {
    this.rollingProperties.setDatePattern("ssHH");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursSmallerUnitAfter() {
    this.rollingProperties.setDatePattern("kkss");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursLargerUnitBefore() {
    this.rollingProperties.setDatePattern("DDHH");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursLargerUnitAfter() {
    this.rollingProperties.setDatePattern("HHDD");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursLargerUnitBeforeSmallerUnitAfter() {
    this.rollingProperties.setDatePattern("DDHHss");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursSmallerUnitBeforeLargerUnitAfter() {
    this.rollingProperties.setDatePattern("ssHHM");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursQuoted() {
    this.rollingProperties.setDatePattern("'kk'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_ERROR, strategy); // default
  }

  public void testDateTimePatternHoursLargerUnitAndQuotedBefore() {
    this.rollingProperties.setDatePattern("DD'.'hh");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursQuotedAndLargerUnitAfter() {
    this.rollingProperties.setDatePattern("KK'_'DD");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursInDay0_11UninterpretedChars() {
    this.rollingProperties.setDatePattern("_KK.log");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursInDay1_12UninterpretedChars() {
    this.rollingProperties.setDatePattern("_hh.log");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursInDay0_23UninterpretedChars() {
    this.rollingProperties.setDatePattern("_HH.log");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHoursInDay1_24UninterpretedChars() {
    this.rollingProperties.setDatePattern("_kk.log");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HOUR, strategy);
  }

  public void testDateTimePatternHalfDay() {
    this.rollingProperties.setDatePattern("a");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);
  }

  public void testDateTimePatternHalfDayQuotedBefore() {
    this.rollingProperties.setDatePattern("'.'a");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);
  }

  public void testDateTimePatternHalfDayQuotedAfter() {
    this.rollingProperties.setDatePattern("a'.log'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);
  }

  public void testDateTimePatternHalfDayQuotedAround() {
    this.rollingProperties.setDatePattern("'.'a'.log'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);
  }

  public void testDateTimePatternHalfDayQuotedContainingUnit() {
    this.rollingProperties.setDatePattern("a'a'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);
  }

  public void testDateTimePatternHalfDayQuotedContainingSmallerUnit() {
    this.rollingProperties.setDatePattern("a'HH'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);
  }

  public void testDateTimePatternHalfDaySmallerUnitBefore() {
    this.rollingProperties.setDatePattern("ssa");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);
  }

  public void testDateTimePatternHalfDaySmallerUnitAfter() {
    this.rollingProperties.setDatePattern("ass");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);
  }

  public void testDateTimePatternHalfDayLargerUnitBefore() {
    this.rollingProperties.setDatePattern("DDa");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);
  }

  public void testDateTimePatternHalfDayLargerUnitAfter() {
    this.rollingProperties.setDatePattern("aDD");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);
  }

  public void testDateTimePatternHalfDayLargerUnitBeforeSmallerUnitAfter() {
    this.rollingProperties.setDatePattern("DDass");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);
  }

  public void testDateTimePatternHalfDaySmallerUnitBeforeLargerUnitAfter() {
    this.rollingProperties.setDatePattern("ssaM");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);
  }

  public void testDateTimePatternHalfDayQuoted() {
    this.rollingProperties.setDatePattern("'a'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_ERROR, strategy); // default
  }

  public void testDateTimePatternHalfDayLargerUnitAndQuotedBefore() {
    this.rollingProperties.setDatePattern("DD'.'a");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);
  }

  public void testDateTimePatternHalfDayQuotedAndLargerUnitAfter() {
    this.rollingProperties.setDatePattern("a'_'DD");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);
  }

  public void testDateTimePatternHalfDayUninterpretedChars() {
    this.rollingProperties.setDatePattern("_a.log");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_HALF_DAY, strategy);
  }

  public void testDateTimePatternDayInYear() {
    this.rollingProperties.setDatePattern("DD");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDayInMonth() {
    this.rollingProperties.setDatePattern("dd");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDayOfWeekInMonth() {
    this.rollingProperties.setDatePattern("FF");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDayInWeek() {
    this.rollingProperties.setDatePattern("EEE");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDaysQuotedBefore() {
    this.rollingProperties.setDatePattern("'.'dd");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDaysQuotedAfter() {
    this.rollingProperties.setDatePattern("DD'.log'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDaysQuotedAround() {
    this.rollingProperties.setDatePattern("'.'EEE'.log'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDaysQuotedContainingUnit() {
    this.rollingProperties.setDatePattern("dd'EEE'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDaysQuotedContainingSmallerUnit() {
    this.rollingProperties.setDatePattern("FF'HH'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDaysSmallerUnitBefore() {
    this.rollingProperties.setDatePattern("ssdd");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDaysSmallerUnitAfter() {
    this.rollingProperties.setDatePattern("Fss");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDaysLargerUnitBefore() {
    this.rollingProperties.setDatePattern("wwdd");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDaysLargerUnitAfter() {
    this.rollingProperties.setDatePattern("DDww");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDaysLargerUnitBeforeSmallerUnitAfter() {
    this.rollingProperties.setDatePattern("Wddss");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDaysSmallerUnitBeforeLargerUnitAfter() {
    this.rollingProperties.setDatePattern("ssEEEM");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDaysQuoted() {
    this.rollingProperties.setDatePattern("'DdFE'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_ERROR, strategy); // default
  }

  public void testDateTimePatternDaysLargerUnitAndQuotedBefore() {
    this.rollingProperties.setDatePattern("W'.'F");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDaysQuotedAndLargerUnitAfter() {
    this.rollingProperties.setDatePattern("EEE'_'ww");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDayInMonthUninterpretedChars() {
    this.rollingProperties.setDatePattern("_dd.log");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDayInWeekUninterpretedChars() {
    this.rollingProperties.setDatePattern("_EEE.log");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDayInYearUninterpretedChars() {
    this.rollingProperties.setDatePattern("_DDD.log");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDayOfWeekInMonthUninterpretedChars() {
    this.rollingProperties.setDatePattern("_F.log");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternWeekInYear() {
    this.rollingProperties.setDatePattern("ww");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeekInMonth() {
    this.rollingProperties.setDatePattern("W");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeeksQuotedBefore() {
    this.rollingProperties.setDatePattern("'.'ww");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeeksQuotedAfter() {
    this.rollingProperties.setDatePattern("ww'.log'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeeksQuotedAround() {
    this.rollingProperties.setDatePattern("'.'ww'.log'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeeksQuotedContainingUnit() {
    this.rollingProperties.setDatePattern("ww'W'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeeksQuotedContainingSmallerUnit() {
    this.rollingProperties.setDatePattern("W'dd'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeeksSmallerUnitBefore() {
    this.rollingProperties.setDatePattern("ssW");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeeksSmallerUnitAfter() {
    this.rollingProperties.setDatePattern("wwss");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeeksLargerUnitBefore() {
    this.rollingProperties.setDatePattern("Mww");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeeksLargerUnitAfter() {
    this.rollingProperties.setDatePattern("Wyy");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeeksLargerUnitBeforeSmallerUnitAfter() {
    this.rollingProperties.setDatePattern("Mwwss");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeeksSmallerUnitBeforeLargerUnitAfter() {
    this.rollingProperties.setDatePattern("ssWyy");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeeksQuoted() {
    this.rollingProperties.setDatePattern("'Ww'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_ERROR, strategy); // default
  }

  public void testDateTimePatternWeeksLargerUnitAndQuotedBefore() {
    this.rollingProperties.setDatePattern("M'.'ww");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeeksQuotedAndLargerUnitAfter() {
    this.rollingProperties.setDatePattern("W'_'M");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeekInMonthUninterpretedChars() {
    this.rollingProperties.setDatePattern("_W.log");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternWeekInYearUninterpretedChars() {
    this.rollingProperties.setDatePattern("_ww.log");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_WEEK, strategy);
  }

  public void testDateTimePatternMonths02() {
    this.rollingProperties.setDatePattern("MM");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternMonths03() {
    this.rollingProperties.setDatePattern("MMM");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternMonthsQuotedBefore() {
    this.rollingProperties.setDatePattern("'.'MM");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternMonthsQuotedAfter() {
    this.rollingProperties.setDatePattern("MM'.log'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternMonthsQuotedAround() {
    this.rollingProperties.setDatePattern("'.'MMM'.log'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternMonthsQuotedContainingUnit() {
    this.rollingProperties.setDatePattern("MM'M'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternMonthsQuotedContainingSmallerUnit() {
    this.rollingProperties.setDatePattern("MM'ww'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternMonthsSmallerUnitBefore() {
    this.rollingProperties.setDatePattern("ssMM");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternMonthsSmallerUnitAfter() {
    this.rollingProperties.setDatePattern("MMss");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternMonthsLargerUnitBefore() {
    this.rollingProperties.setDatePattern("yyMM");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternMonthsLargerUnitAfter() {
    this.rollingProperties.setDatePattern("MMyy");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternMonthsLargerUnitBeforeSmallerUnitAfter() {
    this.rollingProperties.setDatePattern("yyMMMss");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternMonthsSmallerUnitBeforeLargerUnitAfter() {
    this.rollingProperties.setDatePattern("ssMMyy");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternMonthsQuoted() {
    this.rollingProperties.setDatePattern("'MMM'");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_ERROR, strategy); // default
  }

  public void testDateTimePatternMonthsLargerUnitAndQuotedBefore() {
    this.rollingProperties.setDatePattern("yy'.'MMM");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternMonthsQuotedAndLargerUnitAfter() {
    this.rollingProperties.setDatePattern("MMM'_'yy");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternMonthsUninterpretedChars() {
    this.rollingProperties.setDatePattern("_MM.log");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_MONTH, strategy);
  }

  public void testDateTimePatternDayInMonthInYearGermanLocale() {
    this.rollingProperties.setDatePatternLocale(Locale.GERMANY);
    this.rollingProperties.setDatePattern("'.'uuuu-MM-tt");
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testDateTimePatternDayInMonthGermanLocale() {
    this.rollingProperties.setDatePatternLocale(Locale.GERMAN);
    this.rollingProperties.setDatePattern("tt"); // valid in GERMAN locale, not ENGLISH
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_EACH_DAY, strategy);
  }

  public void testIncorrectDateTimePatternDayInMonthGermanLocale() {
    this.rollingProperties.setDatePatternLocale(Locale.GERMAN);
    this.rollingProperties.setDatePattern("dd"); // valid in ENGLISH locale, not GERMAN
    TimeBasedRollStrategy strategy = TimeBasedRollStrategy
        .findRollStrategy(this.rollingProperties);
    Assert.assertEquals(TimeBasedRollStrategy.ROLL_ERROR, strategy);
  }
}
