/* 
 * Copyright 1999,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.log4j.appender;

import java.io.File;
import java.util.Calendar;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import junit.framework.Assert;
import junit.framework.TestCase;

import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.Layout;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.helpers.FileHelper;

public class TestTimeAndSizeRollingAppender extends TestCase {

  private final FileTestHelper fileTestHelper = new FileTestHelper();

  private Layout layout = null;

  protected void setUp() throws Exception {
    // bin current logging configuration, clearing extant appenders.
    BasicConfigurator.resetConfiguration();
    this.fileTestHelper.setUp();
    PatternLayout patternLayout = new PatternLayout();
    patternLayout.setConversionPattern("%d %p %m");
    this.layout = patternLayout;
  }

  protected void tearDown() throws Exception {
    BasicConfigurator.resetConfiguration();
    this.fileTestHelper.tearDown();
  }

  public void testLogRollingByFileSize() {

    int maxBackups = 5;
    Logger logger = Logger.getRootLogger();

    TimeAndSizeRollingAppender appender = null;
    appender = new TimeAndSizeRollingAppender();
    appender.setAppend(true);
    appender.setBufferedIO(false);
    appender.setFile(this.fileTestHelper.getTestDir()
        + System.getProperty("file.separator") + "log.txt");
    appender.setLayout(this.layout);
    appender.setMaxFileSize("4KB"); // small enough to roll frequently
    appender.setMaxRollFileCount(maxBackups);
    appender.setName("composite_roll_test");
    appender.setDatePattern("'.'mm");
    appender.setScavengeInterval(100L);

    appender.activateOptions();

    BasicConfigurator.configure(appender);

    // roughly 33KB
    for (int i = 0; i < 128; i++) {
      logger.info(this.fileTestHelper.generateBlah());
    }

    try {
      Thread.sleep(250L); // wait longer than the configured period
    } catch (InterruptedException e) {
      Thread.currentThread().interrupt();
    }

    // make sure the appender is finished writing to the stream, and close it
    // down
    BasicConfigurator.resetConfiguration();

    // backup files (e.g. log.txt.00, log.txt.01, etc) plus working file (i.e.
    // log.txt)
    int expectedMaxNumberOfLogFiles = appender.getMaxRollFileCount() + 1;

    File testDir = new File(this.fileTestHelper.getTestDir());
    if (testDir.exists()) {
      File[] logFiles = testDir.listFiles();
      assertNotNull(logFiles);
      assertTrue(logFiles.length > 1);
      assertTrue(logFiles.length == expectedMaxNumberOfLogFiles);

      // look for files with the '.mm', or '.mm.digit' suffix
      Pattern pattern = Pattern.compile("log.txt.\\d{2}(.\\d{1,2})?\\Z");
      for (int i = 0; i < logFiles.length; i++) {
        File logFile = logFiles[i];
        assertNotNull(logFile);
        if (!"log.txt".equals(logFile.getName())) {
          Matcher matcher = pattern.matcher(logFile.getName());
          assertTrue(matcher.matches());
        }
      }
    }
  }

  public static class MockFileRollEventListener implements
      FileRollEventListener {

    private int callCount = 0;

    public MockFileRollEventListener() {
      super();
    }

    public void onFileRoll(FileRollEvent fileRollEvent) {
      this.callCount++;
      fileRollEvent.dispatchToAppender();
      fileRollEvent
          .dispatchToAppender("Custom test message at the beginning of the new file");
    }

    boolean isCalled() {
      return (this.callCount > 0);
    }

    int getCallCount() {
      return this.callCount;
    }
  };

  public void testCustomLogRollEventListener() {

    int maxBackups = 5;
    Logger logger = Logger.getRootLogger();

    TimeAndSizeRollingAppender appender = null;
    appender = new TimeAndSizeRollingAppender();
    appender.setAppend(true);
    appender.setBufferedIO(false);
    appender.setFile(this.fileTestHelper.getTestDir()
        + System.getProperty("file.separator") + "log.txt");
    appender.setLayout(this.layout);
    appender.setMaxFileSize("4KB"); // small enough to roll frequently
    appender.setMaxRollFileCount(maxBackups);
    appender.setName("composite_roll_test");
    appender.setDatePattern("'.'mm");
    appender.setScavengeInterval(-1);
    appender
        .setFileRollEventListener(MockFileRollEventListener.class.getName());
    MockFileRollEventListener listener = (MockFileRollEventListener) appender
        .getGuestFileRollEventListener();

    appender.activateOptions();

    BasicConfigurator.configure(appender);

    // roughly 33KB
    for (int i = 0; i < 128; i++) {
      logger.info(this.fileTestHelper.generateBlah());
    }

    try {
      Thread.sleep(250L); // wait longer than the configured period
    } catch (InterruptedException e) {
      Thread.currentThread().interrupt();
    }

    // make sure the appender is finished writing to the stream, and close it
    // down
    BasicConfigurator.resetConfiguration();

    Assert.assertTrue(listener.isCalled());
    Assert.assertEquals(9, listener.getCallCount());
  }

  public void testLogRollingBySizeAndPerMinute() {

    int maxBackups = 5;
    Logger logger = Logger.getRootLogger();

    TimeAndSizeRollingAppender appender = null;
    appender = new TimeAndSizeRollingAppender();
    appender.setAppend(true);
    appender.setBufferedIO(false);
    appender.setFile(this.fileTestHelper.getTestDir()
        + System.getProperty("file.separator") + "log.txt");
    appender.setLayout(this.layout);
    appender.setMaxFileSize("16KB");
    appender.setMaxRollFileCount(maxBackups);
    appender.setName("composite_roll_test");
    appender.setDatePattern("'.'mm");
    appender.setDatePatternLocale("en_GB");
    appender.setScavengeInterval(100L);

    appender.activateOptions();

    BasicConfigurator.configure(appender);

    // roughly 2.5 mins
    for (int i = 0; i < 30; i++) {
      // roughly 3.6KB
      for (int k = 0; k < 14; k++) {
        logger.info(this.fileTestHelper.generateBlah());
      }
      try {
        Thread.sleep(5000L);
      } catch (InterruptedException e) {
        Thread.currentThread().interrupt();
        break;
      }
    }

    try {
      Thread.sleep(1000L); // wait longer than the configured period
    } catch (InterruptedException e) {
      Thread.currentThread().interrupt();
    }

    // make sure the appender is finished writing to the stream, and close it
    // down
    BasicConfigurator.resetConfiguration();

    // backups plus working file
    int expectedMaxNumberOfLogFiles = appender.getMaxRollFileCount() + 1;

    File testDir = new File(this.fileTestHelper.getTestDir());
    if (testDir.exists()) {
      File[] logFiles = testDir.listFiles();
      assertNotNull(logFiles);
      assertTrue(logFiles.length > 1);
      assertTrue(logFiles.length == expectedMaxNumberOfLogFiles);

      // look for files with the '.mm', or '.mm.digit' suffix
      Pattern pattern = Pattern.compile("log.txt.\\d{2}(.\\d{1,2})?\\Z");
      for (int i = 0; i < logFiles.length; i++) {
        File logFile = logFiles[i];
        assertNotNull(logFile);
        if (!"log.txt".equals(logFile.getName())) {
          Matcher matcher = pattern.matcher(logFile.getName());
          assertTrue(matcher.matches());
        }
      }
    }
  }

  public void testForeignLogFiles() {

    this.fileTestHelper.writeFile("foreign.log");

    int maxBackups = 3;
    Logger logger = Logger.getRootLogger();

    TimeAndSizeRollingAppender appender = null;
    appender = new TimeAndSizeRollingAppender();
    appender.setAppend(true);
    appender.setBufferedIO(false);
    appender.setFile(this.fileTestHelper.getTestDir()
        + System.getProperty("file.separator") + "log.txt");
    appender.setLayout(this.layout);
    appender.setMaxFileSize("4KB");
    appender.setMaxRollFileCount(maxBackups);
    appender.setName("composite_roll_test");
    appender.setDatePattern("'.'mm");
    appender.setScavengeInterval(100L); // test

    appender.activateOptions();

    BasicConfigurator.configure(appender);

    // roughly 20KB
    for (int i = 0; i < 80; i++) {
      logger.info(this.fileTestHelper.generateBlah());
    }

    try {
      Thread.sleep(250L); // wait longer than the configured period
    } catch (InterruptedException e) {
      Thread.currentThread().interrupt();
    }

    // make sure the appender is finished writing to the stream, and close it
    // down
    BasicConfigurator.resetConfiguration();

    // backups plus working file, plus foreign file
    int expectedMaxNumberOfLogFiles = appender.getMaxRollFileCount() + 2;

    File testDir = new File(this.fileTestHelper.getTestDir());
    if (testDir.exists()) {
      File[] logFiles = testDir.listFiles();
      assertNotNull(logFiles);
      assertTrue(logFiles.length > 1);
      assertTrue(logFiles.length == expectedMaxNumberOfLogFiles);

      // look for files with the '.mm', or '.mm.digit' suffix
      Pattern pattern = Pattern.compile("log.txt.\\d{2}(.\\d{1,2})?");

      for (int i = 0; i < logFiles.length; i++) {
        File logFile = logFiles[i];
        assertNotNull(logFile);
        if ((!"log.txt".equals(logFile.getName()))
            && (!"foreign.log".equals(logFile.getName()))) {
          Matcher matcher = pattern.matcher(logFile.getName());
          assertTrue(matcher.matches());
        }
      }
    }
  }

  public void testMultipleAppenders() {

    int maxBackups = 3;
    Logger logger = Logger.getRootLogger();

    TimeAndSizeRollingAppender firstAppender = null;
    firstAppender = new TimeAndSizeRollingAppender();
    firstAppender.setAppend(true);
    firstAppender.setBufferedIO(false);
    firstAppender.setFile(this.fileTestHelper.getTestDir()
        + System.getProperty("file.separator") + "appender1.log");
    firstAppender.setLayout(this.layout);
    firstAppender.setMaxFileSize("8KB"); // test
    firstAppender.setMaxRollFileCount(maxBackups);
    firstAppender.setName("First Appender");
    firstAppender.setDatePattern("'.'mm");
    firstAppender.setScavengeInterval(100L);

    TimeAndSizeRollingAppender secondAppender = null;
    secondAppender = new TimeAndSizeRollingAppender();
    secondAppender.setAppend(true);
    secondAppender.setBufferedIO(false);
    secondAppender.setFile(this.fileTestHelper.getTestDir()
        + System.getProperty("file.separator") + "appender2.log");
    secondAppender.setLayout(this.layout);
    secondAppender.setMaxFileSize("1KB"); // test
    secondAppender.setMaxRollFileCount(maxBackups);
    secondAppender.setName("Second Appender");
    secondAppender.setDatePattern("'.'mm");
    secondAppender.setScavengeInterval(100L);

    firstAppender.activateOptions();
    BasicConfigurator.configure(firstAppender);
    secondAppender.activateOptions();
    BasicConfigurator.configure(secondAppender);

    // roughly 55KB
    for (int i = 0; i < 220; i++) {
      logger.info(this.fileTestHelper.generateBlah());
    }

    try {
      Thread.sleep(250L); // wait longer than the configured period
    } catch (InterruptedException e) {
      Thread.currentThread().interrupt();
    }

    // make sure the appender is finished writing to the stream, and close it
    // down
    BasicConfigurator.resetConfiguration();

    // backups plus working files, plus second appender's logs
    int expectedMaxNumberOfLogFiles = 2 + (firstAppender.getMaxRollFileCount() * 2);

    File testDir = new File(this.fileTestHelper.getTestDir());
    if (testDir.exists()) {
      File[] logFiles = testDir.listFiles();
      assertNotNull(logFiles);
      assertTrue(logFiles.length > 1);
      assertTrue(logFiles.length == expectedMaxNumberOfLogFiles);

      // look for files with the '.mm', or '.mm.digit' suffix
      Pattern pattern1 = Pattern.compile("appender1.log.\\d{2}(.\\d{1,2})?");
      Pattern pattern2 = Pattern.compile("appender2.log.\\d{2}(.\\d{1,2})?");

      int count1 = 0;
      int count2 = 0;
      // suffix
      for (int i = 0; i < logFiles.length; i++) {
        File logFile = logFiles[i];
        assertNotNull(logFile);
        if (!(logFile.getName().equals("appender1.log") || logFile.getName()
            .equals("appender2.log"))) {
          Matcher matcher = pattern1.matcher(logFile.getName());
          if (matcher.matches()) {
            count1++;
          } else {
            matcher = pattern2.matcher(logFile.getName());
            if (matcher.matches()) {
              count2++;
            }
          }
        }
      }
      assertEquals(maxBackups, count1);
      assertEquals(maxBackups, count2);
    }
  }

  public void testPreExistingLogFiles() {

    Calendar cal = Calendar.getInstance();
    this.fileTestHelper.writeFile("log.txt");
    int minute = cal.get(Calendar.MINUTE);
    this.fileTestHelper.writeFile("log.txt." + ((minute < 10) ? "0" : "")
        + minute);
    this.fileTestHelper.writeFile("log.txt." + ((++minute < 10) ? "0" : "")
        + (minute + 1));
    this.fileTestHelper.writeFile("log.txt." + ((minute < 10) ? "0" : "")
        + (minute + 1) + ".1");
    try {
      // wait a minute
      Thread.sleep(61 * 1000);
    } catch (InterruptedException e) {
      Thread.currentThread().interrupt();
      Assert.fail("Test interrupted");
    }

    int maxBackups = 5;
    Logger logger = Logger.getRootLogger();

    TimeAndSizeRollingAppender appender = null;
    appender = new TimeAndSizeRollingAppender();
    appender.setAppend(true);
    appender.setBufferedIO(false);
    appender.setFile(this.fileTestHelper.getTestDir()
        + System.getProperty("file.separator") + "log.txt");
    appender.setLayout(this.layout);
    appender.setMaxFileSize("1KB");
    appender.setMaxRollFileCount(maxBackups);
    appender.setName("composite_roll_test");
    appender.setDatePattern("'.'mm");
    appender.setScavengeInterval(100L); // test

    appender.activateOptions();

    BasicConfigurator.configure(appender);

    // roughly 7KB
    for (int i = 0; i < 28; i++) {
      logger.info(this.fileTestHelper.generateBlah());
    }

    try {
      Thread.sleep(250L); // wait longer than the configured period
    } catch (InterruptedException e) {
      Thread.currentThread().interrupt();
    }

    // make sure the appender is finished writing to the stream, and close it
    // down
    BasicConfigurator.resetConfiguration();

    // backups plus working file
    int expectedMaxNumberOfLogFiles = appender.getMaxRollFileCount() + 1;

    File testDir = new File(this.fileTestHelper.getTestDir());
    if (testDir.exists()) {
      File[] logFiles = testDir.listFiles();
      assertNotNull(logFiles);
      assertTrue(logFiles.length > 1);
      assertTrue(logFiles.length == expectedMaxNumberOfLogFiles);

      // look for files with the '.mm', or '.mm.digit' suffix
      Pattern pattern = Pattern.compile("log.txt.\\d{2}(.\\d{1,2})?");

      for (int i = 0; i < logFiles.length; i++) {
        File logFile = logFiles[i];
        assertNotNull(logFile);
        if (!"log.txt".equals(logFile.getName())) {
          Matcher matcher = pattern.matcher(logFile.getName());
          assertTrue(matcher.matches());
        }
      }
    }
  }

  public void testPreExistingLogFilesRollableBySize() {

    this.fileTestHelper.writeFile("log.txt", 256);
    this.fileTestHelper.writeFile("log.txt.01", 256);
    this.fileTestHelper.writeFile("log.txt.02", 256);
    this.fileTestHelper.writeFile("log.txt.02.1", 256);

    int maxBackups = 5;
    Logger logger = Logger.getRootLogger();

    TimeAndSizeRollingAppender appender = null;
    appender = new TimeAndSizeRollingAppender();
    appender.setAppend(true);
    appender.setBufferedIO(false);
    appender.setFile(this.fileTestHelper.getTestDir()
        + System.getProperty("file.separator") + "log.txt");
    appender.setLayout(this.layout);
    appender.setMaxFileSize("1KB");
    appender.setMaxRollFileCount(maxBackups);
    appender.setName("composite_roll_test");
    appender.setDatePattern("'.'mm");
    appender.setScavengeInterval(100L); // test

    appender.activateOptions();

    BasicConfigurator.configure(appender);

    // roughly 3KB
    for (int i = 0; i < 12; i++) {
      logger.info(this.fileTestHelper.generateBlah());
    }

    try {
      Thread.sleep(250L); // wait longer than the configured period
    } catch (InterruptedException e) {
      Thread.currentThread().interrupt();
    }

    // make sure the appender is finished writing to the stream, and close it
    // down
    BasicConfigurator.resetConfiguration();

    // backups plus working file
    int expectedMaxNumberOfLogFiles = appender.getMaxRollFileCount() + 1;

    File testDir = new File(this.fileTestHelper.getTestDir());
    if (testDir.exists()) {
      File[] logFiles = testDir.listFiles();
      assertNotNull(logFiles);
      assertTrue(logFiles.length > 1);
      assertTrue(logFiles.length == expectedMaxNumberOfLogFiles);

      // look for files with the '.mm', or '.mm.digit' suffix
      Pattern pattern = Pattern.compile("log.txt.\\d{2}(.\\d{1,2})?");

      for (int i = 0; i < logFiles.length; i++) {
        File logFile = logFiles[i];
        assertNotNull(logFile);
        if (!"log.txt".equals(logFile.getName())) {
          Matcher matcher = pattern.matcher(logFile.getName());
          assertTrue(matcher.matches());
        }
      }
    }
  }

  public void testPreExistingLogFilesDoNotCauseLargerThanExpectedStartupLogFile() {

    this.fileTestHelper.writeFile("log.txt", 12); // 3096 bytes

    int maxBackups = 5;
    Logger logger = Logger.getRootLogger();

    TimeAndSizeRollingAppender appender = null;
    appender = new TimeAndSizeRollingAppender();
    appender.setAppend(true);
    appender.setBufferedIO(false);
    String filename = this.fileTestHelper.getTestDir()
        + System.getProperty("file.separator") + "log.txt";
    File logFile = new File(filename);
    appender.setFile(filename);
    appender.setLayout(this.layout);
    appender.setMaxFileSize("4KB"); // 4096 bytes
    appender.setMaxRollFileCount(maxBackups);
    appender.setName("composite_roll_test");
    appender.setDatePattern("'.'MM"); // monthly roll to effectively eliminate
    // roll by time from this test
    appender.setScavengeInterval(100L); // test

    appender.activateOptions();

    BasicConfigurator.configure(appender);

    // 1548 bytes
    for (int i = 0; i < 6; i++) {
      logger.info(this.fileTestHelper.generateBlah());
      Assert.assertTrue(logFile.exists());
      Assert.assertTrue(logFile.canRead());
      // assert less than 4096 + 258
      Assert.assertTrue("log file exceeded permitted maximum size", (logFile
          .length() < 4400));
    }

    // make sure the appender is finished writing to the stream, and close it
    // down
    BasicConfigurator.resetConfiguration();

    // backups plus working file
    int expectedMaxNumberOfLogFiles = 2;

    File testDir = new File(this.fileTestHelper.getTestDir());
    if (testDir.exists()) {
      File[] logFiles = testDir.listFiles();
      assertNotNull(logFiles);
      assertTrue(logFiles.length > 1);
      assertTrue(logFiles.length == expectedMaxNumberOfLogFiles);

      // look for files with the '.mm', or '.mm.digit' suffix
      Pattern pattern = Pattern.compile("log.txt.\\d{2}(.\\d{1,2})?");

      for (int i = 0; i < logFiles.length; i++) {
        logFile = logFiles[i];
        assertNotNull(logFile);
        if (!"log.txt".equals(logFile.getName())) {
          Matcher matcher = pattern.matcher(logFile.getName());
          assertTrue(matcher.matches());
        }
      }
    }
  }

  public void testBackupLogZipCompression() {

    int maxBackups = 5;
    Logger logger = Logger.getRootLogger();

    TimeAndSizeRollingAppender appender = null;
    appender = new TimeAndSizeRollingAppender();
    appender.setAppend(true);
    appender.setBufferedIO(false);
    appender.setFile(this.fileTestHelper.getTestDir()
        + System.getProperty("file.separator") + "log.txt");
    appender.setLayout(this.layout);
    appender.setMaxFileSize("4KB"); // small enough to roll frequently
    appender.setMaxRollFileCount(maxBackups);
    appender.setName("composite_roll_test");
    appender.setDatePattern("'.'mm");
    appender.setScavengeInterval(-1); // disable scavenging
    appender.setCompressionAlgorithm("ZIP");

    appender.activateOptions();

    BasicConfigurator.configure(appender);

    // roughly 33KB
    for (int i = 0; i < 128; i++) {
      logger.info(this.fileTestHelper.generateBlah());
    }

    try {
      Thread.sleep(500L); // wait long enough for compression
    } catch (InterruptedException e) {
      Thread.currentThread().interrupt();
    }

    // make sure the appender is finished writing to the stream, and close it
    // down
    BasicConfigurator.resetConfiguration();

    File testDir = new File(this.fileTestHelper.getTestDir());
    assertTrue(testDir.exists());
    File[] logFiles = testDir.listFiles();
    assertNotNull(logFiles);
    assertTrue(logFiles.length > 1);
    for (int i = 0; i < logFiles.length; i++) {
      if (logFiles[i].getName().endsWith(".zip")) {
        Assert.assertTrue(new FileHelper().isZip(logFiles[i]));
      } else {
        Assert.assertEquals("log.txt", logFiles[i].getName());
      }
    }
  }
}
