/*
 *
 *	demod_piano.c - piano tone demodulator
 *	for tuning programme
 *	by Guenther Montag
 *	inspired by:
 *
 *      demod_zvei.c -- ZVEI signalling demodulator/decoder
 *      Copyright (C) 1996  
 *          Thomas Sailer (sailer@ife.ee.ethz.ch, hb9jnx@hb9w.che.eu)
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* ---------------------------------------------------------------------- */

#include "term.h"
#include "piano.h"
#include "filter.h"
#include <math.h>
#include <string.h>
#include <ctype.h>

/* ---------------------------------------------------------------------- */

int tabsize;
float piano_freq[TONETABSIZE];
//float pianonoise = 0.9;
float pianonoise = 0.999999;
//float pianonoise = 1;
float bias; 
char foundnote[8];

static void generate_piano_freqtable(int tabsize)
{
	int i;
	verbinfo ("piano tuning table modified by:"
	    "bias %1.4f, soundcorr %1.4f and macro PHINC:\n", 
	    bias, soundcorr);
	for (i = 0; i < tabsize; i++) {
	    piano_freq[i] = PHINC(bias * tonetab[i]);
	    //verbinfo ("%f\t", piano_freq[i]);
	    //if (i % 5 == 0)  verbinfo ("\n");
	}
	verbinfo ("lowest frequency in table: %f\n", 
	    tonetab[0]);
	verbinfo ("lowest bias-corrected frequency in table: %f\n", 
	    bias*tonetab[0]);
	verbinfo ("lowest bias- and soundcorr- corrected frequency in table: %f\n", 
	    bias/soundcorr*tonetab[0]);
	verbinfo ("highest frequency in table: %f\n", 
	    tonetab[i-1]);
	verbinfo ("highest bias-corrected frequency in table: %f\n", 
	    bias*tonetab[i-1]);
	verbinfo ("highest bias- and soundcorr- corrected frequency in table: %f\n", 
	    bias/soundcorr*tonetab[i-1]);
}
	
/* ---------------------------------------------------------------------- */

static void piano_init(struct demod_state *s)
{
	tabsize = TONETABSIZE;
	memset(&s->l1.piano, 0, sizeof(s->l1.piano));
	generate_piano_freqtable(tabsize);
	statusinfo(1, "note detection %s", params);
	statusinfo(2, "");
}

/* ---------------------------------------------------------------------- */

int find_max_idx(const float *f, float noise, int tabsize)
{
	float en = 0;
	int idx = -1, i;

	for (i = 0; i < tabsize; i++) {
	    if (f[i] > en) {
		en = f[i];
		idx = i;
	    }
	}
	if (idx < 0) {
	    verbinfo("max_idx: no energy.\n");
	    return -1;
	}
	en *= noise;
	for (i = 0; i < tabsize; i++)
	    if (idx != i && f[i] > en) {
		verbinfo("max_idx: noise\n");
		return -1;
	    }
	return idx;
}

/* ---------------------------------------------------------------------- */

char* tone_number_to_name(int tone)
{
	int n = tone % 12;
	if 	(n ==  0) return (    "a");
	if 	(n ==  1) return (    "a#");
	if 	(n ==  2) return (    "b");
	if 	(n ==  3) return (    "c");
	if 	(n ==  4) return (    "c#");
	if 	(n ==  5) return (    "d");
	if 	(n ==  6) return (    "d#");
	if 	(n ==  7) return (    "e");
	if 	(n ==  8) return (    "f");
	if 	(n ==  9) return (    "f#");
	if 	(n == 10) return (    "g");
	if 	(n == 11) return (    "g#");
return(0);
}

/* ---------------------------------------------------------------------- */

static inline int process_block(struct demod_state *s, int tabsize)
{
	float tote;
	float totte[tabsize * 2];
	int i, j;

	tote = 0;
	for (i = 0; i < BLOCKNUM; i++)
		tote += s->l1.piano.energy[i];
	for (i = 0; i < tabsize * 2; i++) {
		totte[i] = 0;
		for (j = 0; j < BLOCKNUM; j++)
			totte[i] += s->l1.piano.tenergy[j][i];
	}
	for (i = 0; i < tabsize; i++)
		totte[i] = fsqr(totte[i]) + fsqr(totte[i+tabsize]);
	memmove(s->l1.piano.energy+1, s->l1.piano.energy, 
		sizeof(s->l1.piano.energy) - sizeof(s->l1.piano.energy[0]));
	s->l1.piano.energy[0] = 0;
	memmove(s->l1.piano.tenergy+1, s->l1.piano.tenergy, 
		sizeof(s->l1.piano.tenergy) - sizeof(s->l1.piano.tenergy[0]));
	memset(s->l1.piano.tenergy, 0, sizeof(s->l1.piano.tenergy[0]));
	tote *= (BLOCKNUM*BLOCKLEN*0.5);  /* adjust for block lengths */
	//verbinfo("\npiano: Energies:\t%8.5f", tote);
	/*
	for (i = 0; i < tabsize; i++) {
		if (i % 6 == 0)	verbinfo("\n");
		verbinfo("%8.5f\t", totte[i]);
	}
	*/
	if ((i = find_max_idx(totte, pianonoise, tabsize)) < 0)
		return -1;
	/*
	if ((tote * 0.4) > totte[i])
		return -1;
	*/
	if ((tote * 0.01) > totte[i])
		return -1;
	return i;
}

/* ---------------------------------------------------------------------- */

static void piano_demod(struct demod_state *s, float *buffer, int length)
{
	float s_in;
	int i,  count = 0;

	for (; length > 0; length--, buffer++) {
		s_in = *buffer;
		s->l1.piano.energy[0] += fsqr(s_in);
		for (i = 0; i < TONETABSIZE; i++) {
			s->l1.piano.tenergy[0][i] += COS(s->l1.piano.ph[i]) * s_in;
			s->l1.piano.tenergy[0][i+TONETABSIZE] += SIN(s->l1.piano.ph[i]) * s_in;
			s->l1.piano.ph[i] += piano_freq[i];
		}
		if ((s->l1.piano.blkcount--) <= 0) {
			s->l1.piano.blkcount = BLOCKLEN;
			i = process_block(s, TONETABSIZE);
			if (i >= 0) {
	    		    if (i == s->l1.piano.lastch) count++;
			    else {
				count = 0;
				s->l1.piano.lastch = i;
			    }
			    /*
			    maybe i arranged it a bit hairy
			    as my piano starts with an A, Octave -1 (55 Hz)
			    I made that A  to be tone nr. 0
			    but an octave changes per tradition at C
			    which is A + 3 or -9 -tones....
			    */
			    admin_octave = ((i + 9) / 12) -2;	
			    sprintf(foundnote, "%s", tone_number_to_name(i));
			    if (admin_octave < 0) {
				foundnote[0] = toupper(foundnote[0]);
			    }
			    admin_octave_2_user_octave();
			    info
				("autodetection: heard tone %s (~%5.0f Hz), octave %d \n",
		        	foundnote, bias * tonetab[i], user_octave);
			} else {
			    //count = 0;
			    /* if there are gaps within stable detect, 
			       I do not reset the count */
			    verbinfo("no tone found.\n");
		        }
//			s->l1.piano.lastch = i;
			
			if (count) {
			    info("\n stable tone detection -> fine tuning.\n");
			    dem_mask[1] = 1;
	    		    dem_mask[0] = 0;
			    tone = i;
			    init_demod();
			    return;
			}
		}			
	}
}
				
/* ---------------------------------------------------------------------- */

const struct demod_param demod_piano = {
	"piano", SAMPLE_RATE, 0, piano_init, piano_demod
};

/* ---------------------------------------------------------------------- */
