/*
 *
 *	demod_fine.c - piano tone fine-tuning demodulator
	for tuning programme
 *	by Guenther Montag
 *	inspired by:
 *
 *      demod_zvei.c -- ZVEI signalling demodulator/decoder
 *      Copyright (C) 1996  
 *          Thomas Sailer (sailer@ife.ee.ethz.ch, hb9jnx@hb9w.che.eu)
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* ---------------------------------------------------------------------- */

#include "term.h"
#include "piano.h"
#include "filter.h"
#include <math.h>
#include <string.h>

/* ---------------------------------------------------------------------- */

int tabsize;
float tuningfreq, foundbias, foundchamberhzbias;
float real_freq[FINETABSIZE], fine_freq[FINETABSIZE];
//const float finenoise = 0.999999999999997;
const float finenoise = 1;
//const float finenoise = 0.995;

char fine_note_status[32];
char fine_bias_status[32];
	
/* ---------------------------------------------------------------------- */
	
void generate_fine_freqtable(float frequency,  int tabsize)
{
	int i;
	int tabhalf = (tabsize -1) / 2 ;
	float exponent;
	// e.g. tabsize = 65, tabhalf = 32
	// should look symmetric for graphic display
	verbinfo ("creating demod table for %f\n", tuningfreq);
	verbinfo ("center frequency: %f\n", frequency);
	for (i = 0; i < tabsize; i++) {
	    exponent = (i - tabhalf)/(12.0  * tabhalf);
	    real_freq[i] =  (frequency * pow(2, exponent));
	    //verbinfo ("%f\t", real_freq[i]);
	    //if ((i + 1) % 5 == 0)  verbinfo ("\n");
	    fine_freq[i] = PHINC(real_freq[i]);
	}
	//verbinfo ("\n");
}
	
/* ------------------------------------------------------------------- */

void fine_init(struct demod_state *s)
{
	float step;
	
	tabsize = FINETABSIZE;
	memset(&s->l1.fine, 0, sizeof(s->l1.fine));
	if (autotune || noteinput) {
	    tuningfreq = tonetab[tone] * bias;
	    generate_fine_freqtable(tuningfreq, tabsize);
	    step = real_freq[tabsize/2 + 1] - real_freq[tabsize/2];
	    statusinfo(1, "fine tuning %s", params);
	    statusinfo 
		(2, "%s, octave %d, ~%0.0fHz, range +- 1/2 tone, %0.0f-%0.0f Hz, step ~%0.2f Hz", 
		foundnote, user_octave, tuningfreq, 
		real_freq[0], real_freq[tabsize-1], step);
	}
	if (freqinput) {
	    tuningfreq = inputfreq;
	    generate_fine_freqtable(tuningfreq, tabsize);
	    step = real_freq[tabsize/2 + 1] - real_freq[tabsize/2];
	    statusinfo(1, "fine tuning %s", params);
	    statusinfo(2, "around ~%0.0fHz, range %0.0f-%0.0fHz, step ~%0.1f Hz", 
		tuningfreq, real_freq[0], real_freq[tabsize-1], step);
	}
}

/* ---------------------------------------------------------------------- */

static inline int process_block(struct demod_state *s, int tabsize)
{
    float tote, sum, avg, max;
    float totte[tabsize * 2];
    int i, j;
    static int nomaxcount = 0, outcount = 0, samecount = 0, same = 0;

    tote = 0; sum = 0; avg = 0;
    for (i = 0; i < BLOCKNUM; i++)
	tote += s->l1.fine.energy[i];
    for (i = 0; i < tabsize * 2; i++) {
	totte[i] = 0;
	for (j = 0; j < BLOCKNUM; j++)
	    totte[i] += s->l1.fine.tenergy[j][i];
    }
	
    for (i = 0; i < tabsize; i++)
	totte[i] = fsqr(totte[i]) + fsqr(totte[i+tabsize]);
    memmove(s->l1.fine.energy+1, s->l1.fine.energy, 
    	sizeof(s->l1.fine.energy) - sizeof(s->l1.fine.energy[0]));
    s->l1.fine.energy[0] = 0;
    memmove(s->l1.fine.tenergy+1, s->l1.fine.tenergy, 
	sizeof(s->l1.fine.tenergy) - sizeof(s->l1.fine.tenergy[0]));
    memset(s->l1.fine.tenergy, 0, sizeof(s->l1.fine.tenergy[0]));
    tote *= (BLOCKNUM*BLOCKLEN*0.5);  /* adjust for block lengths */

	// numeric output
    //verbinfo("fine: Energies: total: \t%8.5f", tote);
    for (i = 0; i < tabsize; i++) {
	//if (i % 8 == 0)	verbinfo("\n");
	//verbinfo("%8.5f ", totte[i]);
	sum += totte[i];
    }

    avg = sum / tabsize;
    //verbinfo("\nfine: average energy:\t\t%8.5f\n", avg);
    //if ((j = find_max_idx(totte, finenoise, tabsize)) < 0) {
    j = find_max_idx(totte, finenoise, tabsize);

    if ( j < 0 ) {            
    	verbinfo("fine tune: no maximum found.\n");
	nomaxcount++;
    } else nomaxcount = 0;
/*
    if (j == 0 || j == tabsize ) {
	verbinfo("fine tune: maximum out of range.\n");
//	outcount++;
    } else outcount = 0;  
*/
/*
    if (nomaxcount > SAMEMAX) {
        warninfo("fine tune: maximum not found %d times.\n", SAMEMAX);
	info("I Switch back to raw mode.\n");
	nomaxcount = 0;
	dem_mask[1] = 0;
	dem_mask[0] = 1;
	init_demod();
    	return -1; 
    } 
*/
    max = totte[j];
    
    foundbias = real_freq[j] / tuningfreq - 1.0;
    foundchamberhzbias = 440.0 * foundbias;
    /*
    verbinfo("fine: maximum energy:\t\t%8.5f at position %d\n", max, j);
    verbinfo("fine: resonance frequency:\t%8.5f \n", real_freq[j]);
    verbinfo("fine: found bias:\t\t%8.5f \n", foundbias);
    verbinfo("fine: found chambertone deviation:\t%d\n", foundchamberhzbias);
    */

    if (! (nomaxcount || outcount)) {
	/* " g r a p h i c "  output */
	// only what looks like constant result is printed out 
	// for exact symmetry, and the "a" of "piano" headline should
	// be over the green spot...
	if (samecount) {
	  for (i = 0; i < (COLS - tabsize) / 2 ; i++) {
	    RxOut(' ');
	  }
	  for (i = 0; i < tabsize; i++) {
	    if (totte[i] == max ) { 
		if (foundbias < 0) {
		    attrset(COLOR_PAIR(12));
		    RxOut('-');
		    attrset(A_NORMAL);
		    continue;
		}
		if (foundbias == 0) {
		    attrset(COLOR_PAIR(10));
		    RxOut(' ');
		    attrset(A_NORMAL);
		    continue;
		}
		if (foundbias > 0) {
		    attrset(COLOR_PAIR(11));
		    RxOut('+');
		    attrset(A_NORMAL);
		    continue;
		}
	    }
	    if (totte[i] >= max * 0.8) { 
		RxOut('o');
		continue;
	    }
	    else 
	        RxOut('.');
	  }
	  RxOut('\n');
	}
	
        /* " n u m e r i c "  output */
	if (j == same) {
	    samecount++;
	    //verbinfo("samecount:%d", samecount);
	}
	else if (j != same) {
    	    samecount = 0;
    	    same = j;
	}
	if ((! samecount) && (real_freq[j]))
	    info("\n\tfreq= %4.0f Hz, deviation: %+2.4f, Hz-of-a bias: %+3.0f", 
	    real_freq[j], foundbias, foundchamberhzbias);
	else if (samecount) 
	    info (".");
	//if ((tote * 0.02) > totte[j]) return -1;
    }  
	return j ;
}

/* ---------------------------------------------------------------------- */

static void fine_demod(struct demod_state *s, float *buffer, int length)
{
	float s_in;
	int i;
	//verbinfo("fine tuning!\n");
	for (; length > 0; length--, buffer++) {
		s_in = *buffer;
		s->l1.fine.energy[0] += fsqr(s_in);
		for (i = 0; i < FINETABSIZE; i++) {
			s->l1.fine.tenergy[0][i] += COS(s->l1.fine.ph[i]) * s_in;
			s->l1.fine.tenergy[0][i+FINETABSIZE] += SIN(s->l1.fine.ph[i]) * s_in;
			s->l1.fine.ph[i] += fine_freq[i];
		}
		if ((s->l1.fine.blkcount--) <= 0) {
		    s->l1.fine.blkcount = BLOCKLEN;
		    i = process_block(s, FINETABSIZE);
		}
		s->l1.fine.lastch = i;
	}
}
				
/* ---------------------------------------------------------------------- */

const struct demod_param demod_fine = {
	"fine", SAMPLE_RATE, 0, fine_init, fine_demod
};

/* ---------------------------------------------------------------------- */
