/**************************************************************

  FTP - file transfer protocol based program

  By Erick Engelke and Dean Roth

  This program is distributed in the hope that it will be
  useful, but without any warranty; without even the implied
  warranty of merchantability or fitness for a particular
  purpose.

  Erick Engelke
  erick@development.watstar.uwaterloo.ca

**************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <conio.h>
#include <io.h>
#include <dos.h>
#include <dir.h>
#include <mem.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <ctype.h>
#include <process.h>

#include "tcp.h"

#pragma warn -pia
#pragma warn -rch

extern unsigned _stklen = (16 * 1024);
extern int directvideo = 0;

#define FTPCTL_PORT 21

#define DATA_BINARY 0
#define DATA_ASCII  1

/* These should be made as large as possible */
#define FILEBUFFERLEN    16384
#define SCREENBUFFERLEN   1024

#define MAXSTRING   127
#define MAXFILES    256
#define MAXDIRS     256
#define MAXHOST      64

static char proxy[MAXHOST];
static char server[MAXHOST];
static char username[MAXHOST];

tcp_Socket ftpctl, ftpdata;

word ftpctl_port = FTPCTL_PORT;
int connected = 0;
int passive = 0;
char responsebuf[1024];
char buffer[1024];
char linebuffer[1024];

char *filebuffer, *screenbuffer;

int outstanding;
extern int debug_on;
int quiet = 0;

FILE *infile = (FILE *) NULL;

char unixbuf[256];
char dosbuf[100];

/*
 * tounix - if this is to be found from dos name,
 * strip path, etc.
 */
char *
tounix(char *filename)
{
  char *p;

  p = filename;
  while (p = strchr(p, '\\'))
    *p++ = '/';
  return (filename);
}

char *
todos(char *filename)
{
  char *p, *q, *lastq = NULL;
  int i = 1;
  int comment = 0;              /* shall we comment ? */

  p = filename;
  while (p = strchr(p, '/'))
    *p++ = '\\';
  strcpy(dosbuf, filename);
  if (p = strchr(dosbuf, '.')) {
    /* keep dot */
    p += 2;
    /* point to second char in case last extension */
    lastq = q = p;
    /* find next extension */
    while (q = strchr(q, '.')) {
      comment = 1;
      if (*(++q) == '.')
        continue;
      *p++ = *q;
      lastq = ++q;
      i++;
    }
    while (i < 3) {
      if (*lastq == '.')
        continue;
      if ((*p++ = *lastq++) == 0)
        break;
      i++;
    }
    *p = 0;                     /* terminate */
    if (*lastq)
      comment = 1;
  }

  if (comment)
    printf("Using truncated local filename %s\n", dosbuf);
  return (dosbuf);
}

/***********************/
/* FUNCTION PROTOTYPES */
/***********************/
static char Fgetchar(char *);

char *
keygets(char *s, char *prompt, int echo)
{
  int len = 0;
  unsigned redraw;
  char ch;

  printf("%s", prompt);

  for (len = 0; len < 512;) {
    ch = getch();
    redraw = 0;
    switch (ch) {
      case 27:                 /* clear the line */
        s[len = 0] = 0;
        if (echo)
          redraw = 1;
        break;
      case 13:
      case 10:
        printf("\n");
        s[len] = 0;
        return (s);
        break;
      case 8:
      case 127:
        if (len > 0) {
          redraw = 1;
          s[--len] = 0;
        }
        break;
      default:
        if (echo && !(isprint(ch)))
          break;
        s[len++] = ch;
        s[len] = 0;
        if (echo)
          printf("%c", ch);
        else
          printf("*");
        break;
    }
    if (redraw) {
      printf("\r%-79.79s\r", " ");
      printf("%s", prompt);
      len = strlen(s);
      if (echo)
        printf("%s", s);
      else {
        for (ch = 1; ch < len; ++ch)
          printf("*");
      }
      redraw = 0;
    }
  }
  return (s);
}

char *
linegets(char *prompt, char *s, int echo)
{
  char *p;

  do {
    if (infile) {
      if (!fgets(s, 255, infile)) {
        if (!quiet)
          printf("Script file done, "
                 "returning to command mode\n");
        infile = (FILE *) NULL;
        continue;
      }
      else {
        if (p = strchr(s, '\n'))
          *p = 0;
        if (p = strchr(s, '\r'))
          *p = 0;
        if (!quiet && (*s != '@') && echo)
          printf("%s%s\n", prompt, s);
        if (*s == '@')
          movmem(s + 1, s, strlen(s));
      }
    }
    else {
      s = keygets(s, prompt, echo);
    }
    break;
  } while (1);
  return (s);
}

void
dosmessage(int x, char *msg)
{
  if (x)
    printf("Unable to %s, %s\n", msg, sys_errlist[errno]);
  else
    printf("%s successful\n", msg);
}

void
ldir(char *p)
{
  struct ffblk ffblk;
  int status;
  char *ptr;
  unsigned min, hour, day, mon, year;
  int lines;

  if (!quiet)
    printf("DIR %s:\n", p);
  if (status = findfirst(p, &ffblk, FA_DIREC)) {
    printf("  ERROR: %s\n", sys_errlist[errno]);
    return;
  }

  lines = 0;

  while (status == 0) {

    if (lines >= 20) {
      printf("--- more ---  [ <SPACE> <RETURN>  <Q>uit ]");
      switch (Fgetchar(" \rQ")) {
        case ' ':
        case '\r':
          printf("\r%-79.79s\r", " ");
          lines = 0;
          break;
        case 'Q':
          printf("\r%-79.79s\r", " ");
          goto EXIT;
          break;
      }
    }

    min = (ffblk.ff_ftime >> 5) & 63;
    hour = (ffblk.ff_ftime >> 11) % 12;
    day = ffblk.ff_fdate & 31;
    mon = (ffblk.ff_fdate >> 5) & 15;
    year = (ffblk.ff_fdate >> 9) + 1980;

    if (ptr = strchr(ffblk.ff_name, '.'))
      *ptr++ = 0;
    else
      ptr = "";

    if (ffblk.ff_attrib & FA_DIREC)
      printf(" %-8s.%-3s %8s ",
             ffblk.ff_name, ptr, "<DIR>");
    else
      printf(" %-8s %-3s %8lu ",
             ffblk.ff_name, ptr, ffblk.ff_fsize);

    printf(" %2u-%02u-%4u   %2u:%02u%c\n",
           mon, day, year % 100,
           (hour > 12) ? hour - 12 : hour, min,
           (hour > 11) ? 'p' : 'a');

    lines++;

    status = findnext(&ffblk);
  }

EXIT:
  return;
}

int
getresponse(tcp_Socket * s, int (*fn) (const char *))
{
  int code, i;
  int status;

  outstanding = 0;
  if (!connected)
    return (600);
  sock_mode(s, TCP_MODE_ASCII);
  sock_wait_input(s, sock_delay, NULL, &status);
  sock_gets(s, (unsigned char *)responsebuf, sizeof(responsebuf));
  code = atoi(responsebuf);
  do {
    (*fn) (responsebuf);

    if (atoi(responsebuf) == code) {
      for (i = 0; i < 5; ++i) {
        if (isdigit(responsebuf[i]))
          continue;
        if (responsebuf[i] == ' ') {
          sock_mode(s, TCP_MODE_BINARY);
          return (code);
        }
        if (responsebuf[i] == '-')
          break;
      }
    }
    sock_wait_input(s, sock_delay, NULL, &status);
    sock_gets(s, (unsigned char *)responsebuf, sizeof(responsebuf));
  } while (1);
sock_err:
  switch (status) {
    case 1:                    /* foreign host closed */
      break;
    case -1:                   /* timeout */
      printf("ERROR: %s\n", sockerr(s));
      break;
  }
  connected = 0;
  return (221);
}

int
getport(char *response, longword *ip, int *port)
{
  int addr[4], data[2];
  byte *p1, *p2;

  while ((*response < '0') || (*response > '9')) response++;

  if (sscanf(response, "%d,%d,%d,%d,%d,%d",
	     &addr[3], &addr[2], &addr[1], &addr[0],
	     &data[1], &data[0]) != 6) {
    return (-1);
  }

  p1 = (byte *) ip;
  p2 = (byte *) port;

  p1[3] = addr[3]; p1[2] = addr[2];
  p1[1] = addr[1]; p1[0] = addr[0];
  p2[1] = data[1]; p2[0] = data[0];

  return (0);
}

/*
 * sendport - negotiate port, return 0 on success
 */
int
sendport(tcp_Socket * s, longword hisip, int (*puts) ())
{
  longword ip;
  byte *p1, *p2;
  time_t now;
  static int dataport = 0;
  int status;

  if (passive) {
    sock_printf(s, "PASV\r\n");
    if ((status = getresponse(s, puts)) > 299) {
      return (status);
    }
    if (getport(responsebuf+4, &ip, &dataport)) {
      printf("Unable to parse PASV response\n");
      return (-1);
    }
    if (!tcp_open(&ftpdata, 0, ip, dataport, NULL)) {
      printf("Unable to connect to data port");
      return (-1);
    }
    sock_sturdy(&ftpdata, 100);      
  }
  else {
    ip = gethostid();

    if (dataport == 0) {
      time(&now);
      dataport = 1024 + (word) (now >> 16) + (word) (now);
    }

    tcp_listen(&ftpdata, ++dataport, hisip, 0, NULL, 0);
    sock_sturdy(&ftpdata, 100);

    p1 = (byte *) &ip;
    p2 = (byte *) &dataport;
    sock_printf(s, "PORT %hu,%hu,%hu,%hu,%hu,%hu\r\n",
                p1[3], p1[2], p1[1], p1[0], p2[1], p2[0]);
    if ((status = getresponse(s, puts)) > 299) {
      sock_close(&ftpdata);
      return (status);
    }
  }
  return (0);
}

void
sendcommand(tcp_Socket * s, char *cmd, char *param)
{
  if (!connected)
    printf("You are not connected to anything right now\n");
  else {
    sock_printf(s, (param) ? "%s %s\r\n" : "%s\r\n",
                cmd, param);
    outstanding = 1;
  }
}

int
remotecbrk(tcp_Socket * s)
{
  char ch;

  if (watcbroke) {
    /* we received a control break - first reset the flag */
    watcbroke = 0;
    /* attempt to do something about it */
    printf("\rUser pressed control break.  "
           "Do you wish to disconnect [Y/N] ");
    do {
      ch = toupper(getch());
      if (ch == 'Y') {
        printf("Y\nBreaking connection\n");
        sock_abort(s);
        break;
      }
      if (ch == 'N') {
        printf("N\nIgnoring control break\n");
        break;
      }
    } while (1);
  }
  return (0);
}

/*
 * remoteop
 *      cmd =  "RETR"
 *      data = "RemoteFileName"
 *      file = "LOCALNAM.FIL"
 */
int
remoteop(tcp_Socket * s, longword hisip,
         char *cmd, char *data, char *file, int get,
         int ascii, int screen, void (*upcall) (char *), int hash)
{
  struct stat statbuf;
  int f = -1, len, status, writestatus = 0;
  longword totallen = 0, lasthash = 0;
  longword starttime, endtime, totaltime, speed, skipping = 0;
  long now, last = 0;
  word offlen = 0;

  if ((status = sendport(s, hisip, puts)) != 0)
    return (status);

  if (screen == 0) {
    if (get) {
      if (access(file, 0) == 0) {
	stat(file, &statbuf);
	if (statbuf.st_mode & S_IFREG) {
	  printf("File '%s' already exists, "
		 "do you wish to replace it? (Y/N) ", file);
	  if (Fgetchar("YN") == 'N') {
	    printf("N\nSkipping...\n");
	    return (-1);
	  }
	  else {
	    printf("Y\nErasing file");
	    status = unlink(file);
	    if (status) {
	      printf("...failed\n");
	      return (-1);
	    }
	    printf("\n");
	  }
	}
      }
    }
  }

  if (get && screen) {
    /******************************************/
    /* THIS MUST BE DONE BEFORE sendcommand() */
    /******************************************/
    sock_setbuf(&ftpdata, (unsigned char *)filebuffer, FILEBUFFERLEN);
    sock_mode(&ftpdata, TCP_MODE_ASCII);
  }

  sendcommand(s, cmd, data);

  if ((status = getresponse(s, puts)) >= 200) {
    sock_close(&ftpdata);
    return (status);
  }

  if (screen == 0) {
    f = open(file,
             ((get) ? (O_CREAT | O_TRUNC | O_RDWR) : O_RDONLY) |
             ((ascii) ? O_TEXT : O_BINARY),
             S_IWRITE | S_IREAD);
    if (f == -1) {
      sock_close(&ftpdata);
      printf("Unable to open file: %s\n", sys_errlist[errno]);
      return (-1);
    }
  }

  /* opening connection */

  sock_wait_established(&ftpdata, sock_delay, NULL, &status);

  if (get) {
    int lines = 0;

    if (screen) {
#define SM_PAUSE  1
#define SM_QUIT   2
#define SM_NOSTOP 3             /* do not pause - continuous */
      int mode = SM_PAUSE;

      do {
        /*************************************/
        /* THIS GETS US OUT OF INFINITE LOOP */
        /*************************************/
        sock_wait_input(&ftpdata, sock_delay, NULL, &status);

        sock_gets(&ftpdata, (unsigned char *)screenbuffer, SCREENBUFFERLEN);
        if (!totallen)
          starttime = set_ttimeout(0);
        totallen += (longword) strlen(screenbuffer);
        endtime = set_ttimeout(0);

        /* either do malloc or dump to screen */
        if (upcall)
          upcall(screenbuffer);
        else {
          if (mode != SM_QUIT)
            puts(screenbuffer);

          if (mode == SM_PAUSE) {
            lines++;
            if (lines >= 20) {
              printf("--- more ---  "
                     "[ <SPACE> <CR>  <C>ontinuous  <Q>uit ]");
              switch (Fgetchar(" \rCQ")) {
                case ' ':
                case '\r':
                  printf("\r%-79.79s\r", " ");
                  lines = 0;
                  break;

                case 'Q':
                  printf("\r%-79.79s\r"
                         "(flushing data stream...)\n", " ");
                  mode = SM_QUIT;
                  break;
                case 'C':
                  printf("\r%-79.79s\r", " ");
                  mode = SM_NOSTOP;
                  break;
              }
            }
          }
        }

      } while (1);
    }
    else {
      /*****************/
      /* GET TO A FILE */
      /*****************/

      do {
        sock_wait_input(&ftpdata, sock_delay,
                        (sockfunct_t) remotecbrk, &status);
        len = sock_fastread(&ftpdata, (unsigned char *)filebuffer + offlen,
                            FILEBUFFERLEN - offlen);
        offlen += len;
        if (!totallen)
          starttime = set_ttimeout(0);
        totallen += (longword) len;
        endtime = set_ttimeout(0);
        if (hash) {
          lasthash += len;
          while (lasthash > hash) {
            printf("\rGot %8ld bytes\r", (long) totallen);
            lasthash -= hash;
          }
        }
        if (offlen > FILEBUFFERLEN - 512) {
          if (skipping)
            offlen = 0;
          else {
            /* printf("writing %u bytes\n", offlen); */
            writestatus = (write(f, filebuffer, offlen)
                           != offlen);
            offlen = 0;
            if (writestatus != 0) {
              printf("disk full...\n");
              sendcommand(s, "ABOR", NULL);
              skipping = 1;
            }
          }
        }
      } while (1);
    }
  }
  else {

    /***********************/
    /* GET INFO ABOUT FILE */
    /***********************/
    fstat(f, &statbuf);

    /* put to remote end */
    do {
      if (watcbroke) {
        /* got a control break - let's kill this baby */
        printf("\rUser pressed control break, "
               "aborting send\n");
        close(f);
        sock_close(&ftpdata);
        sock_wait_closed(&ftpdata, sock_delay, NULL, &status);
      }

      if (hash) {
        time(&now);
        if (now != last) {
          last = now;
          printf("\rPut %8ld bytes of %ld (%d%%)\r",
               (long) totallen - (long) sock_tbused(&ftpdata),
                 (long) statbuf.st_size,
                 statbuf.st_size > 0 ?
                 (100L * totallen) / statbuf.st_size : 100);
        }
      }

      if (sock_tbused(&ftpdata) == 0) {
        len = read(f, filebuffer, FILEBUFFERLEN);
        if (!totallen)
          starttime = set_ttimeout(0);
        totallen += (longword) len;
        endtime = set_ttimeout(0);
        if (len == -1) {
          dosmessage(1, "write file");
          close(f);
          sock_close(&ftpdata);
          return (-1);
        }
        if (len == 0) {
          sock_close(&ftpdata);
          sock_wait_closed(&ftpdata, sock_delay, NULL,
                           &status);
        }
        sock_enqueue(&ftpdata, (unsigned char *)filebuffer, len);
      }
      sock_tick(&ftpdata, &status);
    } while (1);
  }

EXIT:
  sock_close(&ftpdata);

  /* only if it is a file and we are getting it
     and we haven't had errors */
  if ((screen == 0) && (get != 0) && (writestatus == 0)) {
    if (writestatus = (write(f, filebuffer, offlen) != offlen))
      printf("disk full...\n");
  }
  if (f != -1)
    close(f);

  status = getresponse(s, puts);
  if (status < 300) status = 0;

  totaltime = ((endtime - starttime) * 6400L) / 1165L;
  speed = (39 * totallen / 4) / (totaltime + 1);
  if (hash)
    printf("\n");
  printf("Transfer of %0lu Kbytes (%ld bytes) in "
         "%0lu.%02lu seconds ( %0lu.%02lu Kbytes/s ).\n",
         totallen / 1024L,
         totallen,
         totaltime / 100L, totaltime % 100L,
         speed / 100L,
         speed % 100L);
  return (status);

sock_err:
  if (status == -1)
    printf("LXTCP socket error: %d\n", status);

  goto EXIT;
}

void
text2dos(char *p)
{
  while (*p) {
    if (!isprint(*p)) {
      *p = 0;
      break;
    }
    ++p;
  }
}

/***********************************************/
/* BUILD LIST OF FILE NAMES TO BE USED BY MGET */
/***********************************************/
char **dumplist;
int dumpcur;
char **dirlist;
int dircur;
void
dumper(char *s)
{
  char *p;

  /* printf("dumper: s = [%s]\n", s); */

  if (dumpcur < MAXFILES) {

    if (s[0] == ' ') {          /* blank line */
      return;
    }
    if (!s[0]) {                /* NULL string */
      return;
    }
    if (strchr(s, ':')) {       /* directory? */
      return;
    }

    if ((p = (dumplist[dumpcur] = malloc(strlen(s) + 1)))
        != NULL) {
      strcpy(p, s);
      text2dos(p);
      dumpcur++;
    }
    else {
      printf("out of memory... ignoring %s\n", s);
    }
  }
}

static char
Fgetchar(As_allowed)
  char *As_allowed;

{
  char *cptr;
  char ch;

  while (1) {
    ch = (char) toupper(getch());

    cptr = As_allowed;

    while (*cptr) {
      if (*cptr == ch)
        goto EXIT;

      cptr++;
    }
  }

EXIT:

  return (ch);
}

/*****************************************************/
/* GET LIST OF LOCAL FILES MATCHING PATTERN FOR MPUT */
/*****************************************************/
static char *
Ffilelist(char *As_pattern, int Ai_state, int attr)
{
  static struct ffblk ffblk;
  int status;

  if (Ai_state == 1)
    status = findfirst(As_pattern, &ffblk, attr);
  else
    status = findnext(&ffblk);

  if (status) {
    /* printf("  ERROR: %s\n", sys_errlist[errno]); */
    return ((char *) NULL);
  }
  else {
    strlwr(ffblk.ff_name);
    return (strlwr(ffblk.ff_name));
  }
}

static int
Fmget(tcp_Socket * s, longword hisip, char *data,
      int ascii, int interactive, int hash)
{
  char *p;
  int i;

  dumpcur = 0;
  dumplist = calloc(MAXFILES, sizeof(char *));

  if (dumplist == NULL) {
    printf("Insufficient memory to start "
           "multifile operation\n");
    return (-1);
  }
  /*********************/
  /* GET LIST OF FILES */
  /*********************/
  if (ascii != DATA_ASCII) {
    sendcommand(s, "TYPE", "A");
    if (getresponse(s, puts) != 200)
      goto EXIT;
  }

  remoteop(s, hisip, "NLST", data, NULL, 1, 1, 1, dumper, 0);

  if (ascii != DATA_ASCII) {
    sendcommand(s, "TYPE", "I");
    if (getresponse(s, puts) != 200) {
      /* datamode = DATA_ASCII; */
      printf("WARNING: Host left connection in ASCII mode");
      goto EXIT;
    }
  }

  printf("There are %u matching files\n", dumpcur);

  for (i = 0; i < dumpcur; ++i) {
    p = dumplist[i];

    if (interactive) {

      printf("GET [%s] (Yes,No,Quit,Rest) > ", p);

      switch (Fgetchar("YNQR")) {
        case 'N':
          printf("\n");
          continue;

        case 'Y':
          printf("\n");
          break;

        case 'Q':
          printf("\n");
          goto EXIT;
          break;
        case 'R':
          printf("\nWill GET rest without prompting\n");
          interactive = 0;
          break;
      }
    }

    remoteop(s, hisip, "RETR", p, todos(p), 1, ascii, 0,
             NULL, hash);
  }

EXIT:

  /* clean up memory */
  for (i = 0; i < dumpcur; ++i) {
    if (p = dumplist[i])
      free(p);
  }

  free(dumplist);
  return (0);
}

static int
Fmput(tcp_Socket * s, longword hisip, char *data,
      int ascii, int interactive, int hash)
{
  char *p;

  p = (char *) NULL;

  while (p = Ffilelist(data, p == (char *) NULL ? 1 : 0, 0)) {
    if (interactive) {
      printf("PUT [%s] (Yes,No,Quit,Rest) > ", p);

      switch (Fgetchar("YNQR")) {
        case 'N':
          printf("\n");
          continue;

        case 'Y':
          printf("\n");
          break;

        case 'Q':
          printf("\n");
          goto EXIT;
          break;
        case 'R':
          printf("\nWill PUT rest without prompting\n");
          interactive = 0;
          break;
      }
    }

    /*************/
    /* SEND FILE */
    /*************/
    remoteop(s, hisip, "STOR", tounix(p), p, 0, ascii, 0,
             NULL, hash);
  }

EXIT:

  return (0);

}

static int
Fdget(tcp_Socket * s, longword hisip, char *data,
      int ascii, int hash)
{
  char *p, *q;
  int i, status;

  dumpcur = 0;
  dumplist = calloc(MAXFILES, sizeof(char *));

  if (dumplist == NULL) {
    printf("Insufficient memory to start "
           "multifile operation\n");
    return (-1);
  }
  /*********************/
  /* GET LIST OF FILES */
  /*********************/
  if (ascii != DATA_ASCII) {
    sendcommand(s, "TYPE", "A");
    if (getresponse(s, puts) != 200)
      goto EXIT;
  }

  remoteop(s, hisip, "NLST", data, NULL, 1, 1, 1, dumper, 0);

  if (ascii != DATA_ASCII) {
    sendcommand(s, "TYPE", "I");
    if (getresponse(s, puts) != 200) {
      /* datamode = DATA_ASCII; */
      printf("WARNING: Host left connection in ASCII mode");
      goto EXIT;
    }
  }

  if ((dumpcur > 0) && (access(todos(data), 0) != 0)) {
    dosmessage((status = mkdir(todos(data))), "make directory");
    if (status != 0) return (-1);
  }

  for (i = 0; i < dumpcur; ++i) {
    strcpy(unixbuf, data);
    strcat(unixbuf, "/");
    strcat(unixbuf, dumplist[i]);
    p = &unixbuf[0]; 

    if ((status = remoteop(s, hisip, "RETR", p, todos(p), 1, ascii, 0,
                           NULL, hash)) != 0) {
      if (status == 550) {
        /* may be a directory - push it onto the list */
        if ((dirlist != NULL) && (dircur < MAXDIRS)) {
          if ((q = (dirlist[dircur] = malloc(strlen(p) + 1)))
              != NULL) {
            strcpy(q, p);
            text2dos(q);
            printf("Recursing into directory %s\n", q);
            dircur++;
          }
          else {
            printf("out of memory... ignoring %s\n", p);
          }
        }
      }
    }
  }

EXIT:

  /* clean up memory */
  for (i = 0; i < dumpcur; ++i) {
    if (p = dumplist[i])
      free(p);
  }

  free(dumplist);
  return (0);
}

static int
Fdput(tcp_Socket * s, longword hisip, char *data,
      int ascii, int hash)
{
  struct stat statbuf;
  char *p, *q;
  int status;

  p = (char *) NULL;

  sendcommand(s, "MKD", tounix(data));
  if ((status = getresponse(s, puts)) != 257) {
    printf("Trying \"XMKD\" instead of \"MKD\"\n");
    sendcommand(s, "XMKD", tounix(data));
    status = getresponse(s, puts);
  }

  if (status != 257) {
    printf("Unable to create directory %s\n", tounix(data));
    return (status);
  }

  strcpy(dosbuf, data);
  strcat(dosbuf, "\\*.*");

  while (p = Ffilelist(&dosbuf[0], p == (char *) NULL ? 1 : 0, FA_DIREC)) {
    if (p[0] == '.') continue;
    strcpy(dosbuf, data);
    strcat(dosbuf, "\\");
    strcat(dosbuf, p);
    p = &dosbuf[0]; 
    stat(p, &statbuf);
    if (statbuf.st_mode & S_IFREG) {
      remoteop(s, hisip, "STOR", tounix(p), p, 0, ascii, 0,
               NULL, hash);
    }
    else {
      /* may be a directory - push it onto the list */
      if ((dirlist != NULL) && (dircur < MAXDIRS)) {
        if ((q = (dirlist[dircur] = malloc(strlen(p) + 1)))
            != NULL) {
          strcpy(q, p);
          printf("Recording subdirectory %s\n", q);
          dircur++;
        }
        else {
          printf("out of memory... ignoring %s\n", p);
        }
      }
    }
  }

EXIT:

  return (0);

}

static int
Frget(tcp_Socket * s, longword hisip, char *data,
      int ascii, int hash)
{
  char *p;
  int i;

  /* max no. directories in list for an "rget" */
  dircur = 0;
  dirlist = calloc(MAXDIRS, sizeof(char *));

  if (dirlist == NULL) {
    printf("Insufficient memory to start "
           "recursive operation\n");
    return (-1);
  }

  if ((p = (dirlist[dircur] = malloc(strlen(data) + 1)))
      != NULL) {
    strcpy(p, data);
    dircur++;
  }

  for (i = 0; i < dircur; ++i) {
    p = dirlist[i];
    /* Fdget will add new directories to dirlist */
    Fdget(s, hisip, p, ascii, hash);
  }

EXIT:

  /* clean up memory */
  for (i = 0; i < dircur; ++i) {
    if (p = dirlist[i])
      free(p);
  }

  free(dirlist);
  return (0);
}

static int
Frput(tcp_Socket * s, longword hisip, char *data,
      int ascii, int hash)
{
  char *p;
  int i;

  /* max no. directories in list for an "rget" */
  dircur = 0;
  dirlist = calloc(MAXDIRS, sizeof(char *));

  if (dirlist == NULL) {
    printf("Insufficient memory to start "
           "recursive operation\n");
    return (-1);
  }

  if ((p = (dirlist[dircur] = malloc(strlen(data) + 1)))
      != NULL) {
    strcpy(p, data);
    dircur++;
  }

  for (i = 0; i < dircur; ++i) {
    p = dirlist[i];
    /* Fdpet will add new directories to dirlist */
    Fdput(s, hisip, p, ascii, hash);
  }

EXIT:

  /* clean up memory */
  for (i = 0; i < dircur; ++i) {
    if (p = dirlist[i])
      free(p);
  }

  free(dirlist);
  return (0);
}

void
wait(char *string)
{
  int a, b;
  time_t now, when, diff;
  struct time t;
  struct date d;

  getdate(&d);

  if (sscanf(string, "%u:%u", &a, &b) != 2) {
    printf("wait hh:mm with hours in 24 hour format\n");
    return;
  }
  t.ti_hour = a;

  t.ti_min = b;

  when = dostounix(&d, &t);
  time(&now);
  if (now > when)
    when += 24L * 60L * 60L;

  unixtodos(when, &d, &t);

  printf("Waiting until %u:%02u\n", t.ti_hour, t.ti_min);

  watcbroke = 0;
  do {
    if (watcbroke) {
      watcbroke = 0;
      printf("wait broken by control-break\n");
      break;
    }
    time(&now);
    diff = when - now;
    a = (int)(diff % 60);
    diff /= 60;
    printf("    %lu:%02lu:%02u left\r",
           diff / 60L, diff % 60L, a);
  } while (now < when);
  printf("done waiting         \n");
}

void
dorun(char *fname)
{
  if (infile = fopen(fname, "rt"))
    printf("running commands from: %s\n", fname);
  else
    printf("ERROR: unable to read '%s'\n", fname);
}
typedef struct {
  char *cmd;
  /* if 1, does not need to be connected to work */
  int safety;
  int cmdnum;
  char *help;
} cmdtype;

enum {
  STARTLIST,
  DIR, LS, GET, PUT, DEL, CHDIR, RMDIR, MKDIR,
  MPUT, MGET, RPUT, RGET, BGET, BPUT,
  LDIR, LDEL, LCHDIR, LRMDIR, LMKDIR,
  ASCII, BIN, EBC, MODE, OPEN, CLOSE, USER, PASS,
  QUIT, HELP, HASH, INT, NONINT, STATUS, SHELL, DEBUG,
  WAIT, RUN, PWD, LPWD, TYPE, RHELP, ECHO, QUIET,
  QUOTE, PASV,
  ENDLIST
};

cmdtype cmds[] =
{
  {"STATUS", 1, STATUS, "show current status"},
  {"OPEN", 1, OPEN, "connect to host"},
  {"CLOSE", 0, CLOSE, "disconnect from host"},
  {"USER", 0, USER, "new user name"},
  {"PASS", 0, PASS, "enter password"},
  {"DIR", 0, DIR, "list remote directory"},
  {"LS", 0, LS, "list remote files"},
  {"GET", 0, GET, "get a remote file"},
  {"REC", 0, GET, "get a remote file"},
/*
   {"BGET", 0, BGET, "get a binary file"},
   {"BPUT", 0, BPUT, "put a binary file"},
 */
  {"PUT", 0, PUT, "put a local file"},
  {"SEND", 0, PUT, "put a local file"},
  {"MGET", 0, MGET, "get several files"},
  {"MPUT", 0, MPUT, "put multiple files"},
  {"RGET", 0, RGET, "get dir hierarchy"},
  {"RPUT", 0, RPUT, "put dir gierarchy"},
  {"DEL", 0, DEL, "delete a remote file"},
  {"RM", 0, DEL, "delete a remote file"},
  {"CD", 0, CHDIR, "change remote directory"},
  {"CHDIR", 0, CHDIR, "change remote directory"},
  {"RD", 0, RMDIR, "remove remote directory"},
  {"RMDIR", 0, RMDIR, "remove remote directory"},
  {"MD", 0, MKDIR, "make remote directory"},
  {"MKDIR", 0, MKDIR, "make remote directory"},
  {"ASCII", 0, ASCII, "set ASCII mode"},
  {"BINARY", 0, BIN, "set binary mode"},
  {"EBCDIC", 0, EBC, "set EBCDIC mode"},
  {"MODE", 0, MODE, "advanced set mode"},
  {"LDIR", 1, LDIR, "list local directory"},
  {"LLS", 1, LDIR, "list local directory"},
  {"LDEL", 1, LDEL, "delete local file"},
  {"LCD", 1, LCHDIR, "change local directory"},
  {"LCHDIR", 1, LCHDIR, "change local directory"},
  {"LRD", 1, LRMDIR, "remove local directory"},
  {"LRMDIR", 1, LRMDIR, "remove local directory"},
  {"LMD", 1, LMKDIR, "make local directory"},
  {"LMKDIR", 1, LMKDIR, "make local directory"},
  {"LPWD", 1, LPWD, "print local directory"},
  {"TYPE", 0, TYPE, "display remote *text* file"},
  {"QUIT", 1, QUIT, "quit program"},
  {"BYE", 1, QUIT, "quit program"},
  {"PWD", 0, PWD, "print current directory"},
  {"HELP", 1, HELP, "list instructions"},
  {"?", 1, HELP, "list instructions"},
  {"REMOTEHELP", 0, RHELP, "list cmds remote supports"},
  {"HASH", 1, HASH, "toggle/set hash markings"},
  {"PROMPT", 1, INT, "(ON/OFF) set prompt mode"},
  {"WAIT", 1, WAIT, "wait until hh:mm"},
  {"RUN", 1, RUN, "run a script file"},
  {"ECHO", 1, ECHO, "echo some text to screen"},
  {"QUIET", 1, QUIET, "ON | OFF for quiet scripts"},
  {"QUOTE", 1, QUOTE, "send arbitrary remote command"},
  {"!", 1, SHELL, "DOS shell"},
#ifdef DEBUG
  {"DEBUG", 1, DEBUG, "toggle debug mode"},
#endif
  {"x", 1, ENDLIST, NULL}};

int
text2cmd(char *line, char *linebuf, char **pp, char **qp)
{
  char cmdnum;
  int temp, matches;
  cmdtype *closeto, *cur;
  char *command, *parm1, *parm2;

  *qp = *pp = NULL;
  command = strtok(line, " ");
  if (!command)
    return (STARTLIST);         /* nothing there */

  parm1 = strtok((char *) NULL, " ");

  if (parm1) {
    parm2 = strtok((char *) NULL, " ");
    strcpy(linebuf, parm1);
  }
  else {
    parm2 = (char *) NULL;
    *linebuf = (char) 0;
  }

  cmdnum = matches = 0;
  closeto = NULL;

  do {
    cur = &cmds[cmdnum];
    if (*cur->cmd > 'Z')
      break;

    if (strnicmp(command, cur->cmd, strlen(cur->cmd)) == 0) {
      *pp = parm1;
      *qp = parm2;
      if (!(cur->safety || connected)) {
        printf("You cannot use %s until you are connected "
               "to another system\n",
               cur->cmd);
        printf("Use OPEN machinename to connect to a machine, "
               "or QUIT to exit\n");
        return (ENDLIST);
      }
      return (cur->cmdnum);
    }
    temp = strnicmp(command, cur->cmd, strlen(command));
    if (temp == 0) {
      if (matches == 0) {
        closeto = cur;
        ++matches;
      }
      else {
        if (matches == 1)
          printf("Ambiguous command, could be %s",
                 closeto->cmd);
        printf(", %s", cur->cmd);
        closeto = NULL;
        ++matches;
      }
    }
    ++cmdnum;
  } while (1);
  if (closeto != NULL) {
    *pp = parm1;
    *qp = parm2;
    printf(">>>> %s %s %s\n", closeto->cmd,
           parm1 ? parm1 : " ", parm2 ? parm2 : " ");
    return (closeto->cmdnum);
  }
  if (matches != 0)
    printf("\n");
  return (ENDLIST);
}

void
doshell(void)
{
  char *cptr;

  if (!(cptr = getenv("COMSPEC")))
    return;

  spawnl(P_WAIT, cptr, cptr, (char *) NULL);
}

dohelp(char *cmd)
{
  cmdtype *p[2], *q[2];
  char buffer[512];
  int i, found;

  /* set to 1 if we have to add the textual note */
  int addnote = 0;

  for (i = 0; cmds[i].cmdnum != ENDLIST; ++i);
  p[0] = cmds;
  p[1] = cmds + ++i / 2;

  if (cmd != NULL) {
    found = 0;
    i = toupper(*cmd);
    while (p[0]->cmdnum != ENDLIST) {
      if (i == toupper(p[0]->cmd[0])) {
        if (!(p[0]->safety || connected))
          addnote = 1;
        printf("%c %-8s : %s\n",
               (p[0]->safety || connected) ? ' ' : '*',
               p[0]->cmd, p[0]->help);
        ++found;
      }
      p[0]++;
    }
    if (!found)
      printf("No similar functions found\n");
    if (addnote)
      printf("\n* indicates commands not applicable "
             "because you are not connected yet\n");
    printf("\n");
    return (0);
  }

  printf("LXFTP %d.%d", VERSION/10, VERSION%10);

  if (RELEASE > 0) {
    printf("b%d", RELEASE);
  }

  printf(" - %s <%s>\n",
         "HP200LX TCP/IP Development Kit",
         "http://lxtcp.hplx.net/");

  do {
    *buffer = 0;
    for (i = 0; i < 2; ++i) {
      if (p[i]->cmdnum == ENDLIST) {
        if (i)
          printf("%s\n", buffer);
        printf("\n");
        if (addnote)
          printf("* indicates commands not applicable "
                 "at the moment\n");
        else
          printf("\n");
        printf("To invoke with a command file, "
               "use command line option: -f fname\n");
        return (0);
      }
      if (p[i]->safety || connected)
        strcat(buffer, "  ");
      else {
        strcat(buffer, "* ");
        addnote = 1;
      }

      strcat(buffer, p[i]->cmd);
      q[i] = p[i];
      while ((++q[i])->cmdnum == p[i]->cmdnum) {
        strcat(buffer, ", ");
        strcat(buffer, q[i]->cmd);
      }
      memset(strchr(buffer, 0), ' ',
             sizeof(buffer) - 1 - strlen(buffer));
      buffer[40 * i + 10] = 0;
      strcat(buffer, ": ");
      strcat(buffer, p[i]->help);
      memset(strchr(buffer, 0), ' ',
             sizeof(buffer) - 1 - strlen(buffer));
      p[i] = q[i];
      buffer[(i + 1) * 40 - 1] = 0;
    }
    printf("%s\n", buffer);
  } while (1);
}

ftpconnect(tcp_Socket * s, longword host, int port)
{
  int status;

  outstanding = 0;
  if (!tcp_open(s, 0, host, port, NULL)) {
    printf("Sorry, unable to connect to that machine "
           "right now!\n");
    return (-1);
  }
  sock_sturdy(s, 100);
  sock_mode(s, TCP_MODE_ASCII);
  sock_mode(s, TCP_MODE_NONAGLE);

  printf("waiting...\r");
  sock_wait_established(s, sock_delay, NULL, &status);
  connected = 1;
  outstanding = 1;
  return (0);

sock_errE:
  printf("connection could not be established\n");

sock_err:
  printf("connection attempt timed out\n");
  sock_close(s);
  return (-1);
}

int
ftp(longword host, char *hoststring)
{
  tcp_Socket *s;
  int status, temp;
  char tempbuf[20];
  char *p, *q;
  int hashon = 4096, exitnow = 0, datamode = DATA_ASCII;
  int interactive = 1;

  s = &ftpctl;

  if (host) {
    ftpconnect(s, host, ftpctl_port);
  }

  do {
  process:
    if (outstanding)
      switch (getresponse(s, puts)) {
        case 220:
          /* opening message */
          *buffer = 0;
          if (*proxy && strchr(server, '.') &&
              ((p = strchr(proxy, '@')) != NULL)) {
            /* Proxy requires authentication. */
            strcpy(buffer, proxy);
            /* Copy the proxy username into buffer. */
            buffer[(int)(p-proxy)] = 0;
          }
          else if (*username) {
            /* Username specified on command line. */
            strcpy(buffer, username);
            *username = 0;
          }
        user:
          if (*buffer == 0)
            linegets("Userid : ", buffer, 1);
          if (q = strrchr(buffer, ' '))
            q++;
          else
            q = buffer;
          if (*proxy && strchr(server, '.')) {
            if ((p = strchr(proxy, '@')) != NULL) {
              /* Proxy authentication is only done once. */
              strcpy(proxy, ++p);
            }
            else {
              /* Pass destination hostname to proxy. */
              strcat(q, "@");
              strcat(q, server);
            }
          }
          sendcommand(s, "USER", q);
          goto process;
        case 221:
          /* closing message */
          sock_close(s);
          connected = 0;
          sock_wait_closed(s, sock_delay, NULL, &status);
          break;
        case 230:
          /* successfully logged in */
          if (*username) {
            /* Username specified on command line. */
            strcpy(buffer, username);
            *username = 0;
            goto user;
          }
          break;
        case 226:
          /* file successfully transferred */
          break;
        case 331:
          /* needs a password */
        pass:
          *buffer = 0;
          linegets("Password : ", buffer, 0);
          if (q = strchr(buffer, ' '))
            q++;
          else
            q = buffer;
          sendcommand(s, "PASS", q);
          memset(buffer, 0, sizeof(buffer));
          goto process;
        default:
          break;
      }
    if (exitnow)
      break;

    /* turn on intelligent cbreak handling */
    wathndlcbrk = 1;
    /* but clear last ^C */
    watcbroke = 0;
    *buffer = 0;
    linegets("FTP> ", buffer, 1);
    switch (text2cmd(buffer, linebuffer, &p, &q)) {
      case STARTLIST:
        break;
      case ENDLIST:
        printf("    COMMAND NOT UNDERSTOOD\n");
        break;
      case STATUS:
        if (connected) {
          printf("Connected\n");
          printf(" mode            : %s\n",
                 (datamode) ? "ASCII" : "BINARY");
          printf(" local directory : %s\n",
                 getcwd(buffer, sizeof(buffer)));
          printf(" connected to    : %s\n", server);
          printf("\nRemote Status:\n");
          sendcommand(s, "STAT", NULL);
        }
        else
          printf("Not connected\n");
        break;
      case OPEN:
        if (connected)
          printf("still connected, must CLOSE current "
                 "session first\n");
        else {
          strncpy(server, p, MAXHOST);
          if ((p = strchr(server, '@')) != NULL) {
            *p = 0;
            strncpy(username, server, MAXHOST);
            p++;
            strncpy(server, p, MAXHOST);
          }
          server[MAXHOST-1] = 0;
          if (*proxy && strchr(server, '.')) {
            if ((p = strchr(proxy, '@')) != NULL) {
              hoststring = ++p;
            }
            else {
              hoststring = proxy;
            }
          }
          else {
            hoststring = server;
          }
          printf("resolving '%s'\n", hoststring);
          if ((host = resolve(hoststring)) == NULL)
            printf("unable to resolve '%s'\n", hoststring);
          else {
            printf("trying '%s' [%s]\n",
                   hoststring, inet_ntoa(tempbuf, host));
            temp = ftpctl_port;
            if (q && (temp = atoi(q)))
              printf("using port %u for this connection\n",
                     temp);
            ftpconnect(s, host, temp);
          }
          datamode = DATA_ASCII;
        }
        break;

      case USER:
        *buffer = 0;
        if (p)
          movmem(p, buffer, strlen(p) + 1);
        goto user;
      case PASS:
        goto pass;
      case QUOTE:
        if (p) {
          if (q) {
            /* remove the \0 separator */
            q = strchr(p, 0);
            *q = ' ';
          }
          sendcommand(s, p, NULL);
        }
        else
          printf("ERROR: nothing to quote\n");
        break;
      case CLOSE:
        sendcommand(s, "QUIT", NULL);
        break;
      case DIR:
        if (datamode != DATA_ASCII) {
          sendcommand(s, "TYPE", "A");
          if (getresponse(s, puts) != 200)
            break;
        }
        remoteop(s, host, "LIST", p,
                 NULL, 1, 1, 1, NULL, 0);

        if (datamode != DATA_ASCII) {
          sendcommand(s, "TYPE", "I");
          if (getresponse(s, puts) != 200) {
            datamode = DATA_ASCII;
            printf("WARNING: Host left connection "
                   "in ASCII mode");
          }
        }
        break;

      case LS:
        if (datamode != DATA_ASCII) {
          sendcommand(s, "TYPE", "A");
          if (getresponse(s, puts) != 200)
            break;
        }
        remoteop(s, host, "NLST", p,
                 NULL, 1, 1, 1, NULL, 0);

        if (datamode != DATA_ASCII) {
          sendcommand(s, "TYPE", "I");
          if (getresponse(s, puts) != 200) {
            datamode = DATA_ASCII;
            printf("WARNING: Host left connection "
                   "in ASCII mode");
          }
        }
        break;

      case TYPE:
        if (datamode != DATA_ASCII) {
          sendcommand(s, "TYPE", "A");
          if (getresponse(s, puts) != 200)
            break;
        }

        remoteop(s, host, "RETR", p,
                 p, 1, 1, 1, NULL, 0);

        if (datamode != DATA_ASCII) {
          sendcommand(s, "TYPE", "I");
          if (getresponse(s, puts) != 200) {
            datamode = DATA_ASCII;
            printf("WARNING: Host left connection "
                   "in ASCII mode");
          }
        }
        break;
      case GET:
        remoteop(s, host, "RETR", p,
                 (q) ? q : todos(p), 1, 0, 0, NULL, hashon);
        break;

      case PUT:
        remoteop(s, host, "STOR", (q) ? q : tounix(p),
                 p, 0, 0, 0, NULL, hashon);
        break;

      case MGET:
        Fmget(s, host, p, datamode, interactive, hashon);
        break;

      case MPUT:
        Fmput(s, host, p, datamode, interactive, hashon);
        break;

      case RGET:
        Frget(s, host, p, datamode, hashon);
        break;

      case RPUT:
        Frput(s, host, p, datamode, hashon);
        break;

      case DEL:
        sendcommand(s, "DELE", p);
        break;
      case CHDIR:
        sendcommand(s, "CWD", p);
        break;
      case RMDIR:
        sendcommand(s, "RMD", p);
        if (getresponse(s, puts) != 250) {
          printf("Trying \"XRMD\" instead of \"RMD\"\n");
          sendcommand(s, "XRMD", p);
        }
        break;
      case MKDIR:
        sendcommand(s, "MKD", p);
        if (getresponse(s, puts) != 257) {
          printf("Trying \"XMKD\" instead of \"MKD\"\n");
          sendcommand(s, "XMKD", p);
        }
        break;

      case LDIR:
        if (!p)
          p = "*.*";
        ldir(p);
        break;
      case LDEL:
        dosmessage(unlink(p), "delete file");
        break;
      case PWD:
        sendcommand(s, "PWD", p);
        if (getresponse(s, puts) != 257) {
          printf("Trying \"XPWD\" instead of \"PWD\"\n");
          sendcommand(s, "XPWD", p);
        }
        break;

      case LCHDIR:
        if (p[1] == ':') {
          dosmessage(!setdisk(toupper(*p) - 'A'),
                     "change disk");
          if (p[2] == 0)
            break;
          p += 2;
        }
        dosmessage(chdir(p), "change directory");
        break;
      case LRMDIR:
        dosmessage(rmdir(p), "remove directory");
        break;
      case LMKDIR:
        dosmessage(mkdir(p), "make directory");
        break;
      case LPWD:
        p = getcwd((char *) NULL, 128);
        if (p) {
          printf("%s\n", p);
          free(p);
        }
        break;
      case QUIT:
        if (!connected)
          return (0);
        sendcommand(s, "QUIT", NULL);
        exitnow = 1;
        break;
      case HELP:
        dohelp(p);
        break;

      case RHELP:
        if (datamode != DATA_ASCII) {
          sendcommand(s, "TYPE", "A");
          if (getresponse(s, puts) != 200)
            break;
        }

        remoteop(s, host, "HELP", p,
                 p, 1, 1, 1, NULL, 0);

        if (datamode != DATA_ASCII) {
          sendcommand(s, "TYPE", "I");
          if (getresponse(s, puts) != 200) {
            datamode = DATA_ASCII;
            printf("WARNING: Host left connection "
                   " ASCII mode");
          }
        }
        break;

      case ASCII:
        sendcommand(s, "TYPE", "A");
        if (getresponse(s, puts) == 200)
          datamode = DATA_ASCII;
        break;
      case BIN:
        sendcommand(s, "TYPE", "I");
        if (getresponse(s, puts) == 200)
          datamode = DATA_BINARY;
        break;
      case INT:
        if (!p)
          interactive = !interactive;
        else {
          if (!stricmp(p, "OFF"))
            interactive = 0;
          else
            interactive = 1;
        }
        printf("Interactive mode %s\n",
               (interactive) ? "enabled" : "disabled");
        break;
      case HASH:
        if (p) {
          if (!stricmp(p, "on"))
            hashon = 1024;
          else
            hashon = atoi(p);
        }
        else {
          if (hashon)
            hashon = 0;
          else
            hashon = 1024;
        }
        if (hashon)
          printf("Hash mark for every %u bytes\n", hashon);
        else
          printf("Hash marks disabled\n");
        break;
      case WAIT:
        wait(p);
        break;
      case SHELL:
        doshell();
        break;
      case RUN:
        dorun(p);
        break;
      case ECHO:
        printf(p);
        break;
      case QUIET:
        if (!stricmp(p, "ON"))
          quiet = 1;
        else if (!stricmp(p, "OFF"))
          quiet = 0;
        else
          printf("QUIET requires either ON or OFF parameter");
        break;
      case DEBUG:
        printf("Debug mode %s\n",
               (debug_on ^= 2) ? "enabled" : "disabled");
        break;
      default:
        printf(" ** NOT IMPLEMENTED **\n");
        break;
    }
  } while (1);
  sock_close(s);
  sock_wait_closed(s, sock_delay, NULL, &status);

sock_err:
  switch (status) {
    case 1:                    /* foreign host closed */
      break;
    case -1:                   /* timeout */
      printf("ERROR: %s\n", sockerr(s));
      break;
  }
  connected = 0;
  sock_close(s);
  printf("\n");
  if (!exitnow)
    goto process;
  return (0);
}

void (*old_init)(char *directive, char *value);
void new_init   (char *directive, char *value)
{
  if (!strnicmp(directive, "lxftp.", 6)) {
    strcpy(directive, directive+6);
  }
  /*fprintf(stderr, "Setting %s to %s.\n", directive, value);*/
  if (!stricmp(directive, "debug")) {
    if (atoi(value)) {
      dbug_init();
    }
    tcp_set_debug_state(atoi(value));
  }
  else if (!stricmp(directive, "ftp.proxy")) {
    if (!strnicmp(value, "ftp://", 6)) {
      strcpy(value, value+6);
    }
    strncpy(proxy, value, MAXHOST);
    proxy[MAXHOST-1] = 0;
  }
  else if (!stricmp(directive, "ftp.passive")) {
    passive = atoi(value);
  }
  else {
    if (old_init) (*old_init)(directive, value);
  }
}

void
main(int argc, char *argv[])
{
  char *hoststring = NULL;
  longword host = 0;
  int status, i;
  char *p;

  printf("LXFTP %d.%d", VERSION/10, VERSION%10);

  if (RELEASE > 0) {
    printf("b%d", RELEASE);
  }

  printf(" - %s <%s>\n",
         "HP200LX TCP/IP Suite",
         "http://lxtcp.hplx.net/");

  *server = *proxy = *username = 0;

  old_init = usr_init;
  usr_init = new_init;

  if (getenv("LXTCP.CFG")) {
    tcp_config_file(getenv("LXTCP.CFG"));
  }

  sock_init();

  filebuffer = malloc(FILEBUFFERLEN);
  screenbuffer = malloc(SCREENBUFFERLEN);
  if (filebuffer == NULL || screenbuffer == NULL) {
    printf("Insufficient memory to run FTP.");
    if (filebuffer != NULL)
      free(filebuffer);
    if (screenbuffer != NULL)
      free(screenbuffer);
  }

  printf("Type HELP for more information\n");

  /* don't allow control breaks */
  tcp_cbrk(1);

  for (i = 1; i < argc; ++i) {
    if (!stricmp(argv[i], "-f")) {
      if (++i == argc) {
        printf("ERROR: missing name for command file");
        exit(3);
      }
      if (infile = fopen(argv[i], "rt"))
        printf("running commands from: %s\n", argv[i]);
      else {
        printf("ERROR: unable to read '%s'\n", argv[i]);
        exit(3);
      }
      continue;
    }
    else if (strchr(argv[i], '=') != NULL) {
      char *directive, *value;
      directive = argv[i];
      if ((value = strchr(directive, '=')) != 0) {
        *value = 0; value++; strupr(directive);
        usr_init(directive, value);
      }
    }
    else if (hoststring == NULL) {
      strncpy(server, argv[i], MAXHOST);
      if ((p = strchr(server, '@')) != NULL) {
        *p = 0;
        strncpy(username, server, MAXHOST);
        p++;
        strncpy(server, p, MAXHOST);
      }
      server[MAXHOST-1] = 0;
      if (*proxy && strchr(server, '.')) {
        if ((p = strchr(proxy, '@')) != NULL) {
          hoststring = ++p;
        }
        else {
          hoststring = proxy;
        }
      }
      else {
        hoststring = server;
      }
      printf("looking up '%s'...\n", hoststring);
      if ((host = resolve(hoststring)) == 0L) {
        printf("Could not resolve host '%s'\n", hoststring);
        exit(3);
      }
    }
    else if (atoi(argv[i])) {
      ftpctl_port = atoi(argv[i]);
    }
  }

  status = ftp(host, server);

  if (filebuffer != NULL)
    free(filebuffer);
  if (screenbuffer != NULL)
    free(screenbuffer);

  exit(status);
}
